﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Nintendo.AudioTool.Detail
{
    internal class SystemAudioCaptureImpl : IDisposable
    {
        public int AdspLoad { get; private set; }

        private static readonly int ReadTimeOutMsec = 3000;

        // Task
        Task<bool> _recordingTask;

        private event EventHandler<SystemAudioCaptureEventArgs> _dataAvailableHandle;
        private event EventHandler<SystemAudioCaptureStoppedEventArgs> _captureStoppedHandle;
        private AutoResetEvent _requestExitEvent = new AutoResetEvent(false);

        private void DetachEventHandler()
        {
            _dataAvailableHandle = null;
            _captureStoppedHandle = null;
        }

        private void ThrowIfExitRequested()
        {
            if(_requestExitEvent.WaitOne(0))
            {
                throw new OperationCanceledException();
            }
        }

        private void CaptureWaveform(SystemAudioCapture.WaveFormat format, NetworkStream st)
        {
            // TODO: 録音の設定をやり取りする
            int intervalMsec = 20;
            try
            {
                using (var sw = new System.IO.StreamWriter(st, Encoding.ASCII, 1024, true))
                using (var br = new System.IO.BinaryReader(st, Encoding.ASCII, true))
                {
                    br.BaseStream.ReadTimeout = ReadTimeOutMsec;

                    const string WaveRequireString = "msg[Waveform Require]";
                    while (true)
                    {
                        // 波形要求の送信
                        sw.Write(WaveRequireString);
                        sw.Flush();

                        int bytes = (format.BitDepth / 8) * format.ChannelCount * format.SampleRate * intervalMsec / 1000;
                        var buf = br.ReadBytes(bytes);
                        _dataAvailableHandle?.Invoke(this, new SystemAudioCaptureEventArgs(buf, bytes));

                        var adspLoad = br.ReadBytes(4);
                        AdspLoad = BitConverter.ToInt32(adspLoad, 0);
                        ThrowIfExitRequested();
                    }
                }
            }
            catch (OperationCanceledException)
            {
                _captureStoppedHandle?.Invoke(this, new SystemAudioCaptureStoppedEventArgs());
            }
            catch (Exception e)
            {
                if (_requestExitEvent.WaitOne(0))
                {
                    _captureStoppedHandle?.Invoke(this, new SystemAudioCaptureStoppedEventArgs());
                }
                else
                {
                    _captureStoppedHandle?.Invoke(this, new SystemAudioCaptureStoppedEventArgs(e));
                }
            }
        }

        public void Start(SystemAudioCapture.WaveFormat format, NetworkStream st, EventHandler<SystemAudioCaptureEventArgs> handler, EventHandler<SystemAudioCaptureStoppedEventArgs> stopHandler)
        {
            if (_recordingTask?.IsCompleted == false)
            {
                return;
            }

            _dataAvailableHandle = handler;
            _captureStoppedHandle = stopHandler;
            _requestExitEvent.Reset();
            _recordingTask = Task.Run(() => { return StartImpl(format, st); });
        }

        private bool StartImpl(SystemAudioCapture.WaveFormat format, NetworkStream st)
        {
            // 波形の録音
            CaptureWaveform(format, st);
            return true;
        }

        public void Stop()
        {
            if (_recordingTask?.IsCompleted == false)
            {
                _requestExitEvent.Set();
            }

            _recordingTask = null;
        }

        public void Dispose()
        {
            Stop();
        }
    }
}
