﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Nintendo.Audio.Windows.CoreAudio.Interface;
using System;
using System.Runtime.InteropServices;
using System.Threading;

namespace Nintendo.Audio.Windows.CoreAudio
{
    public class SpatialAudioClient : IDisposable
    {
        private ISpatialAudioClient _target;

        public uint MaxDynamicObjectCount
        {
            get
            {
                Marshal.ThrowExceptionForHR(_target.GetMaxDynamicObjectCount(out var count));
                return count;
            }
        }

        public SpatialAudioClient(object target)
        {
            _target = target as ISpatialAudioClient;
        }

        public SpatialAudioClient(ISpatialAudioClient target)
        {
            _target = target;
        }

        public IAudioFormatEnumerator GetSupportedAudioObjectFormatEnumerator()
        {
            if (_target == null)
            {
                return null;
            }
            Marshal.ThrowExceptionForHR(_target.GetSupportedAudioObjectFormatEnumerator(out var enumerator));
            return enumerator;
        }

        public object ActivateSpatialAudioStream(SpatialAudioObjectRenderStreamActivationParams param, Guid interfaceId)
        {
            var activationParamsPtr = Marshal.AllocHGlobal(Marshal.SizeOf(typeof(SpatialAudioObjectRenderStreamActivationParams)));
            Marshal.StructureToPtr(param, activationParamsPtr, false);

            var pv = new PropVariant();
            pv.vt = (ushort)VarEnum.VT_BLOB;
            pv.blob.cbSize = (uint)Marshal.SizeOf(typeof(SpatialAudioObjectRenderStreamActivationParams));
            pv.blob.pBlobData = activationParamsPtr;
            var pvPtr = Marshal.AllocHGlobal(Marshal.SizeOf(typeof(PropVariant)));
            Marshal.StructureToPtr(pv, pvPtr, false);

            Marshal.ThrowExceptionForHR(_target.ActivateSpatialAudioStream(pvPtr, interfaceId, out var stream));

            Marshal.FreeHGlobal(activationParamsPtr);
            Marshal.FreeHGlobal(pvPtr);

            return stream;
        }

        public object ActivateSpatialAudioStream(AudioObjectType type, AudioStreamCategory category, Guid interfaceId, out EventWaitHandle eventWaitHandle)
        {
            eventWaitHandle = new EventWaitHandle(false, EventResetMode.AutoReset);

            var waveFormatPtr = new AudioFormatEnumerator(GetSupportedAudioObjectFormatEnumerator()).GetFormat(0);
            var activationParams = new SpatialAudioObjectRenderStreamActivationParams()
            {
                objectFormat = waveFormatPtr,
                staticObjectType = type,
                minDynamicObjectCount = MaxDynamicObjectCount,
                maxDynamicObjectCount = MaxDynamicObjectCount,
                category = category,
                eventHandle = eventWaitHandle.SafeWaitHandle.DangerousGetHandle(),
                notifyObject = IntPtr.Zero,
            };

            return ActivateSpatialAudioStream(activationParams, interfaceId);
        }

        public void Dispose()
        {
            if (_target != null)
            {
                Marshal.ReleaseComObject(_target);
                _target = null;
            }
        }
    }
}
