﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "coredump_FormatTypes.h"

//==============================================================================
namespace coredump {
//==============================================================================

struct coredump_compiled_data_module
{
    explicit coredump_compiled_data_module( )
    {
        memset( this, 0, sizeof(coredump_compiled_data_module) );
    }

    coredump_compiled_data_module( char* pName, char* pId, u64 Address, u64 Size )
    {
        memcpy( m_Name, pName, sizeof( m_Name ) );
        memcpy( m_Id, pId, sizeof( m_Id ) );
        m_Address = Address;
        m_Size = Size;
    }
    char    m_Name[1024];
    char    m_Id[32];
    u64     m_Address;
    u64     m_Size;
};

//==============================================================================

struct coredump_compiled_data_thread
{
    explicit coredump_compiled_data_thread( )
    {
        memset( this, 0, sizeof(coredump_compiled_data_thread) );
    }

    coredump_compiled_data_thread( u64 Id, char* pName, u64 StackSize, u64 StackAddress )
    {
        m_Id = Id;
        memcpy( m_Name, pName, sizeof( m_Name ) );
        m_StackSize = StackSize;
        m_StackAddress = StackAddress;
    }
    u64     m_Id;
    char    m_Name[32];
    u64     m_StackSize;
    u64     m_StackAddress;
};

//==============================================================================

struct coredump_compiled_data
{
    u64                             m_ProcessId;
    u32                             m_ExceptionId;
    u64                             m_ExceptionAddr;
    u64                             m_CurrentThreadId;
    u64                             m_ProgramId;
    bool                            m_Is64Bit;
    bool                            m_Is64BitAddressSpace;
    bool                            m_QuickDump;
    char                            m_ProcessName[1024];
    char                            m_Args[1024];
    char*                           m_pTTY;
    u32                             m_NumberOfThreads;
    u32                             m_NumberOfModules;
    coredump_compiled_data_thread*  m_pThreads;
    coredump_compiled_data_module*  m_pModules;

    explicit    coredump_compiled_data()
                {
                    memset( this, 0, sizeof( coredump_compiled_data ) );
                }

                ~coredump_compiled_data()
                {
                    if( m_pThreads != NULL )
                    {
                        delete [] m_pThreads;
                    }
                    if( m_pModules != NULL )
                    {
                        delete [] m_pModules;
                    }
                    if( m_pTTY != NULL )
                    {
                        delete [] m_pTTY;
                    }
                };

    //==============================================================================

    void Init( u64 ProcessId, u64 CurrentThreadId, bool Is64Bit, bool Is64BitAddressSpace, u64 ProgramId, bool QuickDump = false )
    {
        m_ProcessId = ProcessId;
        m_Is64Bit = Is64Bit;
        m_Is64BitAddressSpace = Is64BitAddressSpace;
        m_QuickDump = QuickDump;
        m_CurrentThreadId = CurrentThreadId;
        m_ProgramId = ProgramId;
    }

    //==============================================================================

    void SetName( char* pName )
    {
        strcpy( m_ProcessName, pName);
    }

    //==============================================================================

    void SetArgs( char* pArgs, coredump_compiled_data* pData )
    {
        strcpy( m_Args, pArgs );
    }

    //==============================================================================

    void SetException( u32 ExceptionId, u64 ExceptionAddress )
    {
        m_ExceptionId = ExceptionId;
        m_ExceptionAddr = ExceptionAddress;
    }

    //==============================================================================

    void SetTTY( char* pTTY )
    {
        if( pTTY != NULL )
        {
            u32 Length = strlen( pTTY );
            if( Length > 0 )
            {
                m_pTTY = new char [Length + 1];
                strcpy( m_pTTY, pTTY );
            }
        }
    }

    //==============================================================================

    void AddThread( u64 ThreadId, char* pThreadName, u64 ThreadSize, u64 pThreadStack )
    {
        coredump_compiled_data_thread* pOrigList = m_pThreads;
        m_pThreads = new coredump_compiled_data_thread[m_NumberOfThreads + 1];
        if( pOrigList != NULL )
        {
            memcpy( m_pThreads, pOrigList, sizeof(coredump_compiled_data_thread) * m_NumberOfThreads );
            delete [] pOrigList;
        }
        m_pThreads[m_NumberOfThreads] = coredump_compiled_data_thread( ThreadId, pThreadName, ThreadSize, pThreadStack );
        m_NumberOfThreads += 1;
    }

    //==============================================================================

    void AddModule( char* pModuleName, char* pModuleId, u64 LoadAddress, u64 Size )
    {
        coredump_compiled_data_module* pOrigList = m_pModules;
        m_pModules = new coredump_compiled_data_module[m_NumberOfModules + 1];
        if( pOrigList )
        {
            memcpy( m_pModules, pOrigList,  sizeof(coredump_compiled_data_module) * m_NumberOfModules );
            delete [] pOrigList;
        }

        m_pModules[m_NumberOfModules] = coredump_compiled_data_module( pModuleName, pModuleId, LoadAddress, Size );
        m_NumberOfModules += 1;
    }

    //==============================================================================

};

//==============================================================================

//==============================================================================
} // namespace coredump
//==============================================================================

