﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <memory>
#include <sstream>
#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/account/account_Types.h>
#include <nn/util/util_Uuid.h>

#include "SettingsManager_ErrorCode.h"

//!< コマンドを実行します。
#define COMMAND_DO(command)\
    do { if (!(command)) { return false; } } while (NN_STATIC_CONDITION(0))

//!< 動的配列が指定された値を含むか否かを表す値を返します。
template<typename T>
bool Contains(const ::std::vector<T> values, const T& value) NN_NOEXCEPT
{
    return ::std::find(values.begin(), values.end(), value) != values.end();
}

//!< バッファをアロケートします。
template<typename T>
bool AllocateBuffer(
    ::std::unique_ptr<::std::vector<T>>* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto ptr = new ::std::vector<T>(size, T());

    if (ptr == nullptr)
    {
        PrintErrorCode(ErrorCode::MemoryAllocationFailure, size);

        return false;
    }

    pOutValue->reset(ptr);

    return true;
}

//!< 要素を指定された文字列で連結します。
template<typename Iterator>
::std::string Join(
    const ::std::string& separator, Iterator head, Iterator tail) NN_NOEXCEPT
{
    Iterator iter = head;

    ::std::ostringstream stream;

    while (iter != tail)
    {
        if (iter != head)
        {
            stream << separator;
        }

        stream << *iter;

        ++iter;
    }

    return stream.str();
}

//!< 文字列を指定された文字列で分割します。
::std::vector<::std::string> Split(
    const ::std::string& value, const ::std::string& separator) NN_NOEXCEPT;

//!< 紛失したキーを取得します。
bool GetMissingKeys(
    ::std::vector<::std::string>* pOutValue,
    const ::std::vector<::std::string>& haystack,
    const ::std::vector<::std::string>& needles) NN_NOEXCEPT;

//!< 整数値をエンコードします。
bool EncodeInteger(
    ::std::string* pOutValue, int64_t value, int base = 0) NN_NOEXCEPT;

//!< 整数値をデコードします。
bool DecodeInteger(
    int64_t* pOutValue, const ::std::string& string, int base = 0) NN_NOEXCEPT;

//!< UUID をエンコードします。
bool EncodeUuid(
    ::std::string* pOutValue, const ::nn::util::Uuid& value) NN_NOEXCEPT;

//!< UUID をデコードします。
bool DecodeUuid(
    ::nn::util::Uuid* pOutValue, const ::std::string& string) NN_NOEXCEPT;

//!< ユーザの識別子を UUID に変換します。
::nn::util::Uuid ConvertToUuid(const ::nn::account::Uid& uid) NN_NOEXCEPT;

//!< UUID をユーザの識別子に変換します。
::nn::account::Uid ConvertToUid(const ::nn::util::Uuid& uuid) NN_NOEXCEPT;
