﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_TimeZone.h>
#include <nn/time/time_LocationName.h>
#include <nn/util/util_StringUtil.h>

#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_TimeZone.h"
#include "SettingsManager_Utility.h"

namespace {

//!< タイムゾーン設定のキー
const char* const SettingNameTimeZoneSettings = "time_zone_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< デバイスの地域名のキー
const char* const KeyDeviceLocationName = "device_location_name";

//!< デバイスの地域名をエクスポートします。
bool ExportDeviceLocationNameFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    ::nn::time::LocationName name = {};

    ::nn::settings::system::GetDeviceTimeZoneLocationName(&name);

    const int count =
        ::nn::util::Strnlen(name._value, ::nn::time::LocationName::Size);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyDeviceLocationName);

        auto currentValueNode = Node::CreateStringNode();

        const ::std::string value(name._value, count);

        COMMAND_DO(currentValueNode.SetValue(value));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue("UTC"));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyDeviceLocationName, ::std::move(node)));

    return true;
}

//!< デバイスの地域名をインポートします。
bool ImportDeviceLocationName(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyDeviceLocationName));

    NameScope nameScope(KeyDeviceLocationName);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string value;
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    ::nn::time::LocationName name = {};

    ::nn::util::Strlcpy(
        name._value, value.c_str(), ::nn::time::LocationName::Size);

    ::nn::settings::system::SetDeviceTimeZoneLocationName(name);

    return true;
}

} // namespace

bool IsSettingNameTimeZoneSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameTimeZoneSettings);
}

bool ExportTimeZoneSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameTimeZoneSettings);

        COMMAND_DO(ExportDeviceLocationNameFlag(&node));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameTimeZoneSettings, ::std::move(node)));

    return true;
}

bool ImportTimeZoneSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameTimeZoneSettings);

    COMMAND_DO(ImportDeviceLocationName(node));

    return true;
}
