﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_NetworkService.h>

#include "SettingsManager_NameScope.h"
#include "SettingsManager_NetworkService.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< ネットワークサービス設定のキー
const char* const SettingNameNetworkServiceSettings =
    "network_service_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 本体情報アップロードが有効か否かを表す値のキー
const char* const KeyConsoleInformationUploadFlag =
    "console_information_upload_flag";

//!< 本体情報アップロードが有効か否かを表す値をエクスポートします。
bool ExportConsoleInformationUploadFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyConsoleInformationUploadFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::GetConsoleInformationUploadFlag()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyConsoleInformationUploadFlag, ::std::move(node)));

    return true;
}

//!< 本体情報アップロードが有効か否かを表す値をインポートします。
bool ImportConsoleInformationUploadFlag(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyConsoleInformationUploadFlag));

    NameScope nameScope(KeyConsoleInformationUploadFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto flag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&flag));

    ::nn::settings::system::SetConsoleInformationUploadFlag(flag);

    return true;
}

} // namespace

bool IsSettingNameNetworkServiceSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameNetworkServiceSettings);
}

bool ExportNetworkServiceSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameNetworkServiceSettings);

        COMMAND_DO(ExportConsoleInformationUploadFlag(&node));
    }

    COMMAND_DO(
        pNode->AppendMember(
            SettingNameNetworkServiceSettings, ::std::move(node)));

    return true;
}

bool ImportNetworkServiceSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameNetworkServiceSettings);

    COMMAND_DO(ImportConsoleInformationUploadFlag(node));

    return true;
}
