﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/os/os_Argument.h>
#include <nn/settings/system/settings_Language.h>

#include "SettingsManager_Command.h"
#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_FileSystem.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< フォーマットバージョン
const char* const FormatVersion = "1.0";

const char* const KeyHead = "head";         //!< 設定情報ヘッダのキー
const char* const KeyBody = "body";         //!< 設定情報ボディのキー
const char* const KeyVersion = "version";   //!< バージョンのキー

//!< コマンド定義を表す構造体です。
struct Command final
{
    //!< コマンドの名前
    const char* name;

    //!< コマンドの実体
    bool (*execute)(const ::std::vector<::std::string>&) NN_NOEXCEPT;
};

//!< プログラム引数を返します。
::std::vector<::std::string> GetArgs() NN_NOEXCEPT;

//!< コマンド定義を返します。
const Command* GetCommand(const ::std::string& name) NN_NOEXCEPT;

//!< 設定のエクスポートをディスパッチします。
bool DispatchExportSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT;

//!< 設定のインポートをディスパッチします。
bool DispatchImportSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT;

//!< 設定をリセットします。
bool DispatchResetSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT;

//!< 設定をダンプします。
bool DispatchDumpSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT;

//!< 設定をパースします。
bool DispatchParseSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT;

} // namespace

extern "C" void nnMain()
{
    // プログラム引数を取得
    ::std::vector<::std::string> args = GetArgs();

    if (args.size() < 2)
    {
        // コマンドが指定されなかった場合は失敗
        PrintErrorCode(ErrorCode::CommandSpecificationMissing);
    }
    else
    {
        // コマンド名を退避
        ::std::string name = args[1];

        // コマンド引数以外を除去
        args.erase(args.begin(), args.begin() + 2);

        // fs サービスを複数セッションで初期化
        ::nn::fs::InitializeWithMultiSessionForTargetTool();

        // ファイルシステム用のアロケータを設定
        ::nn::fs::SetAllocator(AllocateForFileSystem, DeallocateForFileSystem);

        if (name.find("@") == 0)
        {
            name = name.substr(1);
        }

        const Command* const pCommand = GetCommand(name);

        if (pCommand == nullptr)
        {
            // コマンドが見つからなかった場合は失敗
            PrintErrorCode(ErrorCode::CommandNotFound, name);
        }
        else
        {
            // 検出されたコマンドを実行
            pCommand->execute(args);
        }
    }
}

namespace {

::std::vector<::std::string> GetArgs() NN_NOEXCEPT
{
    const int argc = ::nn::os::GetHostArgc();

    char** const argv = ::nn::os::GetHostArgv();

    ::std::vector<::std::string> args;

    for (int i = 0; i < argc; ++i)
    {
        args.push_back(argv[i]);
    }

    return args;
}

const Command* GetCommand(const ::std::string& name) NN_NOEXCEPT
{
    static const Command Commands[] =
    {
        { "Export", DispatchExportSettings },
        { "Import", DispatchImportSettings },
        { "Reset",  DispatchResetSettings  },
        { "Dump",   DispatchDumpSettings   },
        { "Parse",  DispatchParseSettings  },
    };

    for (const Command& command : Commands)
    {
        if (name == command.name)
        {
            return &command;
        }
    }

    return nullptr;
}

//!< 設定情報ヘッダノードをエクスポートします。
bool ExportHeadNode(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    auto versionNode = Node::CreateStringNode();

    COMMAND_DO(versionNode.SetValue(FormatVersion));

    COMMAND_DO(node.AppendMember(KeyVersion, ::std::move(versionNode)));

    COMMAND_DO(pNode->AppendMember(KeyHead, ::std::move(node)));

    return true;
}

bool DispatchExportSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    if (args.size() < 1)
    {
        // ファイルパスが指定されなかった場合は失敗
        PrintErrorCode(ErrorCode::FileSpecificationMissing);

        return false;
    }
    else
    {
        Document document;

        Node rootNode = document.GetRootNode();

        COMMAND_DO(ExportHeadNode(&rootNode));

        auto bodyNode = Node::CreateObjectNode();

        COMMAND_DO(ExportSettings(&bodyNode));

        COMMAND_DO(rootNode.AppendMember(KeyBody, ::std::move(bodyNode)));

        ::std::string text;
        COMMAND_DO(document.Save(&text));

        HostFsFile hostFsFile;
        COMMAND_DO(OpenFile(&hostFsFile, args[0], ::nn::fs::OpenMode_Write));
        COMMAND_DO(SetFileSize(hostFsFile, text.size()));
        COMMAND_DO(WriteFile(hostFsFile, 0, text.data(), text.size()));
        COMMAND_DO(FlushFile(hostFsFile));

        return true;
    }
}

bool DispatchImportSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    if (args.size() < 1)
    {
        // ファイルパスが指定されなかった場合は失敗
        PrintErrorCode(ErrorCode::FileSpecificationMissing);

        return false;
    }
    else
    {
        HostFsFile hostFsFile;
        COMMAND_DO(OpenFile(&hostFsFile, args[0], ::nn::fs::OpenMode_Read));

        ::std::string text;

        auto fileSize = int64_t();
        COMMAND_DO(GetFileSize(&fileSize, hostFsFile));

        if (0 < fileSize)
        {
            ::std::unique_ptr<::std::vector<char>> ptr;
            COMMAND_DO(AllocateBuffer(&ptr, static_cast<size_t>(fileSize)));

            auto readSize = size_t();
            COMMAND_DO(
                ReadFile(&readSize, ptr->data(), ptr->size(), hostFsFile, 0));

            text = ::std::string(ptr->data(), readSize);
        }

        Document document;
        COMMAND_DO(document.Load(::std::move(text)));

        Node rootNode = document.GetRootNode();

        ::std::unique_ptr<const Node> pBodyNode;
        COMMAND_DO(rootNode.GetMember(&pBodyNode, KeyBody));

        COMMAND_DO(ImportSettings(*pBodyNode));

        // 適当な設定値を読み出してキャッシュがフラッシュされていることを保証
        ::nn::settings::LanguageCode languageCode = {};
        ::nn::settings::GetLanguageCode(&languageCode);

        return true;
    }
}

bool DispatchResetSettings(const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    if (args.size() < 1)
    {
        COMMAND_DO(ResetSettings("default"));

        return true;
    }
    else
    {
        COMMAND_DO(ResetSettings(args[0]));

        return true;
    }
}

bool DispatchDumpSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    if (args.size() < 1)
    {
        // ファイルパスが指定されなかった場合は失敗
        PrintErrorCode(ErrorCode::FileSpecificationMissing);

        return false;
    }
    else
    {
        ::std::unique_ptr<::std::vector<int64_t>> ptr;
        COMMAND_DO(DumpSettings(&ptr));

        auto buffer = reinterpret_cast<char*>(ptr->data());
        auto size = ptr->size() * sizeof(int64_t);

        HostFsFile hostFsFile;
        COMMAND_DO(OpenFile(&hostFsFile, args[0], ::nn::fs::OpenMode_Write));
        COMMAND_DO(SetFileSize(hostFsFile, size));
        COMMAND_DO(WriteFile(hostFsFile, 0, buffer, size));
        COMMAND_DO(FlushFile(hostFsFile));

        return true;
    }
}

bool DispatchParseSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    if (args.size() < 2)
    {
        // ファイルパスが指定されなかった場合は失敗
        PrintErrorCode(ErrorCode::FileSpecificationMissing);

        return false;
    }
    else
    {
        ::std::unique_ptr<::std::vector<int64_t>> ptr;

        {
            HostFsFile hostFsFile;
            COMMAND_DO(OpenFile(&hostFsFile, args[0], ::nn::fs::OpenMode_Read));

            auto fileSize = int64_t();
            COMMAND_DO(GetFileSize(&fileSize, hostFsFile));

            const auto count = static_cast<size_t>(
                (fileSize / sizeof(int64_t)) +
                (fileSize % sizeof(int64_t) == 0 ? 0 : 1));

            COMMAND_DO(AllocateBuffer(&ptr, count));

            auto readSize = size_t();
            COMMAND_DO(
                ReadFile(
                    &readSize, reinterpret_cast<char*>(ptr->data()),
                    static_cast<size_t>(fileSize), hostFsFile, 0));
        }

        COMMAND_DO(LoadSettings(ptr.get()));

        ::std::vector<::std::string> rest(args.begin() + 1, args.end());

        COMMAND_DO(DispatchExportSettings(rest));

        return true;
    }
}

} // namespace
