﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Ldn.h>

#include "SettingsManager_Ldn.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< ローカル通信設定のキー
const char* const SettingNameLdnSettings = "ldn_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< ローカル通信のチャンネルのキー
const char* const KeyLdnChannel = "ldn_channel";

//!< ローカル通信のチャンネルをエクスポートします。
bool ExportLdnChannel(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyLdnChannel);

        auto currentValueNode = Node::CreateIntegerNode();

        COMMAND_DO(
            currentValueNode.SetValue(::nn::settings::system::GetLdnChannel()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateIntegerNode();

        COMMAND_DO(defaultValueNode.SetValue(-1));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyLdnChannel, ::std::move(node)));

    return true;
}

//!< ローカル通信のチャンネルをインポートします。
bool ImportLdnChannel(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyLdnChannel));

    NameScope nameScope(KeyLdnChannel);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto ldnChannel = int32_t();
    COMMAND_DO(pCurrentValueNode->GetValue(&ldnChannel));

    ::nn::settings::system::SetLdnChannel(ldnChannel);

    return true;
}

} // namespace

bool IsSettingNameLdnSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameLdnSettings);
}

bool ExportLdnSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameLdnSettings);

        COMMAND_DO(ExportLdnChannel(&node));
    }

    COMMAND_DO(pNode->AppendMember(SettingNameLdnSettings, ::std::move(node)));

    return true;
}

bool ImportLdnSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameLdnSettings);

    COMMAND_DO(ImportLdnChannel(node));

    return true;
}
