﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <memory>
#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/fwdbg/settings_SettingsManagement.h>

#include "SettingsManager_FirmwareDebugSettings.h"
#include "SettingsManager_Utility.h"

namespace {

//!< バイト配列の読み取りを扱うクラスです。
class ByteReader final
{
    NN_DISALLOW_COPY(ByteReader);
    NN_DISALLOW_MOVE(ByteReader);

    //!< バイト配列
    const ::nn::Bit8* m_Buffer;

    //!< バイト配列の長さ
    size_t m_Count;

    //!< オフセット
    size_t m_Offset;

public:
    ByteReader(const ::nn::Bit8 buffer[], size_t count) NN_NOEXCEPT;

    //!< バイト配列の長さを返します。
    size_t GetCount() const NN_NOEXCEPT;

    //!< 現在の読み取り開始位置を返します。
    size_t GetOffset() const NN_NOEXCEPT;

    //!< 読み取り開始位置を進めます。
    bool Proceed(size_t count) NN_NOEXCEPT;

    //!< バイト配列の内容を取得します。
    template<typename T>
    bool Read(T* pOutValue) NN_NOEXCEPT
    {
        return this->Read(reinterpret_cast<char*>(pOutValue), sizeof(T));
    }

    //!< バイト配列の内容を取得します。
    bool Read(char outBuffer[], size_t count) NN_NOEXCEPT;
};

} // namespace

bool DumpFirmwareDebugSettings(
    ::nn::Bit8 outBuffer[], size_t count,
    ::nn::settings::fwdbg::SettingsTarget target,
    const ::std::vector<::std::string>& names) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    ::nn::settings::fwdbg::ReadSettings(outBuffer, count, target);

    ByteReader reader(outBuffer, count);

    auto offset = static_cast<uint32_t>(sizeof(uint32_t));

    auto totalSize = uint32_t();
    COMMAND_DO(reader.Read(&totalSize));

    while (reader.GetOffset() < totalSize)
    {
        const size_t head = reader.GetOffset();

        auto mapKeySize = uint32_t();
        COMMAND_DO(reader.Read(&mapKeySize));

        ::std::unique_ptr<char[]> mapKey(new char[mapKeySize]());
        COMMAND_DO(reader.Read(mapKey.get(), mapKeySize));

        const size_t index =
            ::std::string(mapKey.get(), mapKeySize - 1).find("!");

        const ::std::string name(mapKey.get(), index);

        auto type = uint8_t();
        COMMAND_DO(reader.Read(&type));

        auto mapValueSize = uint32_t();
        COMMAND_DO(reader.Read(&mapValueSize));

        COMMAND_DO(reader.Proceed(mapValueSize));

        if (!Contains(names, name)) { continue; }

        const size_t tail = reader.GetOffset();

        ::std::copy(&outBuffer[head], &outBuffer[tail], &outBuffer[offset]);

        offset += static_cast<uint32_t>(tail - head);
    }

    ::std::memset(&outBuffer[offset], 0, reader.GetCount() - offset);

    ::std::memcpy(outBuffer, &offset, sizeof(offset));

    return true;
}

namespace {

ByteReader::ByteReader(const ::nn::Bit8 buffer[], size_t count) NN_NOEXCEPT
    : m_Buffer(buffer)
    , m_Count(count)
    , m_Offset(0)
{
    // 何もしない
}

size_t ByteReader::GetCount() const NN_NOEXCEPT
{
    return m_Count;
}

size_t ByteReader::GetOffset() const NN_NOEXCEPT
{
    return m_Offset;
}

bool ByteReader::Proceed(size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(m_Offset + count, m_Count);

    m_Offset += count;

    return true;
}

bool ByteReader::Read(char outBuffer[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_LESS_EQUAL(m_Offset + count, m_Count);

    const size_t offset = m_Offset + count;

    auto buffer = reinterpret_cast<::nn::Bit8*>(outBuffer);

    ::std::copy(&m_Buffer[m_Offset], &m_Buffer[offset], buffer);

    m_Offset = offset;

    return true;
}

} // namespace
