﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_File.h>

//!< ファイルシステム用のアロケータです。
void* AllocateForFileSystem(size_t size) NN_NOEXCEPT;

//!< ファイルシステム用のデアロケータです。
void DeallocateForFileSystem(void* addr, size_t size) NN_NOEXCEPT;

//!< ファイルを扱うクラスです。
class IFile
{
public:
    virtual ~IFile() NN_NOEXCEPT { /* 何もしない */ }

    //!< ファイルパスを返します。
    virtual const ::std::string& GetPath() const NN_NOEXCEPT = 0;

    //!< ファイルハンドルを返します。
    virtual ::nn::fs::FileHandle GetHandle() const NN_NOEXCEPT = 0;

    //!< ファイルをオープンします。
    virtual ::nn::Result Open(
        const ::std::string& path, int mode) NN_NOEXCEPT = 0;
};

//!< ホストファイルシステム上のファイルを扱うクラスです。
class HostFsFile final : public IFile
{
    NN_DISALLOW_COPY(HostFsFile);
    NN_DISALLOW_MOVE(HostFsFile);

private:
    //!< ファイルがオープンされているか否かを表す値
    bool m_IsOpen;

    //!< マウント名
    ::std::string m_MountName;

    //!< ファイルパス
    ::std::string m_Path;

    //!< ファイルハンドル
    ::nn::fs::FileHandle m_Handle;

public:
    HostFsFile() NN_NOEXCEPT;

    virtual ~HostFsFile() NN_NOEXCEPT NN_OVERRIDE;

    virtual const ::std::string& GetPath() const NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::fs::FileHandle GetHandle() const NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Open(
        const ::std::string& path, int mode) NN_NOEXCEPT NN_OVERRIDE;

private:
    //!< ファイルをクローズします。
    void Close() NN_NOEXCEPT;
};

//!< ファイルを開きます。
bool OpenFile(IFile* pFile, const ::std::string& path, int mode) NN_NOEXCEPT;

//!< ファイルサイズを設定します。
bool SetFileSize(const IFile& file, int64_t size) NN_NOEXCEPT;

//!< ファイルサイズを取得します。
bool GetFileSize(int64_t* pOutValue, const IFile& file) NN_NOEXCEPT;

//!< ファイルの内容を取得します。
bool ReadFile(
    size_t* pOutSize, char* buffer, size_t bufferSize,
    const IFile& file, int64_t offset) NN_NOEXCEPT;

//!< ファイルへ指定された内容を書き込みます。
bool WriteFile(
    const IFile& file, int64_t offset,
    const char* buffer, size_t bufferSize) NN_NOEXCEPT;

//!< ファイルへの書き込みをフラッシュします。
bool FlushFile(const IFile& file) NN_NOEXCEPT;
