﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/bconfig/bconfig_Api.h>
#include <nn/TargetConfigs/build_Base.h>

#include "SettingsManager_Debug.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< 開発支援機能設定のキー
const char* const SettingNameDebugSettings = "debug_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< ユーザー例外ハンドラの呼び出しを行うか否かを表すフラグのキー
const char* const KeyUserExceptionHandlerFlag = "user_exception_handler_flag";

//!< ユーザー例外ハンドラの呼び出しを行うか否かを表すフラグをエクスポートします。
bool ExportUserExceptionHandlerFlag(Node* pDictNode, bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyUserExceptionHandlerFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(currentValueNode.SetValue(value));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyUserExceptionHandlerFlag, ::std::move(node)));

    return true;
}

//!< ユーザー例外ハンドラの呼び出しを行うか否かを表すフラグをインポートします。
bool ImportUserExceptionHandlerFlag(
    ::nn::bconfig::BootConfig* pOutValue,
    const Node& dictNode,
    const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const char* const key = KeyUserExceptionHandlerFlag;

    if (!Contains(keys, ::std::string(key)))
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    ::nn::bconfig::SetEnableUserExceptionHandler(pOutValue, value);

    return true;
}

#ifdef NN_BUILD_CONFIG_OS_WIN
//!< 開発支援機能設定を保持するクラスです。
class DebugSettingsHolder final
{
    NN_DISALLOW_COPY(DebugSettingsHolder);
    NN_DISALLOW_MOVE(DebugSettingsHolder);

private:
    bool m_IsUserExceptionHandlerFlagAvailable;

    bool m_UserExceptionHandlerFlag;

public:
    DebugSettingsHolder() NN_NOEXCEPT;

    bool IsUserExceptionHandlerFlagAvailable() const NN_NOEXCEPT;

    bool GetUserExceptionHandlerFlag() const NN_NOEXCEPT;

    void SetUserExceptionHandlerFlag(bool value) NN_NOEXCEPT;
};

//!< 開発支援機能設定ホルダを返します。
DebugSettingsHolder& GetDebugSettingsHolder() NN_NOEXCEPT;
#endif

} // namespace

bool IsSettingNameDebugSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameDebugSettings);
}

#ifdef NN_BUILD_CONFIG_OS_HORIZON
bool ExportDebugSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameDebugSettings);

        ::nn::bconfig::BootConfig bootConfig = {};

        ::nn::bconfig::LoadBootConfig(&bootConfig);

        COMMAND_DO(
            ExportUserExceptionHandlerFlag(
                &node,
                ::nn::bconfig::GetEnableUserExceptionHandler(&bootConfig)));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameDebugSettings, ::std::move(node)));

    return true;
}
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
bool ExportDebugSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    if (!GetDebugSettingsHolder().IsUserExceptionHandlerFlagAvailable())
    {
        return true;
    }

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameDebugSettings);

        COMMAND_DO(
            ExportUserExceptionHandlerFlag(
                &node,
                GetDebugSettingsHolder().GetUserExceptionHandlerFlag()));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameDebugSettings, ::std::move(node)));

    return true;
}
#endif

#ifdef NN_BUILD_CONFIG_OS_HORIZON
bool ImportDebugSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameDebugSettings);

    ::std::vector<::std::string> keys;

    COMMAND_DO(node.GetKeys(&keys));

    ::nn::bconfig::BootConfig bootConfig = {};

    ::nn::bconfig::LoadBootConfig(&bootConfig);

    COMMAND_DO(ImportUserExceptionHandlerFlag(&bootConfig, node, keys));

    ::nn::bconfig::SaveBootConfig(&bootConfig);

    return true;
}
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
bool ImportDebugSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameDebugSettings);

    ::std::vector<::std::string> keys;

    COMMAND_DO(node.GetKeys(&keys));

    ::nn::bconfig::BootConfig bootConfig = {};

    COMMAND_DO(ImportUserExceptionHandlerFlag(&bootConfig, node, keys));

    return true;
}
#endif

#ifdef NN_BUILD_CONFIG_OS_HORIZON
bool DumpDebugSettings(TemporaryDatabase* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::nn::bconfig::BootConfig bootConfig = {};

    ::nn::bconfig::LoadBootConfig(&bootConfig);

    pOutValue->userExceptionHandlerFlag =
        static_cast<int32_t>(
            ::nn::bconfig::GetEnableUserExceptionHandler(&bootConfig));

    return true;
}
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
bool DumpDebugSettings(TemporaryDatabase* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->userExceptionHandlerFlag = int32_t();

    return true;
}
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
bool LoadDebugSettings(const TemporaryDatabase& value) NN_NOEXCEPT
{
    GetDebugSettingsHolder().SetUserExceptionHandlerFlag(
        value.userExceptionHandlerFlag == 1);

    return true;
}
#endif

namespace {

#ifdef NN_BUILD_CONFIG_OS_WIN
DebugSettingsHolder::DebugSettingsHolder() NN_NOEXCEPT
    : m_IsUserExceptionHandlerFlagAvailable(false)
    , m_UserExceptionHandlerFlag()
{
    // 何もしない
}

bool DebugSettingsHolder::IsUserExceptionHandlerFlagAvailable(
    ) const NN_NOEXCEPT
{
    return m_IsUserExceptionHandlerFlagAvailable;
}

bool DebugSettingsHolder::GetUserExceptionHandlerFlag() const NN_NOEXCEPT
{
    return m_UserExceptionHandlerFlag;
}

void DebugSettingsHolder::SetUserExceptionHandlerFlag(bool value) NN_NOEXCEPT
{
    m_IsUserExceptionHandlerFlagAvailable = true;

    m_UserExceptionHandlerFlag = value;
}

DebugSettingsHolder& GetDebugSettingsHolder() NN_NOEXCEPT
{
    static DebugSettingsHolder s_DebugSettingsHolder;

    return s_DebugSettingsHolder;
}
#endif

} // namespace
