﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include "detail/settings_KeyValueStore.h"
#include "detail/settings_PrivateSettingsDatabase.h"
#include "detail/settings_SystemSaveData-os.win.h"
#include "detail/settings_SystemSettingsDatabase.h"
#include "SettingsManager_Command.h"
#include "SettingsManager_Debug.h"
#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_TemporaryDatabase.h"
#include "SettingsManager_Utility.h"

bool LoadSettings(::std::vector<int64_t>* buffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    const size_t bufferSize = buffer->size() * sizeof(int64_t);

    if (bufferSize != 6 * SettingsSectionSize)
    {
        // データサイズが不正
        PrintErrorCode(ErrorCode::DataSizeInvalid, bufferSize);

        return false;
    }
    else
    {
        auto ptr = reinterpret_cast<::nn::Bit8*>(buffer->data());

        auto pDatabase = reinterpret_cast<TemporaryDatabase*>(ptr);
        COMMAND_DO(LoadDebugSettings(*pDatabase));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            ::nn::settings::detail::EnableSystemSettingsInMemoryModeForDebug(
                &ptr[SettingsSectionSize * 1], SettingsSectionSize));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            ::nn::settings::detail::EnablePrivateSettingsInMemoryModeForDebug(
                &ptr[SettingsSectionSize * 2], SettingsSectionSize));

        ::nn::settings::detail::SystemSaveData systemSaveData;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            systemSaveData.EnableInMemoryModeForDebug(
                &ptr[SettingsSectionSize * 3], SettingsSectionSize));

        ::nn::settings::detail::SystemSaveData fwdbgSystemData;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            fwdbgSystemData.EnableInMemoryModeForDebug(
                &ptr[SettingsSectionSize * 4], SettingsSectionSize));

        ::nn::settings::detail::SystemSaveData pfCfgSystemData;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            pfCfgSystemData.EnableInMemoryModeForDebug(
                &ptr[SettingsSectionSize * 5], SettingsSectionSize));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            ::nn::settings::detail::ReloadKeyValueStoreForDebug(
                &systemSaveData, &fwdbgSystemData, &pfCfgSystemData));

        return true;
    }
}
