﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/cal/cal.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include "RepairSslCertificate_SerialMap.h"

extern nn::Bit8 ConcatKeyDer_begin[];
extern nn::Bit8 ConcatKeyDer_end[];

const int keySize = 272;

nn::cal::SslKey keyData;

bool GetSslCertificateIndex(uint32_t* pOutOffset, uint32_t* pOutSize, nn::settings::factory::SerialNumber* serial)
{
    uint32_t offset = 0;
    for (auto&& it : data)
    {
        if (strncmp(serial->string, it.serial, 14)==0)
        {
            *pOutOffset = offset;
            *pOutSize = it.size;
            return true;
        }
        offset += it.size + keySize;
    }
    return false;
}

void dumpBinary(nn::Bit8* address, size_t size)
{
    for (size_t i=0; i<size; ++i)
    {
        if (i % 32 == 0)
        {
            NN_LOG("\n");
        }
        NN_LOG("%02X ", address[i]);
    }
    NN_LOG("\n");
}

extern "C" void nnMain()
{
    nn::settings::factory::SerialNumber serial;

    NN_LOG("--------------------------------\n");
    NN_LOG(" Repair SSL Certificate\n");
    NN_LOG("--------------------------------\n\n");

    // 本体のシリアルを取得
    auto result = nn::settings::factory::GetSerialNumber(&serial);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_LOG("Serial number: %s\n\n", serial.string);

    // シリアルに対応する証明書のインデックスを取得
    uint32_t offset;
    uint32_t derSize;
    if (!GetSslCertificateIndex(&offset, &derSize, &serial))
    {
        // シリアルに対応する証明書がなかった（修正の必要なしのはず）
        NN_LOG("Data not found for this target.\n");
        return;
    }

    // dump
#if 0
    dumpBinary(ConcatKeyDer_begin + offset, keySize);
    dumpBinary(ConcatKeyDer_begin + offset + keySize, derSize);
#endif

    // address
    nn::Bit8* keyAddress = ConcatKeyDer_begin + offset;
    nn::Bit8* derAddress = ConcatKeyDer_begin + offset + keySize;

    // copy
    memcpy(keyData.data, keyAddress, keySize);

    // Write
    NN_LOG("Start repairing.\n");
    result = nn::cal::WriteSslKey(keyData);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::cal::WriteSslCertificate(derAddress, derSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_LOG("Repair completed.\n");

    return;
}
