﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "DevMenuCommand_CommandTypes.h"
#include "DevMenuCommand_CommandImplUtility.h"
#include "DevMenuCommand_Processor.h"

#include <utility>
#include <string>
#include <functional>
#include <tuple>
#include <vector>
#include <memory>
#include <type_traits>

namespace nn { namespace devmenucommand {

template <typename F>
struct ValueSetter
{
    F f;
};

template <typename F>
inline auto MakeValueSetter(F&& f) -> ValueSetter<typename std::decay<F>::type>
{
    return {std::forward<F>(f)};
}

template <typename F>
struct ValueGetter
{
    F f;
};

template <typename F>
inline auto MakeValueGetter(F&& f) -> ValueGetter<typename std::decay<F>::type>
{
    return {std::forward<F>(f)};
}

namespace detail {

class ValueCommandBuilderBase
    : public ICommandBuilder
    , public std::enable_shared_from_this<ValueCommandBuilderBase>
{
private:

    std::string m_Name;
    Visibility m_GetterVisibility = Visibility::Public;
    Visibility m_SetterVisibility = Visibility::Public;
    ReflectionTime m_ReflectionTime = ReflectionTime::Instant;
    DeprecatedT m_Deprecated = {false};

public:

    void SetName(std::string value) NN_NOEXCEPT
    {
        this->m_Name = std::move(value);
    }

    const std::string& GetName() const NN_NOEXCEPT
    {
        return m_Name;
    }

    void SetProperty(Visibility value) NN_NOEXCEPT
    {
        this->m_GetterVisibility = value;
        this->m_SetterVisibility = value;
    }

    Visibility GetGetterVisibility() const NN_NOEXCEPT
    {
        return m_GetterVisibility;
    }

    Visibility GetSetterVisibility() const NN_NOEXCEPT
    {
        return m_SetterVisibility;
    }

    void SetProperty(ReflectionTime value) NN_NOEXCEPT
    {
        this->m_ReflectionTime = value;
    }

    ReflectionTime GetReflectionTime() const NN_NOEXCEPT
    {
        return m_ReflectionTime;
    }

    void SetProperty(DeprecatedT deprecated) NN_NOEXCEPT
    {
        this->m_Deprecated = deprecated;
    }

    bool IsDeprecated() const NN_NOEXCEPT
    {
        return this->m_Deprecated._value;
    }

private:

    virtual ProcessResult DoSetValue(const std::string& s) const NN_NOEXCEPT = 0;
    virtual ProcessResult DoGetValue(std::string* pOut) const NN_NOEXCEPT = 0;

protected:

    Result SetValueImpl(bool* outValue, const std::string&) const NN_NOEXCEPT;
    Result GetValueImpl(bool* outValue) const NN_NOEXCEPT;
    std::function<Result(bool* outValue, const Option&)> MakeSetterFunction() const NN_NOEXCEPT;
    std::function<Result(bool* outValue, const Option&)> MakeGetterFunction() const NN_NOEXCEPT;
    std::function<Result(bool* outValue, const Option&)> MakeDefaultSetterFunction(const std::string& value) const NN_NOEXCEPT;

    SubCommand MakeGetterSubCommand(const std::string& name) const NN_NOEXCEPT;
    SubCommand MakeSetterSubCommand(const std::string& name) const NN_NOEXCEPT;

    virtual std::vector<SubCommand> MakeSubCommands() const NN_NOEXCEPT NN_OVERRIDE;
    virtual std::string GetArgumentHelpMessage() const NN_NOEXCEPT = 0;

};

template <typename T>
class ValueCommandBuilderBaseT
    : public ValueCommandBuilderBase
{
private:

    std::function<ProcessResult(T* pOut, const std::string& s, const std::string& valueName)> m_Deserialize;
    std::function<ProcessResult(const T&)> m_Setter;
    std::function<ProcessResult(T* pOut)> m_Getter;
    std::function<std::string(const T&)> m_Serialize;
    std::function<std::string()> m_GetArgumentHelpMessage;
    std::vector<std::function<ProcessResult(const T& x, const std::string& valueName)>> m_Validators;

    virtual ProcessResult DoSetValue(const std::string& s) const NN_NOEXCEPT NN_OVERRIDE
    {
        T v;
        auto deserializeResult = m_Deserialize(&v, s, "value");
        if (!deserializeResult)
        {
            return deserializeResult;
        }
        for (auto&& validate : m_Validators)
        {
            auto validateResult = validate(v, "value");
            if (!validateResult)
            {
                return validateResult;
            }
        }
        return m_Setter(v);
    }

    virtual ProcessResult DoGetValue(std::string* pOut) const NN_NOEXCEPT NN_OVERRIDE
    {
        T v;
        auto processResult = m_Getter(&v);
        if (!processResult)
        {
            return processResult;
        }
        *pOut = m_Serialize(v);
        return ProcessResultSuccess;
    }

protected:

    virtual std::string GetArgumentHelpMessage() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_GetArgumentHelpMessage();
    }

public:

    using ValueCommandBuilderBase::SetProperty;

    template <typename F>
    void SetProperty(const ParserWrapper<F>& parser) NN_NOEXCEPT
    {
        this->m_Deserialize = parser.value;
    }

    template <typename F>
    void SetProperty(const ValueToStringTagWrapper<F>& stringConverter) NN_NOEXCEPT
    {
        this->m_Serialize = stringConverter.value;
    }

    template <typename F>
    void SetProperty(const ValueSetter<F>& setter) NN_NOEXCEPT
    {
        this->m_Setter = setter.f;
    }

    template <typename F>
    void SetProperty(const ValueGetter<F>& getter) NN_NOEXCEPT
    {
        this->m_Getter = getter.f;
    }

    template <typename F>
    void SetProperty(const Wrapper<F, ValidatorTag>& x) NN_NOEXCEPT
    {
        m_Validators.push_back(x.value);
    }

    void SetProperty(const ArgumentHelpMessageHolder& holder) NN_NOEXCEPT
    {
        this->m_GetArgumentHelpMessage = holder.getArgumentHelpMessage;
    }

};

template <typename T>
class ValueCommandBuilder
    : public ValueCommandBuilderBaseT<T>
{
};

template <>
class ValueCommandBuilder<bool>
    : public ValueCommandBuilderBaseT<bool>
{
private:
    using Derived = ValueCommandBuilderBaseT<bool>;
protected:
    virtual std::vector<SubCommand> MakeSubCommands() const NN_NOEXCEPT NN_OVERRIDE;
};

} // detail

}}
