﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "DevMenuCommand_CommandTypes.h"
#include "DevMenuCommand_CommandImplUtility.h"
#include "DevMenuCommand_Processor.h"
#include "../DevMenuCommand_Option.h"

#include <utility>
#include <string>
#include <type_traits>
#include <memory>

// impl
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace devmenucommand {

namespace detail {

template <typename... Types>
class SingleCommandBuilder
    : public ICommandBuilder
{
private:

    using Processor = Processor<Types...>;

    std::shared_ptr<Processor> m_pProcessor = std::make_shared<Processor>();

    std::string m_Name;
    Visibility m_Visibility = Visibility::Public;
    DeprecatedT m_Deprecated = {false};
    ReflectionTime m_ReflectionTime = ReflectionTime::Instant;
    std::string m_HelpMessage = "";

    static void WarnRestart(ReflectionTime reflectionTime) NN_NOEXCEPT
    {
        switch (reflectionTime)
        {
            case ReflectionTime::NeedsRelaunchApplication:
            {
                NN_LOG("Please restart the running application (if running) to apply this change.\n");
                return;
            }
            case ReflectionTime::NeedsRebootSystem:
            {
                NN_LOG("Please restart your target to apply this change.\n");
                return;
            }
            default: return;
        }
    }

    static Result ProcessImpl(bool* outValue, const Option& option, Processor* p, ReflectionTime reflectionTime) NN_NOEXCEPT
    {
        OptionData data;
        data.attributes = option.GetMap();
        for (int i = 0; i < option.GetTargetCount(); ++i)
        {
            data.args.push_back(option.GetTarget(i));
        }
        auto r = p->operator()(data);
        NN_LOG("%s\n", r.message.c_str());
        if (!r)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        WarnRestart(reflectionTime);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

public:

    void SetParameters(const Parameters<Types...>& parameters) NN_NOEXCEPT
    {
        m_pProcessor->SetParameters(parameters);
    }

    template <typename Property>
    void SetProperty(Property&& property) NN_NOEXCEPT
    {
        m_pProcessor->SetProperty(std::forward<Property>(property));
    }

    void SetName(const std::string& name) NN_NOEXCEPT
    {
        this->m_Name = name;
    }

    const std::string& GetName() const NN_NOEXCEPT
    {
        return m_Name;
    }

    void SetProperty(Visibility value) NN_NOEXCEPT
    {
        this->m_Visibility = value;
    }

    Visibility GetVisibility() const NN_NOEXCEPT
    {
        return m_Visibility;
    }

    void SetProperty(ReflectionTime value) NN_NOEXCEPT
    {
        this->m_ReflectionTime = value;
    }

    ReflectionTime GetReflectionTime() const NN_NOEXCEPT
    {
        return m_ReflectionTime;
    }

    void SetProperty(DeprecatedT deprecated) NN_NOEXCEPT
    {
        this->m_Deprecated = deprecated;
    }

    bool IsDeprecated() const NN_NOEXCEPT
    {
        return this->m_Deprecated._value;
    }

    virtual std::vector<SubCommand> MakeSubCommands() const NN_NOEXCEPT NN_OVERRIDE
    {
        return
        {{
            m_Name,
            [p = this->m_pProcessor, reflectionTime = this->m_ReflectionTime](bool* outValue, const Option& option) -> Result
            {
                return ProcessImpl(outValue, option, p.get(), reflectionTime);
            },
            m_Visibility,
            m_Deprecated._value,
            m_pProcessor->GetHelpMessage(),
        }};
    }

};


template <typename>
struct MakeSingleCommandBuilderType;

template <typename... Types>
struct MakeSingleCommandBuilderType<std::tuple<Parameter<Types>...>>
{
    using type = detail::SingleCommandBuilder<Types...>;
};

}

template <typename Parameters>
inline auto MakeSingleCommandBuilder(const std::string& name, const Parameters& parameters)
{
    using Builder = typename detail::MakeSingleCommandBuilderType<Parameters>::type;
    auto p = std::make_shared<Builder>();
    p->SetParameters(parameters);
    p->SetName(name);
    return p;
};

namespace abbreviation
{

template <typename Parameters, typename F, typename... Properties>
inline auto Command(const std::string& name, const Parameters& parameters, F&& f, Properties&&... properties)
{
    auto p = MakeSingleCommandBuilder(name, parameters);
    p->SetProperty(Wrap<ProcessorTag>(std::forward<F>(f)));
    SetProperties(*p, std::forward<Properties>(properties)...);
    return p;
};

template <typename... Types>
inline auto Params(Parameter<Types>&&... parameters)
{
    return std::make_tuple(std::forward<Parameter<Types>>(parameters)...);
}

template <typename T, typename... Properties>
inline auto Param(Properties&&... properties)
{
    return MakeParameter<T>(std::forward<Properties>(properties)...);
}

template <typename... Properties>
inline auto Int(Properties&&... properties)
{
    return MakeParameter<int>(std::forward<Properties>(properties)...);
}

template <typename... Properties>
inline auto Bool(Properties&&... properties)
{
    return MakeParameter<bool>(std::forward<Properties>(properties)...);
}

template <typename... Properties>
inline auto String(Properties&&... properties)
{
    return MakeParameter<std::string>(std::forward<Properties>(properties)...);
}

inline auto CmdOpt(const std::string& s)
{
    auto&& v = SplitString(s, ',');
    NN_ASSERT(v.size() > 0);
    return std::make_tuple(
        MakeOptionDataAttributeExtractor(v),
        MakeParameterString("value of '" + v[0] + "'"),
        MakeHelpString([option = v[0]](auto&& valueString) { return option + " " + valueString; })
    );
}

inline auto CmdFlag(const std::string& s)
{
    auto&& v = SplitString(s, ',');
    NN_ASSERT(v.size() > 0);
    return std::make_tuple(
        MakeOptionDataAttributeFlagExtractor(v),
        MakeParameterString("value of '" + v[0] + "'"),
        MakeHelpString([option = v[0]](auto&&) { return "[" + option + "]"; })
    );
}

inline auto CmdArg(int index = 0)
{
    return std::make_tuple(
        MakeOptionDataArgumentExtractor(index),
        MakeParameterString("argument"),
        MakeHelpString([](auto&& valueString) { return valueString; })
    );
}

}

}}
