﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommandAlbum_ExecuteDownloadAction.h"

#include <string>
#include <vector>
#include <algorithm>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/fs.h>
#include <nn/os.h>

#include "DevMenuCommandAlbum_StorageUtility.h"

namespace album {

namespace {

    const size_t MemoryHeapAddendSizeMax = 100ull * 1024 * 1024;

    size_t g_OriginalMemoryHeapSize;
    size_t g_BoostedMemoryHeapAddendSize = 0;
    void*  g_BoostedMemoryHeapAddendAddress = nullptr;

    // メモリヒープを大きくして、そこからメモリを確保する
    nn::Result BoostAndAllocateMemoryHeap() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(g_BoostedMemoryHeapAddendSize == 0ull);

        nn::os::MemoryInfo info;
        nn::os::QueryMemoryInfo( &info );
        g_OriginalMemoryHeapSize = info.totalMemoryHeapSize;

        size_t addend = static_cast<size_t>(info.totalAvailableMemorySize - info.totalUsedMemorySize);
        addend = std::min(addend, MemoryHeapAddendSizeMax);
        addend = nn::util::align_down(addend, nn::os::MemoryHeapUnitSize);
        NN_RESULT_THROW_UNLESS(addend > 0ull, nn::os::ResultOutOfMemory());
        NN_RESULT_DO(nn::os::SetMemoryHeapSize(g_OriginalMemoryHeapSize + addend));
        g_BoostedMemoryHeapAddendSize = addend;

        uintptr_t address;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, addend));
        g_BoostedMemoryHeapAddendAddress = reinterpret_cast<void*>(address);
        NN_RESULT_SUCCESS;
    }

    // メモリヒープを元に戻す
    void UnboostAndFreeMemoryHeap() NN_NOEXCEPT
    {
        nn::os::FreeMemoryBlock(reinterpret_cast<uintptr_t>(g_BoostedMemoryHeapAddendAddress), g_BoostedMemoryHeapAddendSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(g_OriginalMemoryHeapSize));
        g_BoostedMemoryHeapAddendSize    = 0;
        g_BoostedMemoryHeapAddendAddress = nullptr;
    }

}   // namespace

nn::Result AllocateCopyBuffer() NN_NOEXCEPT
{
    return BoostAndAllocateMemoryHeap();
}

void DeallocateCopyBuffer() NN_NOEXCEPT
{
    UnboostAndFreeMemoryHeap();
}

void* GetCopyBufferAddress() NN_NOEXCEPT
{
    return g_BoostedMemoryHeapAddendAddress;
}

size_t GetCopyBufferSize() NN_NOEXCEPT
{
    return g_BoostedMemoryHeapAddendSize;
}

}   // namespace album
