﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <string>
#include <vector>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemDataUpdateEventPrivate.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/nifm.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/updater/updater.h>
#include <nn/updater/updater_DebugApi.h>
#include <nn/idle/idle_SystemApi.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_Stopwatch.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_SystemCommon.h"
#include "DevMenuCommand_SystemProgramCommand.h"

using namespace nn;
using namespace nne;

namespace
{
    NN_ALIGNAS(4096) uint8_t BootImageUpdateBuffer[64 << 10];

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " systemupdate get-background-state\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate check-latest\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate check-downloaded\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate apply-downloaded\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate download-latest\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate destroy-task\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate get-download-progress\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate get-device-id\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate get-debug-id\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate set-debug-id <debug_id>-<debug_version>\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate set-debug-id-for-content-delivery <debug_id>-<debug_version>\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate invalidate-debug-id\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate request-bgnup\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate check-cardupdate\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate do-cardupdate [--timeout <msec>]\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate get-cardupdate-euladata <eulaDataPath>\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate update-bootimages <bootImagePackageId>\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate update-bootimages-safe <bootImagePackageId>\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate mark-verifying-required <flag>\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate mark-verifying-required-safe <flag>\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate get-verifying-required-mark\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate get-bootimage-package-id\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate get-bootimage-package-id-safe\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate get-bootimage-hashes\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate break-bootimages <phase>\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate break-bootimages-safe <phase>\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate notify-exfat-required\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate notify-exfat-downloaded\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate clear-exfat-status\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate check-content-delivery\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate notify-content-delivery\n"
        "       " DEVMENUCOMMAND_NAME " systemupdate notify-systemdata-update\n"
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    Result GetBackgroundStateCommand(bool* outValue, const Option&)
    {
        auto state = ns::GetBackgroundNetworkUpdateState();

        std::string stateString;
        switch (state)
        {
        case ns::BackgroundNetworkUpdateState::None: stateString = "None"; break;
        case ns::BackgroundNetworkUpdateState::InProgress: stateString = "InProgress"; break;
        case ns::BackgroundNetworkUpdateState::Ready: stateString = "Ready"; break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        DEVMENUCOMMAND_LOG("%s\n", stateString.c_str());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckLatestCommand(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("Waiting network request accepted...\n");
        {
            Stopwatch s;
            nifm::SubmitNetworkRequestAndWait();
        }

        DEVMENUCOMMAND_LOG("Waiting occupy system update control...\n");
        ns::SystemUpdateControl control;
        {
            Stopwatch s;
            NN_RESULT_DO(control.Occupy());
        }

        DEVMENUCOMMAND_LOG("Waiting check latest update...\n");
        ns::LatestSystemUpdate latest;
        {
            Stopwatch s;
            ns::AsyncLatestSystemUpdate asyncLatest;
            NN_RESULT_DO(control.RequestCheckLatestUpdate(&asyncLatest));
            NN_RESULT_DO(asyncLatest.Get(&latest));
        }

        std::string latestString;
        switch (latest)
        {
        case ns::LatestSystemUpdate::NeedsDownload: latestString = "NeedsDownload"; break;
        case ns::LatestSystemUpdate::Downloaded: latestString = "Downloaded"; break;
        case ns::LatestSystemUpdate::UpToDate: latestString = "UpToDate"; break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        DEVMENUCOMMAND_LOG("%s\n", latestString.c_str());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckDownloadCommand(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("Waiting occupy system update control...\n");
        ns::SystemUpdateControl control;
        {
            Stopwatch s;
            NN_RESULT_DO(control.Occupy());
        }

        DEVMENUCOMMAND_LOG("%s\n", control.HasDownloaded() ? "true" : "false");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ApplyDownloadCommand(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("Waiting occupy system update control...\n");
        ns::SystemUpdateControl control;
        {
            Stopwatch s;
            NN_RESULT_DO(control.Occupy());
        }

        if (!control.HasDownloaded())
        {
            DEVMENUCOMMAND_LOG("System update not downloaded.\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Waiting apply downloaded update...\n");
        {
            Stopwatch s;
            control.ApplyDownloadedUpdate();
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DestroyTaskCommand(bool* outValue, const Option&)
    {
        NN_RESULT_DO(ns::DestroySystemUpdateTask());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DownloadLatestCommand(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("Waiting network request accepted...\n");
        {
            Stopwatch s;
            nifm::SubmitNetworkRequestAndWait();
        }

        DEVMENUCOMMAND_LOG("Waiting occupy system update control...\n");
        ns::SystemUpdateControl control;
        {
            Stopwatch s;
            NN_RESULT_DO(control.Occupy());
        }

        DEVMENUCOMMAND_LOG("Waiting latest system update downloaded...\n");
        {
            Stopwatch s;
            ns::AsyncResult asyncResult;
            NN_RESULT_DO(control.RequestDownloadLatestUpdate(&asyncResult));

            while (!asyncResult.TryWait())
            {
                auto progress = control.GetDownloadProgress();
                DEVMENUCOMMAND_LOG("%12lld / %lld\n", progress.loaded, progress.total);

#ifdef NN_BUILD_CONFIG_OS_HORIZON
                nn::idle::ReportUserIsActive();
#endif
                os::SleepThread(TimeSpan::FromSeconds(1));
            }

            NN_RESULT_DO(asyncResult.Get());
            auto progress = control.GetDownloadProgress();
            DEVMENUCOMMAND_LOG("%12lld / %lld\n", progress.loaded, progress.total);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetDownloadProgressCommand(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("Waiting occupy system update control...\n");
        ns::SystemUpdateControl control;
        {
            Stopwatch s;
            NN_RESULT_DO(control.Occupy());
        }

        auto progress = control.GetDownloadProgress();
        DEVMENUCOMMAND_LOG("%12lld / %lld\n", progress.loaded, progress.total);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetDeviceIdCommand(bool* outValue, const Option&)
    {
        EciDeviceCertificate cert;
        NN_RESULT_DO(GetEciDeviceCertificate(&cert));

        char deviceIdString[32] = {};
        std::memcpy(deviceIdString, cert.name.id, sizeof(cert.name.id));
        Bit64 deviceId = STR_TO_ULL(deviceIdString, nullptr, 16);

        DEVMENUCOMMAND_LOG("%016llx\n", deviceId);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetDebugIdCommand(bool* outValue, const Option&)
    {
        char debugIdString[64];
        settings::fwdbg::GetSettingsItemValue(debugIdString, sizeof(debugIdString), "systemupdate", "debug_id");
        auto debugId = STR_TO_ULL(debugIdString, nullptr, 16);

        uint32_t debugVersion;
        settings::fwdbg::GetSettingsItemValue(&debugVersion, sizeof(debugVersion), "systemupdate", "debug_version");

        DEVMENUCOMMAND_LOG("0x%016llx-%u\n", debugId, debugVersion);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetDebugIdImpl(bool* outValue, const Option& option, const char* idLabel, const char* versionLabel)
    {
        auto debugIdParam = option.GetTarget();
        if (!debugIdParam)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char* endPtr;
        auto id = STR_TO_ULL(debugIdParam, &endPtr, 16);
        if (*endPtr != '-')
        {
            *outValue = false;
            DEVMENUCOMMAND_LOG("%s is invalid debug id format. It must be like \"0x0123456789abcdef-0\".\n", debugIdParam);
            NN_RESULT_SUCCESS;
        }

        char* endPtr2;
        auto version = static_cast<uint32_t>(STR_TO_ULL(++endPtr, &endPtr2, 10));
        if (*endPtr2 != '\0')
        {
            *outValue = false;
            DEVMENUCOMMAND_LOG("%s is invalid debug id format. It must be like \"0x0123456789abcdef-0\".\n", debugIdParam);
            NN_RESULT_SUCCESS;
        }

        char debugIdString[sizeof("0x0123456789abcdef")];
        util::SNPrintf(debugIdString, sizeof(debugIdString), "0x%016llx", id);

        settings::fwdbg::SetSettingsItemValue("systemupdate", idLabel, debugIdString, sizeof(debugIdString));
        settings::fwdbg::SetSettingsItemValue("systemupdate", versionLabel, &version, sizeof(version));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetDebugIdCommand(bool* outValue, const Option& option)
    {
        return SetDebugIdImpl(outValue, option, "debug_id", "debug_version");
    }

    Result SetDebugIdForContentDeliveryCommand(bool* outValue, const Option& option)
    {
        return SetDebugIdImpl(outValue, option, "debug_id_for_content_delivery", "debug_version_for_content_delivery");

    }

    Result InvalidateDebugIdCommand(bool* outValue, const Option&)
    {
        static const char invalidId[] = "0x0000000000000000";
        settings::fwdbg::SetSettingsItemValue("systemupdate", "debug_id", invalidId, sizeof(invalidId));

        static const uint32_t invalidVersion = 0;
        settings::fwdbg::SetSettingsItemValue("systemupdate", "debug_version", &invalidVersion, sizeof(invalidVersion));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RequestBgnupCommand(bool* outValue, const Option&)
    {
        NN_RESULT_DO(ns::RequestBackgroundNetworkUpdate());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckCardUpateCommand(bool* outValue, const Option&)
    {
        os::SystemEvent event;
        ns::GetGameCardUpdateDetectionEvent(&event);

        DEVMENUCOMMAND_LOG("%s\n", event.TryWait() ? "true" : "false");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DoCardUpdateCommand(bool* outValue, const Option& option)
    {
        util::optional<int64_t> timeout = util::nullopt;
        if (option.HasKey("--timeout"))
        {
            timeout = static_cast<int64_t>(STR_TO_ULL(option.GetValue("--timeout"), nullptr, 10));
        }
        auto begin = os::GetSystemTick();

        DEVMENUCOMMAND_LOG("Waiting occupy system update control...\n");
        ns::SystemUpdateControl control;
        {
            Stopwatch s;
            NN_RESULT_DO(control.Occupy());
        }

        NN_ALIGNAS(4096) static char CardUpdateBuffer[os::MemoryPageSize * 1024];

        DEVMENUCOMMAND_LOG("Waiting setup card update...\n");
        {
            Stopwatch s;
            NN_RESULT_DO(control.SetupCardUpdate(CardUpdateBuffer, sizeof(CardUpdateBuffer)));
        }

        DEVMENUCOMMAND_LOG("Waiting prepare card update...\n");
        {
            Stopwatch s;

            ns::AsyncResult asyncResult;
            NN_RESULT_DO(control.RequestPrepareCardUpdate(&asyncResult));
            while (!asyncResult.TryWait())
            {
                auto progress = control.GetPrepareCardUpdateProgress();
                DEVMENUCOMMAND_LOG("%12lld / %lld\n", progress.loaded, progress.total);

                auto span = (os::GetSystemTick() - begin).ToTimeSpan();
                if (timeout && (span.GetMilliSeconds() >= *timeout))
                {
                    DEVMENUCOMMAND_LOG("Request cancel\n");
                    asyncResult.Cancel();
                }
#ifdef NN_BUILD_CONFIG_OS_HORIZON
                nn::idle::ReportUserIsActive();
#endif
                os::SleepThread(TimeSpan::FromSeconds(1));
            }
            NN_RESULT_DO(asyncResult.Get());
        }

        DEVMENUCOMMAND_LOG("Waiting apply card update...\n");
        {
            Stopwatch s;
            control.ApplyCardUpdate();
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetCardUpdateEulaDataCommand(bool* outValue, const Option& option)
    {
        auto eulaDataPath = option.GetTarget();
        if (!eulaDataPath)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Waiting occupy system update control...\n");
        ns::SystemUpdateControl control;
        {
            Stopwatch s;
            NN_RESULT_DO(control.Occupy());
        }

        NN_ALIGNAS(4096) static char CardUpdateBuffer[os::MemoryPageSize * 1024];

        DEVMENUCOMMAND_LOG("Waiting setup card update...\n");
        {
            Stopwatch s;
            NN_RESULT_DO(control.SetupCardUpdate(CardUpdateBuffer, sizeof(CardUpdateBuffer)));
        }

        DEVMENUCOMMAND_LOG("Waiting prepare card update...\n");
        {
            Stopwatch s;

            ns::AsyncResult asyncResult;
            NN_RESULT_DO(control.RequestPrepareCardUpdate(&asyncResult));
            while (!asyncResult.TryWait())
            {
                auto progress = control.GetPrepareCardUpdateProgress();
                DEVMENUCOMMAND_LOG("%12lld / %lld\n", progress.loaded, progress.total);

#ifdef NN_BUILD_CONFIG_OS_HORIZON
                nn::idle::ReportUserIsActive();
#endif
                os::SleepThread(TimeSpan::FromSeconds(1));
            }
            NN_RESULT_DO(asyncResult.Get());
        }

        static char EulaDataBuffer[1024 * 1024];

        size_t eulaDataSize;
        NN_RESULT_DO(control.GetPreparedCardUpdateEulaData(&eulaDataSize, EulaDataBuffer, sizeof(EulaDataBuffer) - 1, eulaDataPath));
        EulaDataBuffer[eulaDataSize] = '\0';

        DEVMENUCOMMAND_LOG("%s\n", EulaDataBuffer);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    updater::BootImageUpdateType GetBootImageUpdateType() NN_NOEXCEPT
    {
        int bootImageUpdateType;
        auto size = settings::fwdbg::GetSettingsItemValue(&bootImageUpdateType, sizeof(bootImageUpdateType), "systeminitializer", "boot_image_update_type");
        // 設定項目の取得に失敗した場合は旧来の挙動にする
        if (size != sizeof(bootImageUpdateType))
        {
            return updater::BootImageUpdateType::Original;
        }
        return updater::GetBootImageUpdateType(bootImageUpdateType);
    }

    Result UpdateBootImagesCommand(bool* outValue, const Option& option)
    {
        auto idParam = option.GetTarget();

        ncm::SystemDataId id;
        id.value = 0;
        if (idParam)
        {
            id.value = STR_TO_ULL(idParam, nullptr, 16);
        }
        if (id.value == 0)
        {
            NN_RESULT_DO(updater::GetBootImagePackageId(&id, updater::TargetBootMode::Normal, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
        }
        DEVMENUCOMMAND_LOG("Update with BootImagePackage: %016llx\n", id.value);
        NN_RESULT_DO(updater::UpdateBootImagesFromPackage(id, updater::TargetBootMode::Normal, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer), GetBootImageUpdateType()));
        DEVMENUCOMMAND_LOG("Update Success\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result UpdateBootImagesSafeCommand(bool* outValue, const Option& option)
    {
        auto idParam = option.GetTarget();

        ncm::SystemDataId id;
        id.value = 0;
        if (idParam)
        {
            id.value = STR_TO_ULL(idParam, nullptr, 16);
        }
        if (id.value == 0)
        {
            NN_RESULT_DO(updater::GetBootImagePackageId(&id, updater::TargetBootMode::Safe, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
        }
        DEVMENUCOMMAND_LOG("Update with BootImagePackageSafe: %016llx\n", id.value);
        NN_RESULT_DO(updater::UpdateBootImagesFromPackage(id, updater::TargetBootMode::Safe, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer), GetBootImageUpdateType()));
        DEVMENUCOMMAND_LOG("Update Success\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result MarkVerifyingRequiredCommand(bool* outValue, const Option& option)
    {
        auto flagParam = option.GetTarget();
        if (!flagParam)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto flag = STR_TO_ULL(flagParam, nullptr, 16);
        if (flag != 0 && flag != 1)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (flag == 0)
        {
            NN_RESULT_DO(updater::MarkVerified(updater::TargetBootMode::Normal, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
            DEVMENUCOMMAND_LOG("Mark verified for normal\n");
        }
        else
        {
            NN_RESULT_DO(updater::MarkVerifyingRequired(updater::TargetBootMode::Normal, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
            DEVMENUCOMMAND_LOG("Mark verifying required for normal\n");
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result MarkVerifyingRequiredSafeCommand(bool* outValue, const Option& option)
    {
        auto flagParam = option.GetTarget();
        if (!flagParam)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto flag = STR_TO_ULL(flagParam, nullptr, 16);
        if (flag != 0 && flag != 1)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (flag == 0)
        {
            NN_RESULT_DO(updater::MarkVerified(updater::TargetBootMode::Safe, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
            DEVMENUCOMMAND_LOG("Mark verified for safe\n");
        }
        else
        {
            NN_RESULT_DO(updater::MarkVerifyingRequired(updater::TargetBootMode::Safe, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
            DEVMENUCOMMAND_LOG("Mark verifying required for safe\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result GetVerifyingRequiredMarkCommand(bool* outValue, const Option&)
    {
        updater::VerifyingRequiredFlags flags;
        NN_RESULT_DO(updater::CheckVerifyingRequired(&flags, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));

        DEVMENUCOMMAND_LOG("Normal: %d, Safe: %d\n", flags.normalRequired, flags.safeRequired);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result GetBootImagePackageIdCommand(bool* outValue, const Option&)
    {
        ncm::SystemDataId id;
        NN_RESULT_DO(updater::GetBootImagePackageId(&id, updater::TargetBootMode::Normal, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));

        DEVMENUCOMMAND_LOG("BootImagePackage: %016llx\n", id.value);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result GetBootImagePackageIdSafeCommand(bool* outValue, const Option&)
    {
        ncm::SystemDataId id;
        NN_RESULT_DO(updater::GetBootImagePackageId(&id, updater::TargetBootMode::Safe, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));

        DEVMENUCOMMAND_LOG("BootImagePackageSafe: %016llx\n", id.value);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result GetBootImageHashesCommand(bool* outValue, const Option&)
    {
        // ハッシュ先頭 4 byte を int 扱いして表示
        // エンディアンは気にしない
        auto f = [](const char* hash) -> uint32_t {return *(reinterpret_cast<const uint32_t*>(hash)); };

        static updater::BootImageHashes hashes;
        bool isExist = true;
        NN_RESULT_TRY(GetBootImageHash(&hashes, updater::TargetBootMode::Normal, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer), GetBootImageUpdateType()))
            NN_RESULT_CATCH(updater::ResultBootImagePackageNotFound) {isExist = false;}
        NN_RESULT_END_TRY
        if (isExist)
        {
            DEVMENUCOMMAND_LOG("---------------------------------------------- Normal --\n");
            DEVMENUCOMMAND_LOG("%10s | %8s | %8s | %8s | %8s |\n", "type", "main", "sub", "file", "size");
            DEVMENUCOMMAND_LOG("%10s | %08x | %08x | %08x | %8d |\n", "bct", f(hashes.bctMain), f(hashes.bctSub), f(hashes.bctFile), hashes.bctSize);
            DEVMENUCOMMAND_LOG("%10s | %08x | %08x | %08x | %8d |\n", "package1", f(hashes.package1Main), f(hashes.package1Sub), f(hashes.package1File), hashes.package1Size);
            DEVMENUCOMMAND_LOG("%10s | %08x | %08x | %08x | %8d |\n", "package2", f(hashes.package2Main), f(hashes.package2Sub), f(hashes.package2File), hashes.package2Size);
            DEVMENUCOMMAND_LOG("--------------------------------------------------------\n");
        }
        else
        {
            DEVMENUCOMMAND_LOG("BootImagePackage is not installed\n");
        }

        isExist = true;
        NN_RESULT_TRY(GetBootImageHash(&hashes, updater::TargetBootMode::Safe, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer), GetBootImageUpdateType()))
            NN_RESULT_CATCH(updater::ResultBootImagePackageNotFound) {isExist = false;}
        NN_RESULT_END_TRY
        if (isExist)
        {
            DEVMENUCOMMAND_LOG("------------------------------------------------ Safe --\n");
            DEVMENUCOMMAND_LOG("%10s | %8s | %8s | %8s | %8s |\n", "type", "main", "sub", "file", "size");
            DEVMENUCOMMAND_LOG("%10s | %08x | %08x | %08x | %8d |\n", "bct", f(hashes.bctMain), f(hashes.bctSub), f(hashes.bctFile), hashes.bctSize);
            DEVMENUCOMMAND_LOG("%10s | %08x | %08x | %08x | %8d |\n", "package1", f(hashes.package1Main), f(hashes.package1Sub), f(hashes.package1File), hashes.package1Size);
            DEVMENUCOMMAND_LOG("%10s | %08x | %08x | %08x | %8d |\n", "package2", f(hashes.package2Main), f(hashes.package2Sub), f(hashes.package2File), hashes.package2Size);
            DEVMENUCOMMAND_LOG("--------------------------------------------------------\n");
        }
        else
        {
            DEVMENUCOMMAND_LOG("BootImagePackageSafe is not installed\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result BreakBootImagesCommand(bool* outValue, const Option& option)
    {
        auto phaseParam = option.GetTarget();
        if (!phaseParam)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto phaseUll = STR_TO_ULL(phaseParam, nullptr, 16);
        if (phaseUll < 1 || 6 < phaseUll)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto phase = static_cast<updater::UpdatePhase>(phaseUll);
        NN_RESULT_DO(updater::SetBootImagesIntermediatePhase(updater::TargetBootMode::Normal, phase, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
        DEVMENUCOMMAND_LOG("Set phase to %llu\n", phase);

        *outValue = true;
        NN_RESULT_SUCCESS;

    }
    Result BreakBootImagesSafeCommand(bool* outValue, const Option& option)
    {
        auto phaseParam = option.GetTarget();
        if (!phaseParam)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto phaseUll = STR_TO_ULL(phaseParam, nullptr, 16);
        if (phaseUll < 1 || 6 < phaseUll)
        {
            DEVMENUCOMMAND_LOG(HelpMessage);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto phase = static_cast<updater::UpdatePhase>(phaseUll);
        NN_RESULT_DO(updater::SetBootImagesIntermediatePhase(updater::TargetBootMode::Safe, phase, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
        DEVMENUCOMMAND_LOG("Set phase for safe to %llu\n", phase);

        *outValue = true;
        NN_RESULT_SUCCESS;

    }


    Result NotifyExFatRequiredCommand(bool* outValue, const Option&)
    {
        NN_RESULT_DO(ns::NotifyExFatDriverRequired());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result NotifyExFatDownloadedCommand(bool* outValue, const Option&)
    {
        NN_RESULT_DO(ns::NotifyExFatDriverRequired());
        NN_RESULT_DO(ns::NotifyExFatDriverDownloaded());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearExFatStatusCommand(bool* outValue, const Option&)
    {
        NN_RESULT_DO(ns::ClearExFatDriverStatus());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckContentDeliveryCommand(bool* outValue, const Option&)
    {
        os::SystemEvent event;
        ns::GetSystemUpdateNotificationEventForContentDelivery(&event);
        NN_LOG("%s\n", event.TryWait() ? "true" : "false");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result NotifyContentDeliveryCommand(bool* outValue, const Option&)
    {
        ns::NotifySystemUpdateForContentDelivery();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result NotifySystemDataUpdateCommand(bool* outValue, const Option&)
    {
        NN_RESULT_DO(fs::NotifySystemDataUpdateEvent());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const SubCommand g_SubCommands[] =
    {
        { "get-background-state", GetBackgroundStateCommand },
        { "check-latest", CheckLatestCommand },
        { "check-downloaded", CheckDownloadCommand },
        { "apply-downloaded", ApplyDownloadCommand },
        { "download-latest", DownloadLatestCommand },
        { "destroy-task", DestroyTaskCommand },
        { "get-download-progress", GetDownloadProgressCommand },
        { "get-device-id", GetDeviceIdCommand },
        { "get-debug-id", GetDebugIdCommand },
        { "set-debug-id", SetDebugIdCommand },
        { "set-debug-id-for-content-delivery", SetDebugIdForContentDeliveryCommand },
        { "invalidate-debug-id", InvalidateDebugIdCommand },
        { "request-bgnup", RequestBgnupCommand },
        { "check-cardupdate", CheckCardUpateCommand },
        { "do-cardupdate", DoCardUpdateCommand },
        { "get-cardupdate-euladata", GetCardUpdateEulaDataCommand },
        {"update-bootimages", UpdateBootImagesCommand },
        {"update-bootimages-safe", UpdateBootImagesSafeCommand},
        {"mark-verifying-required", MarkVerifyingRequiredCommand},
        {"mark-verifying-required-safe", MarkVerifyingRequiredSafeCommand},
        {"get-verifying-required-mark", GetVerifyingRequiredMarkCommand},
        {"get-bootimage-package-id", GetBootImagePackageIdCommand},
        {"get-bootimage-package-id-safe", GetBootImagePackageIdSafeCommand},
        {"get-bootimage-hashes", GetBootImageHashesCommand},
        {"break-bootimages", BreakBootImagesCommand},
        {"break-bootimages-safe", BreakBootImagesSafeCommand},
        { "notify-exfat-required", NotifyExFatRequiredCommand },
        { "notify-exfat-downloaded", NotifyExFatDownloadedCommand },
        { "clear-exfat-status", ClearExFatStatusCommand },
        { "check-content-delivery", CheckContentDeliveryCommand },
        { "notify-content-delivery", NotifyContentDeliveryCommand },
        { "notify-systemdata-update", NotifySystemDataUpdateCommand },
    };
}

Result SystemUpdateCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu systemupdate command. See '" DEVMENUCOMMAND_NAME " systemprogram --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
