﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <algorithm>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>

#include <nn/crypto.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_MaxCount.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/os/os_Types.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_SystemProgramCommand.h"
#include "DevMenuCommand_ProgramIdNameMap.h"
#include "DevMenuCommand_InstallUtil.h"

using namespace nn;

namespace
{
    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " systemprogram list\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram list-details <systemprogram_id>\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram list-details-all\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram install <absolute_nsp_path> [--force]\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram uninstall <sytemprogram_id> | <systemprogram_name>\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram uninstall all-unknown\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram set-initial-system-applet-program <program_id>\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram get-initial-system-applet-program\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram set-overlay-disp-program <program_id>\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram get-overlay-disp-program\n"
        "       " DEVMENUCOMMAND_NAME " systemprogram launch <sytemprogram_id> | <systemprogram_name>\n";

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    Result SystemProgramListCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));

        ProgramIdNameMap map;
        NN_RESULT_DO(map.Initialize());

        std::vector<ncm::ContentMetaKey> list;
        list.resize(ncm::SystemMaxContentMetaCount);
        auto listCount = db.ListContentMeta(list.data(), static_cast<int>(list.size()), ncm::ContentMetaType::Unknown);
        list.resize(static_cast<size_t>(listCount.listed));
        if (list.size() > 0)
        {
            DEVMENUCOMMAND_LOG("%d system content meta(s) found.\n", list.size());
            DEVMENUCOMMAND_LOG("id                        ver.  type            name\n");
            //      0x0123456789abcdef  4294967295  SystemProgram   DevMenu
            for (auto& meta : list)
            {
                DEVMENUCOMMAND_LOG("0x%016llx  %10u  %-14s  %s\n", meta.id, meta.version, devmenuUtil::GetContentMetaTypeString(meta.type), map.GetName(meta.id));
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    const char* HexToString(Bit8* data, size_t size)
    {
        // snprintf 用バッファ
        static char str[128];
        for(int i = 0; i < static_cast<int>(std::min(size, sizeof(str) / 2)); ++i)
        {
            util::SNPrintf(&(str[i * 2]), sizeof(str) - (i * 2), "%02x", data[i]);
        }
        return str;
    }
    class NcaFileHolder
    {
    public:
        NcaFileHolder() : m_list()
        {
        }
        Result Initialize()
        {
            NN_RESULT_DO(fs::MountBis("bissys", fs::BisPartitionId::System));

            fs::DirectoryHandle dirHandle;
            NN_RESULT_DO(fs::OpenDirectory(&dirHandle, "bissys:/Contents/registered/", fs::OpenDirectoryMode_All));
            NN_UTIL_SCOPE_EXIT { fs::CloseDirectory(dirHandle); };

            int64_t count;
            const int numEntries = 16;
            fs::DirectoryEntry entries[numEntries];

            do
            {
                NN_RESULT_DO(fs::ReadDirectory(&count, entries, dirHandle, numEntries));
                for(int i = 0; i < count; ++i)
                {
                    if (entries[i].directoryEntryType == fs::DirectoryEntryType_File)
                    {
                        m_list.push_back(std::make_pair<std::string, bool>(std::string(entries[i].name), false));
                    }
                }
            } while (count != 0);
            NN_RESULT_SUCCESS;
        }
        // 特定の名前のものがあるか見つけて、あったら見つけたマークを付ける
        bool LookupAndMark(const char* name)
        {
            for(auto& entry : m_list)
            {
                if(entry.first == name)
                {
                    entry.second = true;
                    return true;
                }
            }
            return false;
        }
        // 特定の名前のファイルの sha256 ハッシュを計算する
        const char* CalculateSha256(const char* name)
        {
            Bit8 hash[crypto::Sha256Generator::HashSize];
            auto result = CalculateSha256Internal(hash, sizeof(hash), name);
            if(result.IsSuccess())
            {
                return HexToString(hash, crypto::Sha256Generator::HashSize);
            }
            return nullptr;

        }
        Result CalculateSha256Internal(Bit8* hash, size_t hashSize, const char* name)
        {
            nn::crypto::Sha256Generator sha256;
            sha256.Initialize();

            char path[128];
            util::SNPrintf(path, sizeof(path), "bissys:/Contents/registered/%s", name);

            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            int64_t fileSize = 0;
            NN_RESULT_DO(fs::GetFileSize(&fileSize, file));

            const int bufferSize = 1 << 20;
            std::unique_ptr<Bit8[]>  buffer(new Bit8[bufferSize]);
            int64_t offset = 0;
            while(offset < fileSize)
            {
                size_t size = std::min(static_cast<int>(fileSize - offset), bufferSize);
                NN_RESULT_DO(ReadFile(file, offset, buffer.get(), size));
                sha256.Update(buffer.get(), size);
                offset += size;
            }
            sha256.GetHash(hash, hashSize);

            NN_RESULT_SUCCESS;
        }
        void PrintNotMarked()
        {
            for(auto& entry : m_list)
            {
                if(entry.second == false)
                {
                    DEVMENUCOMMAND_LOG(" - %s\n", entry.first.c_str());
                }
            }
        }
        void Finailze()
        {
            fs::Unmount("bissys");
        }
    private:
        std::vector< std::pair<std::string, bool> > m_list;
    };

    Result ListDetailsInternal(util::optional<ncm::SystemProgramId> id)
    {
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));

        ProgramIdNameMap map;
        NN_RESULT_DO(map.Initialize());

        NcaFileHolder holder;
        NN_RESULT_DO(holder.Initialize());
        NN_UTIL_SCOPE_EXIT { holder.Finailze(); };

        std::vector<ncm::ContentMetaKey> list;
        list.resize(ncm::SystemMaxContentMetaCount);
        auto listCount = db.ListContentMeta(list.data(), static_cast<int>(list.size()), ncm::ContentMetaType::Unknown);
        list.resize(static_cast<size_t>(listCount.listed));
        if (list.size() > 0)
        {
            for (auto& meta : list)
            {
                if (id && id->value != meta.id)
                {
                    continue;
                }
                std::vector<ncm::ContentInfo> contentInfos;
                contentInfos.resize(16);
                int count;
                NN_RESULT_DO(db.ListContentInfo(&count, contentInfos.data(), static_cast<int>(list.size()), meta, 0));
                contentInfos.resize(static_cast<size_t>(count));

                DEVMENUCOMMAND_LOG("0x%016llx, ver.%u, %s, %s\n", meta.id, meta.version, devmenuUtil::GetContentMetaTypeString(meta.type), map.GetName(meta.id));
                for(auto& content : contentInfos)
                {
                    char name[128];
                    util::SNPrintf(name, sizeof(name), "%s.nca", HexToString(content.id.data, sizeof(content.id.data)));
                    bool found = holder.LookupAndMark(name);

                    int64_t size =  (static_cast<int64_t>(content.sizeHigh) << 32) + content.sizeLow;
                    DEVMENUCOMMAND_LOG(" - %-16s: %s %s, %lld, [%.8s]\n", devmenuUtil::GetContentTypeString(content.type), (found ? "" : "NOT FOUND!!"), name, size, holder.CalculateSha256(name));
                }
                DEVMENUCOMMAND_LOG("\n");
            }
        }
        if(!id)
        {
            DEVMENUCOMMAND_LOG("Unknowns:\n");
            holder.PrintNotMarked();
        }
        NN_RESULT_SUCCESS;
    }
    Result SystemProgramListDetailsCommand(bool* outValue, const Option& option)
    {
        auto target = option.GetTarget();
        if (std::strlen(target) == 0)
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " systemprogram list-details <systemprogram_id>");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ncm::SystemProgramId id;
        id.value = STR_TO_ULL(target, nullptr, 16);
        *outValue = false;
        NN_RESULT_DO(ListDetailsInternal(id));
        *outValue = true;

        NN_RESULT_SUCCESS;
    }
    Result SystemProgramListDetailsAllCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        *outValue = false;
        NN_RESULT_DO(ListDetailsInternal(nullptr));
        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result SystemProgramInstallCommand(bool* outValue, const Option& option)
    {
        auto nspPath = option.GetTarget();
        if (std::strlen(nspPath) == 0)
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " systemprogram install <absolute_nsp_path>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (!devmenuUtil::IsAbsolutePath(nspPath))
        {
            DEVMENUCOMMAND_LOG("'%s' is not an absolute path.\n", nspPath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if ( !devmenuUtil::IsNspFile( nspPath ) )
        {
            DEVMENUCOMMAND_LOG( "Warning: The file extension is not .nsp\n" );
        }

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, nspPath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        static const size_t BufferSize = 16 * 1024 * 1024;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);
        ncm::SubmissionPackageInstallTask task;
        NN_RESULT_DO(task.Initialize(file, ncm::StorageId::BuildInSystem, buffer.get(), BufferSize));

        DEVMENUCOMMAND_LOG("Preparing to install %s...\n", nspPath);
        NN_RESULT_DO(task.Prepare());
        bool needsCleanup = true;
        NN_UTIL_SCOPE_EXIT
        {
            if (needsCleanup)
            {
                task.Cleanup();
            }
        };
        int count;
        const int MaxContentMetaCount = 2048;
        std::unique_ptr<ncm::StorageContentMetaKey[]> keys(new ncm::StorageContentMetaKey[MaxContentMetaCount]);
        NN_RESULT_DO(task.ListContentMetaKey(&count, keys.get(), MaxContentMetaCount, 0));

        NN_RESULT_DO(task.Cleanup());
        needsCleanup = false;

        ncm::ContentMetaDatabase db;
        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuiltInSystem));
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::BuiltInSystem));
        for (int i = 0; i < count; ++i)
        {
            bool hasHigherVersion;
            NN_RESULT_DO(devmenuUtil::CheckHigherVersionAndUninstallIfNeeded(&hasHigherVersion, &db, &storage, keys[i].key, option.HasKey("--force")));
            if (hasHigherVersion)
            {
                DEVMENUCOMMAND_LOG("Higher version exists, install failed.\n");
                DEVMENUCOMMAND_LOG("If you want to install, please use --force option.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_DO(task.Prepare());
        needsCleanup = true;

        NN_RESULT_DO(devmenuUtil::WaitInstallTaskExecute(&task));
        NN_RESULT_DO(task.Commit());
        needsCleanup = false;

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SystemProgramUninstallCommand(bool* outValue, const Option& option)
    {
        auto target = option.GetTarget();
        if (std::strlen(target) == 0)
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " systemprogram uninstall <sytemprogram_id> | <systemprogram_name>\n"
                   "       " DEVMENUCOMMAND_NAME " systemprogram uninstall all-unknown\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ncm::ContentMetaDatabase db;
        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::BuildInSystem));
        ncm::ContentManagerAccessor accessor(&db, &storage);

        if (std::string(target) == "all-unknown")
        {
            ProgramIdNameMap map;
            NN_RESULT_DO(map.Initialize());

            std::vector<ncm::ContentMetaKey> list;
            list.resize(ncm::SystemMaxContentMetaCount);
            auto listCount = db.ListContentMeta(list.data(), static_cast<int>(list.size()), ncm::ContentMetaType::Unknown);
            list.resize(static_cast<size_t>(listCount.listed));
            for (auto& key : list)
            {
                if (!map.HasName(key.id))
                {
                    NN_RESULT_DO(accessor.DeleteAll(key.id));
                }
            }

            NN_RESULT_DO(db.Commit());
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        char* endptr;
        Bit64 id = STR_TO_ULL(target, &endptr, 16);
        if (*endptr != '\0')
        {
            ProgramIdNameMap map;
            NN_RESULT_DO(map.Initialize());

            auto idByName = map.GetId(target);
            if (idByName)
            {
                id = *idByName;
            }
            else
            {
                DEVMENUCOMMAND_LOG("Unknown system program name %s.\n", target);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_DO(accessor.DeleteAll(id));
        NN_RESULT_DO(db.Commit());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SystemProgramSetInitialSystemAppletProgramCommand(bool* outValue, const Option& option)
    {
        auto idString = option.GetTarget();
        if (std::strlen(idString) == 0)
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " systemprogram set-initial-system-applet-program <program_id>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char* endPtr;
        auto id = std::strtoull(idString, &endPtr, 16);
        if (*endPtr != '\0')
        {
            DEVMENUCOMMAND_LOG("Invalid program id format\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char setId[32];
        util::SNPrintf(setId, sizeof(setId), "0x%016llx", id);
        settings::fwdbg::SetSettingsItemValue("ns.applet", "system_applet_id", setId, std::strlen(setId) + 1);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SystemProgramGetInitialSystemAppletProgramCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        char idStr[32];
        settings::fwdbg::GetSettingsItemValue(idStr, sizeof(idStr), "ns.applet", "system_applet_id");

        DEVMENUCOMMAND_LOG("%s\n",idStr);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SystemProgramSetOverlayDispProgramCommand(bool* outValue, const Option& option)
    {
        auto idString = option.GetTarget();
        if (std::strlen(idString) == 0)
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " systemprogram set-overlay-disp-program <program_id>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char* endPtr;
        auto id = std::strtoull(idString, &endPtr, 16);
        if (*endPtr != '\0')
        {
            DEVMENUCOMMAND_LOG("Invalid program id format\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char setId[32];
        util::SNPrintf(setId, sizeof(setId), "0x%016llx", id);
        settings::fwdbg::SetSettingsItemValue("ns.applet", "overlay_applet_id", setId, std::strlen(setId) + 1);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SystemProgramGetOverlayDispProgramCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        char idStr[32];
        settings::fwdbg::GetSettingsItemValue(idStr, sizeof(idStr), "ns.applet", "overlay_applet_id");

        DEVMENUCOMMAND_LOG("%s\n", idStr);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SystemProgramLaunchCommand(bool* outValue, const Option& option)
    {
        auto target = option.GetTarget();
        if (std::strlen(target) == 0)
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " systemprogram launch <sytemprogram_id> | <systemprogram_name>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char* endptr;
        Bit64 id = STR_TO_ULL(target, &endptr, 16);
        if (*endptr != '\0')
        {
            ProgramIdNameMap map;
            NN_RESULT_DO(map.Initialize());

            auto idByName = map.GetId(target);
            if (idByName)
            {
                id = *idByName;
            }
            else
            {
                DEVMENUCOMMAND_LOG("Unknown system program name %s.\n", target);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        os::ProcessId processId;
        ncm::SystemProgramId sysId = { id };
        NN_RESULT_DO(ns::LaunchLibraryApplet(&processId, sysId));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const SubCommand g_SubCommands[] =
    {
        { "list", SystemProgramListCommand },
        { "list-details", SystemProgramListDetailsCommand },
        { "list-details-all", SystemProgramListDetailsAllCommand },
        { "install", SystemProgramInstallCommand },
        { "uninstall", SystemProgramUninstallCommand },
        { "set-initial-system-applet-program", SystemProgramSetInitialSystemAppletProgramCommand },
        { "get-initial-system-applet-program", SystemProgramGetInitialSystemAppletProgramCommand },
        { "set-overlay-disp-program", SystemProgramSetOverlayDispProgramCommand },
        { "get-overlay-disp-program", SystemProgramGetOverlayDispProgramCommand },
        { "launch", SystemProgramLaunchCommand }
    };
}

Result SystemProgramCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // TORIAEZU : ns::InitializeForSystemUpdate() を呼び出しただけでは接続に成功して待たないため、
    // 動作上無意味な API を読んで初期化が終わるまで待つようにする
    ns::GetBackgroundNetworkUpdateState();

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu sytemprogram command. See '" DEVMENUCOMMAND_NAME " systemprogram --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
