﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/lm/lm_LogDestinationType.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_LogCommand.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Option.h"

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
namespace nn { namespace lm {
        void SetDestination(Bit32 destination);
}}
#endif

namespace {
    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " log destination-target-manager\n"
        "       " DEVMENUCOMMAND_NAME " log destination-uart\n"
        "       " DEVMENUCOMMAND_NAME " log destination-tm-and-uart-if-sleep\n"
        "       " DEVMENUCOMMAND_NAME " log destination-all\n"
        ""; // 終端

    void SetLogDestination(int destination)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::lm::SetDestination(destination);
#else
        NN_UNUSED(destination);
#endif
    }

    nn::Result DestinationTMCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        SetLogDestination(nn::lm::LogDestination_TargetManager);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result DestinationUartCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        SetLogDestination(nn::lm::LogDestination_Uart);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result DestinationUartIfSleepCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        SetLogDestination(nn::lm::LogDestination_TargetManager | nn::lm::LogDestination_UartIfSleep);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result DestinationAllCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        SetLogDestination(nn::lm::LogDestination_All);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    struct SubCommand
    {
        std::string name;
        nn::Result(*function)(bool* outValue, const Option&);
    };

    const SubCommand g_SubCommands[] =
    {
        { "destination-target-manager", DestinationTMCommand },
        { "destination-uart", DestinationUartCommand },
        { "destination-tm-and-uart-if-sleep", DestinationUartIfSleepCommand },
        { "destination-all", DestinationAllCommand },
    };
}

nn::Result LogCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu log command. See '" DEVMENUCOMMAND_NAME " log --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
