﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <string>

#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/erpt/erpt_Report.h>
#include <nn/err/err_ReportErrorApi.h>
#include <nn/result/result_HandlingUtility.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_ErrorCommand.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_StrToUll.h"
#include "errors/DevMenuCommand_ErrorExport.h"

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
#include <algorithm>
#include <sstream>
#include <vector>

#include <nn/eupld/eupld_Control.h>
#include <nn/eupld/eupld_Request.h>
#include <nn/nifm.h>
#include <nn/nn_TimeSpan.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/system/settings_ErrorReport.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "errors/DevMenuCommand_ErrorDump.h"
#endif

#if defined(NN_DEVMENUSYSTEM) && defined(_USE_MATH_DEFINES)
// DeMenu が _USE_MATH_DEFINES を定義するため、DevMenuCommand in DevMenu ビルド時に重複定義になってしまうことの W/A
#undef _USE_MATH_DEFINES
#endif

#include <nn/msgpack.h>
#ifdef NN_BUILD_CONFIG_OS_WIN
    #include <nn/nn_Windows.h> // msgpack.h が Windows.h を include していることに対するワークアラウンド
#endif

using namespace nn;

namespace
{
    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    Result ExportCommand(bool* outValue, const Option& option)
    {
        auto pathString = option.GetTarget();

        if( !devmenuUtil::IsAbsolutePath(pathString) )
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " error export <absolute_host_file_path>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_FUNCTION_LOCAL_STATIC(nn::erpt::ReportList, reportList);
        nn::erpt::Manager manager;

        NN_ABORT_UNLESS_RESULT_SUCCESS(manager.Initialize());
        NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };

        NN_ABORT_UNLESS_RESULT_SUCCESS(manager.GetReportList(reportList));

        // レポートなし
        if( reportList.reportCount == 0 )
        {
            DEVMENUCOMMAND_LOG("No error report. Exit.\n");
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        errors::ErrorReportBinaryHeaderFlag flags = errors::ErrorReportBinaryHeaderFlag_All;
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        if( option.HasKey("--flags") )
        {
            auto flagsString = option.GetValue("--flags");
            flags = static_cast<errors::ErrorReportBinaryHeaderFlag>(std::strtol(flagsString, NULL, 16));
        }
#endif
        NN_RESULT_DO(errors::ExportReport(pathString, reportList, flags));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
    bool IsUploadEnabled()
    {
        bool b;
        return settings::fwdbg::GetSettingsItemValue(&b, sizeof(b), "eupld", "upload_enabled") == sizeof(b) ? b : false;
    }

    bool IsAutoUploadWithoutOptedInAccountEnabled()
    {
        bool b;
        return settings::fwdbg::GetSettingsItemValue(&b, sizeof(b), "eclct", "analytics_override") == sizeof(b) ? b : false;
    }

    bool IsProductionMode()
    {
        bool b;
        return settings::fwdbg::GetSettingsItemValue(&b, sizeof(b), "erpt", "production_mode") == sizeof(b) ? b : false;
    }

    int32_t GetOptedInAccountPollPeriod()
    {
        int32_t i;
        return settings::fwdbg::GetSettingsItemValue(&i, sizeof(i), "eclct", "analytics_pollperiod") == sizeof(i) ? i : 0;
    }

    void ShowPleaseRebootLog()
    {
        DEVMENUCOMMAND_LOG("*** Please reboot the target to allow the change to take effect.\n");
    }

    void Trim(std::string &s)
    {
        s.erase(s.begin(), std::find_if(s.begin(), s.end(), std::not1(std::ptr_fun<int, int>(std::isspace))));
        s.erase(std::find_if(s.rbegin(), s.rend(), std::not1(std::ptr_fun<int, int>(std::isspace))).base(), s.end());
    }

    std::vector<std::string> Split(const std::string& input, char delimiter)
    {
        std::istringstream stream(input);

        std::string field;
        std::vector<std::string> result;
        while( std::getline(stream, field, delimiter) )
        {
            Trim(field);
            result.push_back(field);
        }
        return result;
    }

    Result DumpAllCommand(bool* outValue, const Option& option)
    {
        auto dumpReportCountMax = erpt::NumberOfReports;
        if( option.HasKey("--max-count") )
        {
            auto latestOptionString = option.GetValue("--max-count");
            char* endptr;
            dumpReportCountMax = static_cast<decltype(dumpReportCountMax)>(STR_TO_ULL(latestOptionString, &endptr, 10));
            if( *endptr != '\0' || dumpReportCountMax == 0 )
            {
                DEVMENUCOMMAND_LOG("Invalid --max-count value(%s). It must be a number greater than 0.\n", latestOptionString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        util::optional<std::vector<std::string>> categories(nullptr);
        bool useCategoriesToExclude = false;
        if( option.HasKey("--category-filter") )
        {
            auto categoryFilterStr = option.GetValue("--category-filter");
            // '-' 始まりは Option のパーサーによって Value とみなされないため '^' で代用
            if( categoryFilterStr[0] == '^' )
            {
                useCategoriesToExclude = true;
                categoryFilterStr = categoryFilterStr + 1;
            }
            categories = Split(std::string(categoryFilterStr), ',');
        }

        util::optional<std::vector<std::string>> fields(nullptr);
        bool useFieldsToExclude = false;
        if( option.HasKey("--field-filter") )
        {
            auto fieldFilterStr = option.GetValue("--field-filter");
            // '-' 始まりは Option のパーサーによって Value とみなされないため '^' で代用
            if( fieldFilterStr[0] == '^' )
            {
                useFieldsToExclude = true;
                fieldFilterStr = fieldFilterStr + 1;
            }
            fields = Split(std::string(fieldFilterStr), ',');
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        NN_FUNCTION_LOCAL_STATIC(nn::erpt::ReportList, reportList);
        nn::erpt::Manager manager;

        NN_ABORT_UNLESS_RESULT_SUCCESS(manager.Initialize());
        NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };

        NN_ABORT_UNLESS_RESULT_SUCCESS(manager.GetReportList(reportList));
        auto reportCount = reportList.reportCount;
        if( reportCount == 0u )
        {
            DEVMENUCOMMAND_LOG("No report to dump.\n");
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
        DEVMENUCOMMAND_LOG("ReportCount: %d\n", reportCount);
        auto dumpReportCount = std::min(reportCount, dumpReportCountMax);
        if( reportCount != dumpReportCount )
        {
            if( dumpReportCount == 1 )
            {
                DEVMENUCOMMAND_LOG("Dump the latest report.\n");
            }
            else
            {
                DEVMENUCOMMAND_LOG("Dump the latest %d reports.\n", dumpReportCount);
            }
        }

        auto toolVersion = errors::GetToolVersion();
        char toolVersionStr[32];
        errors::GetToolVersionString(toolVersionStr, sizeof(toolVersionStr), toolVersion);
        char toolVersionLabel[128];
        auto toolVersionLabelLength = util::SNPrintf(toolVersionLabel, sizeof(toolVersionLabel), "Dumped by DevMenuCommandSystem ver. %s\n", toolVersionStr);

        const char* outFile = option.HasKey("--out") ? option.GetValue("--out") : nullptr;
        util::optional<fs::FileHandle> fileHandle;
        if( outFile != nullptr )
        {
            if( !devmenuUtil::IsAbsolutePath(outFile) )
            {
                DEVMENUCOMMAND_LOG("Invalid --out(%s). It must be an absolute path.\n", outFile);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_TRY(fs::DeleteFile(outFile))
                NN_RESULT_CATCH(fs::ResultPathNotFound) {}
            NN_RESULT_END_TRY
            NN_RESULT_DO(fs::CreateFile(outFile, 0));
            fileHandle.emplace();
            NN_RESULT_DO(fs::OpenFile(&(*fileHandle), outFile, fs::OpenMode_Write | fs::OpenMode_AllowAppend));

            fs::WriteFile(*fileHandle, 0, toolVersionLabel, toolVersionLabelLength, fs::WriteOption::MakeValue(0));
            errors::BeginDumpOutputToFile(*fileHandle, toolVersionLabelLength);
        }
        else
        {
            DEVMENUCOMMAND_LOG(toolVersionLabel);
        }
        NN_UTIL_SCOPE_EXIT
        {
            if( fileHandle )
            {
                errors::EndDumpOutputToFile();
                nn::fs::FlushFile(*fileHandle);
                nn::fs::CloseFile(*fileHandle);
            }
        };

        for (uint32_t i = 0; i < dumpReportCount; ++i)
        {
            auto reportInfo = reportList.Report[i];
            NN_ABORT_UNLESS_RESULT_SUCCESS(errors::DumpReport(reportInfo.reportId, reportInfo.reportMetaData, categories, useCategoriesToExclude, fields, useFieldsToExclude));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    Result EnableUploadCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        bool enabled = true;
        settings::fwdbg::SetSettingsItemValue("eupld", "upload_enabled", &enabled, sizeof(enabled));
        *outValue = true;
        ShowPleaseRebootLog();
        NN_RESULT_SUCCESS;
    }

    Result DisableUploadCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        bool enabled = false;
        settings::fwdbg::SetSettingsItemValue("eupld", "upload_enabled", &enabled, sizeof(enabled));
        *outValue = true;
        ShowPleaseRebootLog();
        NN_RESULT_SUCCESS;
    }

    Result EnableAutoUploadWithoutOptedInAccountCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        bool enabled = true;
        settings::fwdbg::SetSettingsItemValue("eclct", "analytics_override", &enabled, sizeof(enabled));
        *outValue = true;
        ShowPleaseRebootLog();
        NN_RESULT_SUCCESS;
    }

    Result DisableAutoUploadWithoutOptedInAccountCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        bool enabled = false;
        settings::fwdbg::SetSettingsItemValue("eclct", "analytics_override", &enabled, sizeof(enabled));
        *outValue = true;
        ShowPleaseRebootLog();
        NN_RESULT_SUCCESS;
    }

    Result SetOptedInFlagPollPeriodCommand(bool* outValue, const Option& option)
    {
        auto target = option.GetTarget();
        if (std::strlen(target) == 0)
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " error set-opted-in-flag-poll-period <seconds>\n" );
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char* endptr;
        auto seconds = STR_TO_ULL(target, &endptr, 10);
        if ( *endptr != '\0' || seconds == 0 || !util::IsIntValueRepresentable<int>(seconds) )
        {
            DEVMENUCOMMAND_LOG("<seconds>(%s) must be a number larger than 0 and representable as int.\n", target);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto secondsInInteger = static_cast<int>(seconds);
        settings::fwdbg::SetSettingsItemValue("eclct", "analytics_pollperiod", &secondsInInteger, sizeof(secondsInInteger));
        *outValue = true;
        ShowPleaseRebootLog();
        NN_RESULT_SUCCESS;
    }

    Result GetOptedInFlagPollPeriodCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        uint32_t value;
        if( settings::fwdbg::GetSettingsItemValue(&value, sizeof(value), "eclct", "analytics_pollperiod") == sizeof(value) )
        {
            DEVMENUCOMMAND_LOG("%u\n", value);
        }
        else
        {
            DEVMENUCOMMAND_LOG("Failed to get opted-in flag poll period.\n");
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const char* GetEupldResultString(const nn::Result& result)
    {
        // 代表的なものだけ対応。
        NN_RESULT_TRY(result)
        NN_RESULT_CATCH(eupld::ResultNoReportsToSend)
        {
            return "NoReportsToSend";
        }
        NN_RESULT_CATCH(eupld::ResultUrlNotSet)
        {
            return "ResultUrlNotSet";
        }
        NN_RESULT_CATCH(eupld::ResultNetworkNotAvailable)
        {
            return "ResultNetworkNotAvailable";
        }
        NN_RESULT_CATCH(eupld::ResultAlreadyScheduled)
        {
            return "ResultAlreadyScheduled";
        }
        NN_RESULT_CATCH_ALL
        {
            return "(unknown)";
        }
        NN_RESULT_END_TRY
        NN_ABORT("GetEupldResultString: Never reach here\n");
    }

    Result UploadAllCommand(bool* outValue, const Option& option)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::Initialize());
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        eupld::Request uploadRequest;
        NN_ABORT_UNLESS_RESULT_SUCCESS(uploadRequest.Initialize());
        NN_UTIL_SCOPE_EXIT{ uploadRequest.Finalize(); };

        {
            nn::Result result;
            if( option.HasKey("--force") )
            {
                // 送信済みも含めて全て送信。
                nn::erpt::Manager manager;
                NN_ABORT_UNLESS_RESULT_SUCCESS(manager.Initialize());
                NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };
                nn::erpt::ReportList reportList;
                NN_ABORT_UNLESS_RESULT_SUCCESS(manager.GetReportList(reportList));
                if( reportList.reportCount == 0 )
                {
                    DEVMENUCOMMAND_LOG("No error report to upload.\n");
                    *outValue = true;
                    NN_RESULT_SUCCESS;
                }
                std::unique_ptr<erpt::ReportId[]> reportIdList(new erpt::ReportId[reportList.reportCount]);
                NN_RESULT_THROW_UNLESS(reportIdList != nullptr, nn::os::ResultOutOfMemory());
                for( uint32_t i = 0; i < reportList.reportCount; i++ )
                {
                    reportIdList[i] = reportList.Report[i].reportId;
                }
                result = uploadRequest.UploadSelected(reportIdList.get(), reportList.reportCount);
            }
            else
            {
                // 未送信のものを全て送信。
                result = uploadRequest.UploadAll();
            }
            if( result.IsFailure() )
            {
                DEVMENUCOMMAND_LOG("Failed to upload error report : 0x%08x (%s).\n", result.GetInnerValueForDebug(), GetEupldResultString(result));
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
            if( !uploadRequest.GetEventPointer()->TimedWait(nn::TimeSpan::FromSeconds(30)) )
            {
                DEVMENUCOMMAND_LOG("Failed to upload error report in 30 seconds. Time out.\n");
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }

        {
            auto result = uploadRequest.GetResult();
            if( result.IsFailure() )
            {
                DEVMENUCOMMAND_LOG("Failed to upload error report : 0x%08x (%s).\n", result.GetInnerValueForDebug(), GetEupldResultString(result));
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
            eupld::ReportUploadList reportUploadList;
            NN_ABORT_UNLESS_RESULT_SUCCESS(uploadRequest.GetUploadStatus(reportUploadList));
            DEVMENUCOMMAND_LOG("ReportCount: %d\n", reportUploadList.reportCount);
            bool hasResultUrlNotSet = false;
            for( uint32_t i = 0; i < reportUploadList.reportCount; i++ )
            {
                const auto& report = reportUploadList.Report[i];
                char idStr[64];
                report.reportId.u.uuidRFC4122.ToString(idStr, sizeof(idStr));
                if( report.result.IsSuccess() )
                {
                    DEVMENUCOMMAND_LOG("[%2d] %s : Upload succeeded.\n", i, idStr);
                }
                else
                {
                    DEVMENUCOMMAND_LOG("[%2d] %s : Upload failed : 0x%08x (%s).\n", i, idStr, report.result.GetInnerValueForDebug(), GetEupldResultString(report.result));
                    if( eupld::ResultUrlNotSet::Includes(report.result) )
                    {
                        hasResultUrlNotSet = true;
                    }
                }
            }
            if( hasResultUrlNotSet )
            {
                // enable-upload していない状態（デフォルト）では、eupld プロセスでアップロード先の URL が設定されないため失敗する。
                // Result から何をすればいいかがわかりにくいのでメッセージを出しておく。
                DEVMENUCOMMAND_LOG("Use `error enable-upload` command and reboot to resolve ResultUrlNotSet failure\n");
            }
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CreateSampleErrorReportCommand(bool* outValue, const Option& option)
    {
        uint64_t count = 1;
        if( option.HasKey("--count") )
        {
            auto countString = option.GetValue("--count");
            char* endptr;
            count = STR_TO_ULL(countString, &endptr, 10);
            if( *endptr != '\0' || count == 0 || count > 50 )
            {
                DEVMENUCOMMAND_LOG("Invalid --count value(%s). It must be (count >= 1 && count <= %d)\n", countString, erpt::NumberOfReports);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        auto type = erpt::ReportType::ReportType_Invisible;
        if( option.HasKey("--type") )
        {
            auto typeString = option.GetValue("--type");
            if( util::Strncmp(typeString, "visible", static_cast<int>(strlen(typeString))) == 0 )
            {
                type = erpt::ReportType::ReportType_Visible;
            }
            else if( util::Strncmp(typeString, "invisible", static_cast<int>(strlen(typeString))) == 0 )
            {
                type = erpt::ReportType::ReportType_Invisible;
            }
            else
            {
                DEVMENUCOMMAND_LOG("Invalid --type value(%s). It must be 'visible' or 'invisible'.\n", typeString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        if( type == erpt::ReportType::ReportType_Invisible )
        {
            for( uint64_t i = 0; i < count; i++ )
            {
                err::ErrorCode errorCode{ 2999,9999 };
                err::ReportError(devmenuUtil::GetSelfApplicationId(), errorCode);
            }
        }
        else
        {
            for( uint64_t i = 0; i < count; i++ )
            {
                auto context = erpt::Context(erpt::CategoryId::ErrorInfo);
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorCode, "2999-9999", static_cast<uint32_t>(strlen("2999-9999"))));
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorDescription, "SampleReport", static_cast<uint32_t>(strlen("SampleReport"))));
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(type));
            }
        }
        DEVMENUCOMMAND_LOG("Created %d report%s.\n", count, count > 1 ? "s" : "");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ShowCommand(bool* outValue, const Option& option)
    {
        auto pathString = option.GetTarget();

        if (std::string(pathString) == "")
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " error show <absolute_host_file_path>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        NN_RESULT_DO(errors::DecryptAndDumpReport(outValue, pathString));
        NN_RESULT_SUCCESS;
    }

    Result DecryptCommand(bool* outValue, const Option& option)
    {
        if (!option.HasKey("--in") || !option.HasKey("--out"))
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " error decrypt --in <absolute_host_input_file_path> --out <absolute_host_output_file_path>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto inputPathString = option.GetValue("--in");
        auto outputPathString = option.GetValue("--out");

        NN_RESULT_DO(errors::DecryptReport(outValue, inputPathString, outputPathString));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetErrorReportSharePermission(bool* outValue, const Option& option)
    {
        auto target = option.GetTarget();

        if( std::strncmp(target, "NotConfirmed", sizeof("NotConfirmed")) == 0 )
        {
            settings::system::SetErrorReportSharePermission(settings::system::ErrorReportSharePermission_NotConfirmed);
        }
        else if( std::strncmp(target, "Granted", sizeof("Granted")) == 0 )
        {
            settings::system::SetErrorReportSharePermission(settings::system::ErrorReportSharePermission_Granted);
        }
        else if( std::strncmp(target, "Denied", sizeof("Denied")) == 0 )
        {
            settings::system::SetErrorReportSharePermission(settings::system::ErrorReportSharePermission_Denied);
        }
        else
        {
            DEVMENUCOMMAND_LOG("%s is invalid. Valid options are <NotConfirmed|Granted|Denied>\n", target);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const char* GetErrorReportSharePermissionString(settings::system::ErrorReportSharePermission permission)
    {
        switch( permission )
        {
        case settings::system::ErrorReportSharePermission_NotConfirmed: return "NotConfirmed";
        case settings::system::ErrorReportSharePermission_Granted:      return "Granted";
        case settings::system::ErrorReportSharePermission_Denied:       return "Denied";
        default:                                                        return "(unknown)";
        }
    }

    Result GetErrorReportSharePermission(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("%s\n", GetErrorReportSharePermissionString(settings::system::GetErrorReportSharePermission()));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableProductionModeCommand(bool* outValue, const Option&)
    {
        bool enabled = true;
        settings::fwdbg::SetSettingsItemValue("erpt", "production_mode", &enabled, sizeof(enabled));
        ShowPleaseRebootLog();
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableProductionModeCommand(bool* outValue, const Option&)
    {
        bool enabled = false;
        settings::fwdbg::SetSettingsItemValue("erpt", "production_mode", &enabled, sizeof(enabled));
        ShowPleaseRebootLog();
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CleanupReportsCommand(bool* outValue, const Option&)
    {
        nn::erpt::Manager manager;
        NN_ABORT_UNLESS_RESULT_SUCCESS(manager.Initialize());
        NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };
        NN_ABORT_UNLESS_RESULT_SUCCESS(manager.CleanupReports());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteReportCommand(bool* outValue, const Option& option)
    {
        if( !option.HasTarget() )
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " error delete <report-id>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto target = option.GetTarget();
        nn::erpt::Manager manager;
        NN_ABORT_UNLESS_RESULT_SUCCESS(manager.Initialize());
        NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };
        erpt::ReportId reportId;
        reportId.u.uuidRFC4122.FromString(target);
        NN_RESULT_TRY(manager.DeleteReport(reportId))
            NN_RESULT_CATCH(erpt::ResultInvalidArgument)
            {
                DEVMENUCOMMAND_LOG("Report '%s' is not found.\n", target);
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result InfoCommand(bool* outValue, const Option&)
    {
        // Upload 関係
        eupld::Control c;
        bool isAutoUploadEnabled;
        uint32_t autoUploadTime;
        NN_ABORT_UNLESS_RESULT_SUCCESS(c.GetAutoUpload(&isAutoUploadEnabled, &autoUploadTime));
        DEVMENUCOMMAND_LOG("[Upload]\n");
        DEVMENUCOMMAND_LOG("    Upload                               : %s\n", IsUploadEnabled() ? "Enabled" : "Disabled");
        DEVMENUCOMMAND_LOG("    Auto upload                          : %s (expires in %lld seconds)\n",
            isAutoUploadEnabled ? "Enabled" : "Disabled",
            (nn::TimeSpan::FromSeconds(static_cast<int64_t>(autoUploadTime)) - os::GetSystemTick().ToTimeSpan()).GetSeconds());
        DEVMENUCOMMAND_LOG("    Auto upload wihtout opted-in account : %s\n", IsAutoUploadWithoutOptedInAccountEnabled() ? "Enabled" : "Disabled");
        DEVMENUCOMMAND_LOG("    Opted-in account poll period         : %d seconds\n", GetOptedInAccountPollPeriod());

        // Report 関係
        erpt::Manager manager;
        NN_ABORT_UNLESS_RESULT_SUCCESS(manager.Initialize());
        erpt::StorageUsageStatistics usage;
        manager.GetStorageUsageStatistics(&usage);
        DEVMENUCOMMAND_LOG("[Report]\n");
        char uuidStr[32];
        DEVMENUCOMMAND_LOG("    Storage UUID                         : %s\n", usage.journalUuid.ToString(uuidStr, sizeof(uuidStr)));
        DEVMENUCOMMAND_LOG("    Used storage size                    : %u Bytes\n", usage.usedStorageSize);
        DEVMENUCOMMAND_LOG("    Max report size                      : %u Bytes\n", usage.maxReportSize);
        DEVMENUCOMMAND_LOG("    Stored report count                  : %u (Visible = %u, Invisible = %u)\n",
            usage.reportCount[erpt::ReportType_Visible] + usage.reportCount[erpt::ReportType_Invisible],
            usage.reportCount[erpt::ReportType_Visible], usage.reportCount[erpt::ReportType_Invisible]);
        DEVMENUCOMMAND_LOG("    Transmitted report count             : %u (Visible = %u, Invisible = %u)\n",
            usage.transmittedCount[erpt::ReportType_Visible] + usage.transmittedCount[erpt::ReportType_Invisible],
            usage.transmittedCount[erpt::ReportType_Visible], usage.transmittedCount[erpt::ReportType_Invisible]);
        DEVMENUCOMMAND_LOG("    Untransmitted report count           : %u (Visible = %u, Invisible = %u)\n",
            usage.untransmittedCount[erpt::ReportType_Visible] + usage.untransmittedCount[erpt::ReportType_Invisible],
            usage.untransmittedCount[erpt::ReportType_Visible], usage.untransmittedCount[erpt::ReportType_Invisible]);
        DEVMENUCOMMAND_LOG("    ProdMode Encryption                  : %s\n", IsProductionMode() ? "Enabled" : "Disabled");
        DEVMENUCOMMAND_LOG("    Error report share permission        : %s\n", GetErrorReportSharePermissionString(settings::system::GetErrorReportSharePermission()));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#endif

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " error export <absolute_host_file_path>\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " error dump-all [--max-count <count>] [--category-filter <comma separated erpt::CategoryId string>] [--field-filter <comma separated erpt::FieldId string>] [--out <absolute_host_file_path>]\n"
        "       " DEVMENUCOMMAND_NAME " error enable-upload\n"
        "       " DEVMENUCOMMAND_NAME " error disable-upload\n"
        "       " DEVMENUCOMMAND_NAME " error enable-auto-upload-without-opted-in-account\n"
        "       " DEVMENUCOMMAND_NAME " error disable-auto-upload-without-opted-in-account\n"
        "       " DEVMENUCOMMAND_NAME " error set-opted-in-flag-poll-period <seconds>\n"
        "       " DEVMENUCOMMAND_NAME " error get-opted-in-flag-poll-period\n"
        "       " DEVMENUCOMMAND_NAME " error upload-all [--force]\n"
        "       " DEVMENUCOMMAND_NAME " error create-sample-report [--type <invisible|visible>] [--count [1..50]]\n"
        "       " DEVMENUCOMMAND_NAME " error show <absolute_host_file_path>\n"
        "       " DEVMENUCOMMAND_NAME " error set-report-share-permission <NotConfirmed|Granted|Denied>\n"
        "       " DEVMENUCOMMAND_NAME " error get-report-share-permission\n"
        "       " DEVMENUCOMMAND_NAME " error enable-production-mode\n"
        "       " DEVMENUCOMMAND_NAME " error disable-production-mode\n"
        "       " DEVMENUCOMMAND_NAME " error clean\n"
        "       " DEVMENUCOMMAND_NAME " error delete <report-id>\n"
        "       " DEVMENUCOMMAND_NAME " error info\n"
#endif
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        { "export", ExportCommand },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "dump-all",    DumpAllCommand },
        { "enable-upload", EnableUploadCommand },
        { "disable-upload", DisableUploadCommand },
        { "enable-auto-upload-without-opted-in-account", EnableAutoUploadWithoutOptedInAccountCommand },
        { "disable-auto-upload-without-opted-in-account", DisableAutoUploadWithoutOptedInAccountCommand },
        { "set-opted-in-flag-poll-period", SetOptedInFlagPollPeriodCommand },
        { "get-opted-in-flag-poll-period", GetOptedInFlagPollPeriodCommand },
        { "upload-all", UploadAllCommand },
        { "create-sample-report", CreateSampleErrorReportCommand },
        { "show", ShowCommand },
        { "decrypt", DecryptCommand },
        { "set-report-share-permission", SetErrorReportSharePermission },
        { "get-report-share-permission", GetErrorReportSharePermission },
        { "enable-production-mode", EnableProductionModeCommand },
        { "disable-production-mode", DisableProductionModeCommand },
        { "clean", CleanupReportsCommand },
        { "delete", DeleteReportCommand },
        { "info", InfoCommand },
#endif
    };
}

Result ErrorCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu error command. See '" DEVMENUCOMMAND_NAME " error --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
