﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>
#include <sstream>
#include <istream>
#include <nn/nn_Log.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/nim/nim_DynamicRightsApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nifm.h>

#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_ElicenseUtil.h"
#include "DevMenuCommand_DynamicRightsCommand.h"

using namespace nn;
using namespace nne;
using namespace devmenuUtil;

namespace
{
    const char g_HelpMessage[] =
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "usage: " DEVMENUCOMMAND_NAME " dynamicrights query [--account <user_account_index>|--without-account true] [--type <TYPE>] [--status <STATUS] <rights_id...>\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights assign [--account <user_account_index>|--without-account true] [--type <TYPE>] <rights_id...>\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights assign-all-device-linked\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights extend [--account <user_account_index>] <elicense_id...>\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights sync [--account <user_account_index>]\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights download-ticket [--account <user_account_index>] <elicense_id...>\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights revoke [--account <user_account_index>] <elicense_id...> [-f|--force]\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights revoke-all [--account <user_account_index>] [-f|--force]\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights query-revoke-reason [--account <user_account_index>] <elicense_id...>\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights report-active [--account <user_account_index>] <elicense_id...>\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights report-active-passively [--account <user_account_index>] <elicense_id...>\n"
        "       " DEVMENUCOMMAND_NAME " dynamicrights status-all [--account <user_account_index>] [--status <SERVER_STATUS,...>]\n"
        "\n"
        "       rights_id      <64bit Hexadecimal>\n"
        "       elicense_id    <128bit Hexadecimal>\n"
        "       TYPE           0-4|unknown|device_linked_permanent|permanent|account_restrictive_permanent|temporary\n"
        "       STATUS         0-4|unknown|assignable|already_assigned|not_supported|limit_exceeded\n"
        "       SERVER_STATUS  active|inactive_on_server|inactive\n"
#endif
        "";

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    Result Query(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid && !option.HasKey("--without-account"))
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int rightsIdCount = -1;
        es::RightsId rightsIds[RightsIdsMax];
        NN_RESULT_DO(GetRightsIdsFromTarget(&rightsIdCount, rightsIds, RightsIdsMax, option));
        if (rightsIdCount < 0)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        util::optional<nim::ELicenseType> elicenseTypeFilter;
        NN_RESULT_DO(GetELicenseTypeFromOption(&elicenseTypeFilter, option));
        if (elicenseTypeFilter && *elicenseTypeFilter == nim::ELicenseType::EndOfContents)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        util::optional<nim::ELicenseStatus> elicenseStatusFilter;
        NN_RESULT_DO(GetELicenseStatusFromOption(&elicenseStatusFilter, option));
        if (elicenseStatusFilter && *elicenseStatusFilter == nim::ELicenseStatus::EndOfContents)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nim::AsyncAvailableELicenses async;
        if (option.HasKey("--without-account"))
        {
            NN_RESULT_DO(nim::RequestQueryAvailableELicenses(&async, rightsIds, rightsIdCount));
        }
        else
        {
            NN_RESULT_DO(nim::RequestQueryAvailableELicenses(&async, uid, rightsIds, rightsIdCount));
        }

        int licenseSize = 0;
        NN_RESULT_DO(async.GetSize(&licenseSize));
        if (licenseSize <= 0)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int licenseCount = 0;
        std::vector<nim::AvailableELicense> licenses(licenseSize);
        NN_RESULT_DO(async.Get(&licenseCount, licenses.data(), licenseSize));
        NN_ABORT_UNLESS_EQUAL(licenseCount, licenseSize);

        DEVMENUCOMMAND_LOG("[DynamicRights] Available eLicense Total = %d\n", licenseSize);
        DEVMENUCOMMAND_LOG("--------------------------------------------------------------------\n");
        for (const auto& l : licenses)
        {
            if ((elicenseTypeFilter   && l.licenseType   != *elicenseTypeFilter) ||
                (elicenseStatusFilter && l.licenseStatus != *elicenseStatusFilter))
            {
                continue;
            }
            PrintAvailableELicense(l);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Assign(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid && !option.HasKey("--without-account"))
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int rightsIdCount = -1;
        es::RightsId rightsIds[RightsIdsMax];
        NN_RESULT_DO(GetRightsIdsFromTarget(&rightsIdCount, rightsIds, RightsIdsMax, option));
        if (rightsIdCount < 0)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        util::optional<nim::ELicenseType> elicenseType;
        NN_RESULT_DO(GetELicenseTypeFromOption(&elicenseType, option));
        if (!elicenseType)
        {
            elicenseType = nim::ELicenseType::Temporary;
        }
        else if (elicenseType && *elicenseType == nim::ELicenseType::EndOfContents)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nim::AsyncAssignedELicenses async;
        if (option.HasKey("--without-account"))
        {
            NN_RESULT_DO(nim::RequestAssignELicenses(&async, rightsIds, rightsIdCount, *elicenseType));
        }
        else
        {
            NN_RESULT_DO(nim::RequestAssignELicenses(&async, uid, rightsIds, rightsIdCount, *elicenseType));
        }

        int licenseSize = 0;
        NN_RESULT_DO(async.GetSize(&licenseSize));
        if (licenseSize <= 0)
        {
            DEVMENUCOMMAND_LOG("[DynamicRights] Failed to assign elicense(s)\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int licenseCount = 0;
        std::vector<nim::AssignedELicense> licenses(licenseSize);
        NN_RESULT_DO(async.Get(&licenseCount, licenses.data(), licenseSize));
        NN_ABORT_UNLESS_EQUAL(licenseCount, licenseSize);

        DEVMENUCOMMAND_LOG("[DynamicRights] Assign Results\n");
        for (const auto& l : licenses)
        {
            PrintAssignedELicense(l);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result AssignAllDeviceLinked(bool *outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nim::AsyncAssignedELicenses async;
        NN_RESULT_DO(nim::RequestAssignAllDeviceLinkedELicenses(&async));

        int licenseSize = 0;
        NN_RESULT_DO(async.GetSize(&licenseSize));
        if (licenseSize <= 0)
        {
            DEVMENUCOMMAND_LOG("[DynamicRights] Failed to assign-all-device-linked elicense(s)\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int licenseCount = 0;
        std::vector<nim::AssignedELicense> licenses(licenseSize);
        NN_RESULT_DO(async.Get(&licenseCount, licenses.data(), licenseSize));
        NN_ABORT_UNLESS_EQUAL(licenseCount, licenseSize);

        DEVMENUCOMMAND_LOG("[DynamicRights] Assign All Device Linked Results\n");
        for (const auto& l : licenses)
        {
            PrintAssignedELicense(l);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Extend(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int elicenseIdCount = -1;
        es::ELicenseId elicenseIds[ELicenseIdsMax];
        NN_RESULT_DO(GetELicenseIdsFromTarget(&elicenseIdCount, elicenseIds, ELicenseIdsMax, option));
        if (elicenseIdCount < 0)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nim::AsyncAssignedELicenses async;
        NN_RESULT_DO(nim::RequestExtendELicenses(&async, uid, elicenseIds, elicenseIdCount));

        int licenseSize = 0;
        NN_RESULT_DO(async.GetSize(&licenseSize));
        if (licenseSize <= 0)
        {
            DEVMENUCOMMAND_LOG("[DynamicRights] Failed to extend elicense(s)\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int licenseCount = 0;
        std::vector<nim::AssignedELicense> licenses(licenseSize);
        NN_RESULT_DO(async.Get(&licenseCount, licenses.data(), licenseSize));
        NN_ABORT_UNLESS_EQUAL(licenseCount, licenseSize);

        DEVMENUCOMMAND_LOG("[DynamicRights] Extend Results\n");
        for (const auto& l : licenses)
        {
            PrintAssignedELicense(l);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Sync(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        account::NintendoAccountId naId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nim::AsyncResult async;
        NN_RESULT_DO(nim::RequestSyncELicenses(&async, naId));
        NN_RESULT_DO(async.Get());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DownloadTicket(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int elicenseIdCount = -1;
        es::ELicenseId elicenseIds[ELicenseIdsMax];
        NN_RESULT_DO(GetELicenseIdsFromTarget(&elicenseIdCount, elicenseIds, ELicenseIdsMax, option));
        if (elicenseIdCount < 0)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nim::AsyncResult async;
        NN_RESULT_DO(nim::RequestDownloadETickets(&async, uid, elicenseIds, elicenseIdCount));
        NN_RESULT_DO(async.Get());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RevokeCore(bool *outValue, es::ELicenseId *elicenseIds, int elicenseIdCount, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        account::NintendoAccountId naId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));

        bool isForced = option.HasKey("--force") || option.HasKey("-f");

        // HTTP body
        const char *path;
        RAPIDJSON_NAMESPACE::Document request(RAPIDJSON_NAMESPACE::kObjectType);
        RAPIDJSON_NAMESPACE::Value elicenseIdsJson(RAPIDJSON_NAMESPACE::kArrayType);
        auto& allocator = request.GetAllocator();
        if (elicenseIdCount < 0)
        {
            path = "/v1/elicenses/revoke_all";
        }
        else
        {
            path = "/v1/elicenses/revoke";
            for (int i = 0; i < elicenseIdCount; i++)
            {
                char eLicenseIdString[33];
                RAPIDJSON_NAMESPACE::Value eLicenseIdStringJson;
                elicenseIds[i].ToString(eLicenseIdString, sizeof(eLicenseIdString)),
                eLicenseIdStringJson.SetString(eLicenseIdString, 32, allocator);
                elicenseIdsJson.PushBack(eLicenseIdStringJson, allocator);
            }
            request.AddMember("elicense_ids", elicenseIdsJson, allocator);
        }
        request.AddMember("is_forced", RAPIDJSON_NAMESPACE::Value(isForced), allocator);

        RAPIDJSON_NAMESPACE::Document response;
        *outValue = PostByCurl(&response, naId, path, request);

        NN_RESULT_SUCCESS;
    }

    Result Revoke(bool *outValue, const Option& option) NN_NOEXCEPT
    {
        int elicenseIdCount = -1;
        es::ELicenseId elicenseIds[ELicenseIdsMax];
        NN_RESULT_DO(GetELicenseIdsFromTarget(&elicenseIdCount, elicenseIds, ELicenseIdsMax, option));
        if (elicenseIdCount < 0)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(RevokeCore(outValue, elicenseIds, elicenseIdCount, option));
        NN_RESULT_SUCCESS;
    }

    Result RevokeAll(bool *outValue, const Option& option) NN_NOEXCEPT
    {
        if (option.GetTargetCount() > 0)
        {
            DEVMENUCOMMAND_LOG("revoke-all subcommand takes no target.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(RevokeCore(outValue, nullptr, -1, option));
        NN_RESULT_SUCCESS;
    }

    Result QueryRevokeReason(bool *outValue, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        account::NintendoAccountId naId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));

        int elicenseIdCount = -1;
        es::ELicenseId elicenseIds[ELicenseIdsMax];
        NN_RESULT_DO(GetELicenseIdsFromTarget(&elicenseIdCount, elicenseIds, ELicenseIdsMax, option));
        if (elicenseIdCount < 0)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        for (int i = 0; i < elicenseIdCount; i++)
        {
            nim::AsyncRevokeReason async;
            NN_RESULT_DO(nim::RequestQueryRevokeReason(&async, naId, elicenseIds[i]));

            nim::RevokeReason reason;
            NN_RESULT_DO(async.Get(&reason));

            char eLicenseIdString[33];
            DEVMENUCOMMAND_LOG("ELicenseID:%s RevokeReason:%s\n", elicenseIds[i].ToString(eLicenseIdString, sizeof(eLicenseIdString)), GetStringFrom(reason));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ReportActiveCore(bool* outValue, const Option& option, bool passively) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        account::NintendoAccountId naId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));

        int elicenseIdCount = -1;
        es::ELicenseId elicenseIds[ELicenseIdsMax];
        NN_RESULT_DO(GetELicenseIdsFromTarget(&elicenseIdCount, elicenseIds, ELicenseIdsMax, option));
        if (elicenseIdCount < 0)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nim::AsyncResult async;
        if (passively)
        {
            NN_RESULT_DO(nim::RequestReportActiveELicensesPassively(&async, naId, elicenseIds, elicenseIdCount));
        }
        else
        {
            NN_RESULT_DO(nim::RequestReportActiveELicenses(&async, naId, elicenseIds, elicenseIdCount));
        }
        NN_RESULT_DO(async.Get());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ReportActive(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        return ReportActiveCore(outValue, option, false);
    }

    Result ReportActivePassively(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        return ReportActiveCore(outValue, option, true);
    }

    Result StatusAll(bool *outValue, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        account::NintendoAccountId naId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));

        RAPIDJSON_NAMESPACE::Document response;
        *outValue = GetByCurl(&response, naId, "/v1/elicenses/status_all");
        if (!*outValue)
        {
            NN_RESULT_SUCCESS;
        }

        // filter by status
        if (option.HasKey("--status"))
        {
            std::vector<std::string> statuses;
            std::stringstream ss(option.GetValue("--status"));
            std::string str;
            while (std::getline(ss, str, ','))
            {
                statuses.push_back(str);
            }

            if (response.HasMember("elicenses") && response["elicenses"].IsArray())
            {
                auto& elicenses = response["elicenses"];
                auto f = [statuses](const auto& l) -> bool
                {
                    return l.HasMember("status") && std::find(statuses.begin(), statuses.end(), l["status"].GetString()) == statuses.end();
                };
                elicenses.Erase(std::remove_if(elicenses.Begin(), elicenses.End(), f), elicenses.End());
            }
        }

        RAPIDJSON_NAMESPACE::StringBuffer buffer;
        RAPIDJSON_NAMESPACE::PrettyWriter<RAPIDJSON_NAMESPACE::StringBuffer> writer(buffer);
        response.Accept(writer);
        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());

        NN_RESULT_SUCCESS;
    }
}

Result DynamicRightsCommand(bool* outValue, const Option& option)
{
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
    NN_UNUSED(g_HelpMessage);
#endif

    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(g_HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help" || option.HasKey("--help"))
    {
        DEVMENUCOMMAND_LOG(g_HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    account::InitializeForAdministrator();
    NN_RESULT_DO(time::InitializeForMenu());

    const SubCommand SubCommands[] =
    {
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "query", Query },
        { "assign", Assign },
        { "assign-all-device-linked", AssignAllDeviceLinked },
        { "extend", Extend },
        { "sync", Sync },
        { "download-ticket", DownloadTicket },
        { "revoke", Revoke },
        { "revoke-all", RevokeAll },
        { "query-revoke-reason", QueryRevokeReason },
        { "report-active", ReportActive },
        { "report-active-passively", ReportActivePassively },
        { "status-all", StatusAll },
#endif
    };

    for (const auto& subCommand : SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a " DEVMENUCOMMAND_NAME " dynamicrights command. See '" DEVMENUCOMMAND_NAME " dynamicrights --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}

