﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/system/btm_SystemResult.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/idle/idle_SystemApi.h>

#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadPalma.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Palma.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_RegisteredDevice.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/debug/hid_ControllerColor.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_ControllerCommand.h"

using namespace nn;

namespace
{
    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " controller count-pairing\n"
        "       " DEVMENUCOMMAND_NAME " controller delete-all-pairing\n"
        "       " DEVMENUCOMMAND_NAME " controller button-pairing\n"
        "       " DEVMENUCOMMAND_NAME " controller connect-all\n"
        "       " DEVMENUCOMMAND_NAME " controller disconnect-all\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " controller list\n"
        "       " DEVMENUCOMMAND_NAME " controller update-color <index> --mainColor <r>-<g>-<b> --subColor <r>-<g>-<b> [--thirdColor <r>-<g>-<b>] [--forthColor <r>-<g>-<b>] \n"
        "       " DEVMENUCOMMAND_NAME " controller palma-pairing\n"
#endif
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    const nn::hid::NpadIdType NpadIdsForWireless[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::No4,
        nn::hid::NpadId::No5,
        nn::hid::NpadId::No6,
        nn::hid::NpadId::No7,
        nn::hid::NpadId::No8
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    bool IsUniquePadConnected(nn::bluetooth::Address& address)
    {

        ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];

        auto uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                                  sizeof(uniquePadIds) / sizeof(uniquePadIds[0]));
        for (int i = 0; i < uniquePadIdCount; i++)
        {
            ::nn::bluetooth::Address gotAddress;
            if (::nn::hid::system::GetUniquePadBluetoothAddress(&gotAddress, uniquePadIds[i]).IsSuccess())
            {
                if (gotAddress == address)
                {
                    return true;
                }
            }
        }
        return false;
    }

    bool Connect(nn::bluetooth::Address& address) NN_NOEXCEPT
    {
        bool isConnected = false;


        ::nn::os::MultiWaitType multiWait;

        ::nn::os::SystemEventType timeoutEvent;
        ::nn::os::MultiWaitHolderType timeoutEventHolder;

        ::nn::os::SystemEventType connectionEvent;
        ::nn::os::MultiWaitHolderType connectionEventHolder;

        ::nn::os::TimerEventType timerEvent;
        ::nn::os::MultiWaitHolderType timerEventHolder;

        ::nn::os::InitializeMultiWait(&multiWait);

        ::nn::hid::system::BindConnectionTriggerTimeoutEvent(&timeoutEvent,
                                                             ::nn::os::EventClearMode_ManualClear);
        ::nn::os::InitializeMultiWaitHolder(&timeoutEventHolder, &timeoutEvent);
        ::nn::os::LinkMultiWaitHolder(&multiWait, &timeoutEventHolder);

        ::nn::hid::system::BindUniquePadConnectionEvent(&connectionEvent,
                                                        ::nn::os::EventClearMode_ManualClear);
        ::nn::os::InitializeMultiWaitHolder(&connectionEventHolder, &connectionEvent);
        ::nn::os::LinkMultiWaitHolder(&multiWait, &connectionEventHolder);

        ::nn::os::InitializeTimerEvent(&timerEvent,
                                       ::nn::os::EventClearMode_ManualClear);
        ::nn::os::InitializeMultiWaitHolder(&timerEventHolder, &timerEvent);
        ::nn::os::LinkMultiWaitHolder(&multiWait, &timerEventHolder);

        bool runs = true;

        // 一発目の接続イベントに向けたタイマーをセット
        ::nn::os::StartOneShotTimerEvent(&timerEvent, ::nn::TimeSpan::FromSeconds(1));

#ifdef WORKAROUND
        ::nn::os::TimerEventType workaroundEvent;
        ::nn::os::MultiWaitHolderType workaroundEventHolder;

        ::nn::os::InitializeTimerEvent(&workaroundEvent,
            ::nn::os::EventClearMode_ManualClear);
        ::nn::os::InitializeMultiWaitHolder(&workaroundEventHolder, &workaroundEvent);
        ::nn::os::LinkMultiWaitHolder(&multiWait, &workaroundEventHolder);

        // ワークアラウンドイベントに向けたタイマーをセット
        const auto WorkaroundTimeoutSeconds = 15;
        ::nn::os::StartOneShotTimerEvent(&workaroundEvent, ::nn::TimeSpan::FromSeconds(WorkaroundTimeoutSeconds));
#endif

        while(NN_STATIC_CONDITION(runs))
        {
            auto pHolder = nn::os::WaitAny(&multiWait);

            if(pHolder == &timerEventHolder)
            {
                ::nn::os::ClearTimerEvent(&timerEvent);

                // 接続を試みる
                auto connectionResult = ::nn::hid::system::SendConnectionTrigger(address);
                if (connectionResult.IsFailure())
                {
                    DEVMENUCOMMAND_LOG("[hid] Send Trigger Failed\n");
                    ::nn::os::StartOneShotTimerEvent(&timerEvent, ::nn::TimeSpan::FromSeconds(1));
                }
            }
            if(pHolder == &timeoutEventHolder)
            {
                ::nn::os::ClearSystemEvent(&timeoutEvent);

                DEVMENUCOMMAND_LOG("ConnectionTriggerTimeout\n");
                runs = false;
            }
            if(pHolder == &connectionEventHolder)
            {
                ::nn::os::ClearSystemEvent(&connectionEvent);

                if (IsUniquePadConnected(address) == true)
                {
                    runs = false;
                    isConnected = true;
                }
                else
                {
                    // 接続トリガーをかけているデバイスはまだ接続されていない
                }
            }

#ifdef WORKAROUND
            if(pHolder == &workaroundEventHolder)
            {
                ::nn::os::ClearTimerEvent(&workaroundEvent);

                NN_LOG("Workaround Timeout\n");
                runs = false;
            }
#endif

        }
        return isConnected;
    }

    void Disconnect(nn::hid::system::UniquePadId& uniquePadId)
    {
        ::nn::hid::system::DisconnectUniquePad(uniquePadId);
    }

    void DumpResult(const ::nn::hid::system::RegisteredDevice& device,
                    const bool& isConnected)
    {
        DEVMENUCOMMAND_LOG("\tConnection : %s\n", isConnected ? "Success" : "Fail");
        DEVMENUCOMMAND_LOG("\tBluetooth Address = %x:%x:%x:%x:%x:%x\n"
            , device.address.address[0]
            , device.address.address[1]
            , device.address.address[2]
            , device.address.address[3]
            , device.address.address[4]
            , device.address.address[5]
        );
    }
#endif

    void ConnectAll() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        ::nn::hid::system::RegisteredDevice registeredDevices[::nn::hid::system::RegisteredDeviceCountMax] = {};

        const auto RegisteredDeviceCount = ::nn::hid::system::GetRegisteredDevices(registeredDevices,
                                                                                   NN_ARRAY_SIZE(registeredDevices));

        for (int i = 0; i < RegisteredDeviceCount; i++)
        {
            int retryCount = 0;
            const int RetryCountMax = 5;

            bool runs = true;
            bool isConnected = false;

            ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];
            int connectedDeviceCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                                         NN_ARRAY_SIZE(uniquePadIds));

            auto& device = registeredDevices[i];

            // 既にペアリング台数分接続済なので何もしない
            if (connectedDeviceCount == RegisteredDeviceCount)
            {
                DEVMENUCOMMAND_LOG("[%s] Do nothing, since all the paired %d devices are connected.\n"
                    , __FUNCTION__
                    , RegisteredDeviceCount);
                return;
            }

            while (NN_STATIC_CONDITION(runs))
            {
                DEVMENUCOMMAND_LOG("Device = %d/%d : Retry = %d/%d\n"
                    , i + 1
                    , RegisteredDeviceCount
                    , retryCount
                    , RetryCountMax);

                isConnected = Connect(device.address);

                if (isConnected)
                {
                    runs = false;
                }
                else if (retryCount >= RetryCountMax)
                {
                    runs = false;
                }

                retryCount++;
            }

            DumpResult(device, isConnected);
        }
#endif
    }

    void DisconnectAll() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];

        auto uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                                  NN_ARRAY_SIZE(uniquePadIds));
        for (int i = 0; i < uniquePadIdCount; i++)
        {
            Disconnect(uniquePadIds[i]);
        }
#endif
    }

    Result ControllerCountPairingCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        auto count = nn::btm::system::GetPairedGamepadCount();
#else
        int count = 0;
#endif

        NN_LOG("%d\n", count);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ControllerDeleteAllPairingCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        auto count = nn::btm::system::GetPairedGamepadCount();
#else
        int count = 0;
#endif
        if (count > 0)
        {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
            nn::btm::system::ClearGamepadPairingDatabase();

            auto checkCount = count;
            while (checkCount > 0)
            {
                int passedTime = 0;
#ifdef NN_BUILD_CONFIG_OS_HORIZON
                nn::idle::ReportUserIsActive();
#endif
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
                passedTime += 500;
                if (passedTime > 10000) // 10 秒経っても削除が完了していなかったらタイムアウト扱いにする
                {
                    NN_LOG("Timeout. Deletion of controller pairing settings has not been completed.\n");
                    NN_LOG("Please check paired controllers count by \"controller count-pairing\" option and retry deletion.\n");
                }
                checkCount = nn::btm::system::GetPairedGamepadCount();
            }
#endif

            NN_LOG("%d controller paring settings are deleted.\n", count);
        }
        else
        {
            NN_LOG("No controller pairing settings.\n", count);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }


    Result ControllerButtonPairingCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        *outValue = false;
#if defined(NN_BUILD_CONFIG_OS_WIN)
#else
        nn::hid::InitializeNpad();

        // 接続されているすべてのコントローラーを切断
        DEVMENUCOMMAND_LOG("Disconnecting All Controller\n");
        for (auto& npadId : NpadIdsForWireless)
        {
            nn::hid::DisconnectNpad(npadId);
        }

        // 確実にすべてが切断されることを保障するために 500msec スリープ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        // FullKey と JoyDual のみ許可
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask);
        const auto targetNpadId = nn::hid::NpadId::No1;
        nn::hid::SetSupportedNpadIdType(&targetNpadId, 1);

        DEVMENUCOMMAND_LOG("Press SYNC Button\n");
        DEVMENUCOMMAND_LOG("Waiting for controller to be connected\n");
        // ペアリング待ち受けを開始
        nn::btm::system::StartGamepadPairing();

        // コントローラーが接続されるまで待機
        while (NN_STATIC_CONDITION(true))
        {
            auto npadStyleOnTarget = nn::hid::GetNpadStyleSet(targetNpadId);
            if (npadStyleOnTarget.IsAnyOn())
            {
                auto interfaceType = nn::hid::system::GetNpadInterfaceType(targetNpadId);
                if (interfaceType == nn::hid::system::InterfaceType_Bluetooth)
                {
                    DEVMENUCOMMAND_LOG("Controller Paired\n");
                    *outValue = true;
                    break;
                }

                // Blueotooth 以外でコントローラーが接続されていたら切断
                nn::hid::DisconnectNpad(nn::hid::NpadId::No1);
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }

        // ペアリングを終了
        nn::btm::system::CancelGamepadPairing();

#endif
        NN_RESULT_SUCCESS;
    }


    Result ControllerConnectAllCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        ConnectAll();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ControllerDisconnectAllCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        DisconnectAll();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    bool ParseColor(nn::util::Color4u8Type* pOutValue, const char* key, const Option& option) NN_NOEXCEPT
    {
        if (!option.HasKey(key))
        {
            return false;
        }
        else
        {
            auto colorParam = option.GetValue(key);
            if (!colorParam)
            {
                return false;
            }

            char* endPtr;
            pOutValue->v[0] = static_cast<uint8_t>(STR_TO_ULL(colorParam, &endPtr, 10));
            if (*endPtr != '-')
            {
                DEVMENUCOMMAND_LOG("%s is invalid color format. It must be like \"256-256-256\".\n", colorParam);
                return false;
            }

            char* endPtr2;
            pOutValue->v[1] = static_cast<uint8_t>(STR_TO_ULL(++endPtr, &endPtr2, 10));
            if (*endPtr2 != '-')
            {
                DEVMENUCOMMAND_LOG("%s is invalid color format. It must be like \"256-256-256\".\n", colorParam);
                return false;
            }

            char* endPtr3;
            pOutValue->v[2] = static_cast<uint8_t>(STR_TO_ULL(++endPtr2, &endPtr3, 10));
            if (*endPtr3 != '\0')
            {
                DEVMENUCOMMAND_LOG("%s is invalid color format. It must be like \"256-256-256\".\n", colorParam);
                return false;
            }

            pOutValue->v[3] = 0xff;
        }

        return true;
    }


    bool ListUniquePads() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        nn::hid::system::UniquePadId uniquePadIds[nn::hid::system::UniquePadIdCountMax];
        auto count = nn::hid::system::ListUniquePads(uniquePadIds, NN_ARRAY_SIZE(uniquePadIds));
        NN_LOG("index  No.      Type\n");
        NN_LOG("------------------------------------\n");

        for (int uniquePadIndex = 0; uniquePadIndex < count; ++uniquePadIndex)
        {
            auto& uniquePadId = uniquePadIds[uniquePadIndex];
            NN_LOG("%5lld: ", uniquePadId._storage);
            int controllerNumber;
            if (nn::hid::system::GetUniquePadControllerNumber(&controllerNumber, uniquePadId).IsSuccess())
            {
                switch (controllerNumber)
                {
                case 0:
                    NN_LOG("Handheld ");
                    break;
                case 1:
                    NN_LOG("o---     ");
                    break;
                case 2:
                    NN_LOG("oo--     ");
                    break;
                case 3:
                    NN_LOG("ooo-     ");
                    break;
                case 4:
                    NN_LOG("oooo     ");
                    break;
                case 5:
                    NN_LOG("o--o     ");
                    break;
                case 6:
                    NN_LOG("o-o-     ");
                    break;
                case 7:
                    NN_LOG("o-oo     ");
                    break;
                case 8:
                    NN_LOG("-oo-     ");
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
            else
            {
                NN_LOG("Unexpected Error occur\n");
                return false;
            }
            switch (nn::hid::system::GetUniquePadType(uniquePadId))
            {
            case nn::hid::system::UniquePadType_FullKeyController:
                NN_LOG("Pro Controller ");
                break;
            case nn::hid::system::UniquePadType_LeftController:
                NN_LOG("Joy-Con (L)    ");
                break;
            case nn::hid::system::UniquePadType_RightController:
                NN_LOG("Joy-Con (R)    ");
                break;
            case nn::hid::system::UniquePadType_Embedded:
                NN_LOG("Console        ");
                break;
            case nn::hid::system::UniquePadType_Unknown:
                NN_LOG("Unknown        ");
                break;
            default:
                ;
            }
            NN_LOG("\n");
        }
#endif
        return true;
    }

    Result ControllerListCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        *outValue = ListUniquePads();
        NN_RESULT_SUCCESS;
    }

    Result ControllerUpdateColorCommand(bool* outValue, const Option& option)
    {
        *outValue = false;
#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(option);
#else
        nn::hid::system::UniquePadId uniquePadId;
        uniquePadId._storage = static_cast<int>(std::strtoull(option.GetTarget(), NULL, 10));
        nn::util::Color4u8Type mainColor;
        nn::util::Color4u8Type subColor;
        nn::util::Color4u8Type thirdColor;
        nn::util::Color4u8Type forthColor;

        if (!ParseColor(&mainColor, "--mainColor", option) ||
            !ParseColor(&subColor, "--subColor", option))
        {
            DEVMENUCOMMAND_LOG("Please specify color.\n");
            NN_RESULT_SUCCESS;
        }
        if (ParseColor(&thirdColor, "--thirdColor", option) &&
            ParseColor(&forthColor, "--forthColor", option))
        {
            // 3色目, 4色目未指定の場合は、デザイン情報全体を更新
            // バリエーションは 0固定
            if (nn::hid::debug::UpdateDesignInfo(mainColor, subColor, thirdColor, forthColor, 0, uniquePadId).IsFailure())
            {
                DEVMENUCOMMAND_LOG("Controller Not Connected.\n");
                DEVMENUCOMMAND_LOG("Select index from following device(s).\n\n");
                ListUniquePads();
                NN_RESULT_SUCCESS;
            }
            *outValue = true;
        }
        else
        {
            // 3色目, 4色目未指定の場合は、メイン/サブのみ更新
            if(nn::hid::debug::UpdateControllerColor(mainColor, subColor, uniquePadId).IsFailure())
            {
                DEVMENUCOMMAND_LOG("Controller Not Connected.\n");
                DEVMENUCOMMAND_LOG("Select index from following device(s).\n\n");
                ListUniquePads();
                NN_RESULT_SUCCESS;
            }
            *outValue = true;

        }
#endif
        NN_RESULT_SUCCESS;
    }

    Result ControllerPairingPalmaCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        *outValue = false;
#if defined(NN_BUILD_CONFIG_OS_WIN)
#else
        nn::hid::InitializeNpad();

        // 接続されているすべてのコントローラーを切断
        DEVMENUCOMMAND_LOG("Disconnecting All Controller\n");
        for (auto& npadId : NpadIdsForWireless)
        {
            nn::hid::DisconnectNpad(npadId);
        }

        // 確実にすべてが切断されることを保障するために 500msec スリープ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        // PalmaStyle のみ許可
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStylePalma::Mask);
        const auto targetNpadId = nn::hid::NpadId::No1;
        nn::hid::SetSupportedNpadIdType(&targetNpadId, 1);

        DEVMENUCOMMAND_LOG("Press Button On Plama.\n");
        DEVMENUCOMMAND_LOG("Searching Palma.\n");
        // Palma のペアリング待ち受けを有効
        nn::hid::EnableAnyPalmaConnection();

        // Palma が接続されるまで待機
        while (NN_STATIC_CONDITION(true))
        {
            auto npadStyleOnTarget = nn::hid::GetNpadStyleSet(targetNpadId);
            if (npadStyleOnTarget.IsAnyOn())
            {
                if (npadStyleOnTarget.Test<nn::hid::NpadStylePalma>())
                {
                    nn::hid::PalmaConnectionHandle palmaHandle;
                    if (nn::hid::GetPalmaConnectionHandle(&palmaHandle, targetNpadId).IsSuccess())
                    {
                        DEVMENUCOMMAND_LOG("Found Palma. Bonding...\n");
                        nn::hid::PairPalma(palmaHandle);
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2000));
                        DEVMENUCOMMAND_LOG("Completed\n");
                        *outValue = true;
                        break;
                    }
                }

                // Palma以外が接続されていたら切断
                nn::hid::DisconnectNpad(nn::hid::NpadId::No1);
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }

        // Palma のペアリング待ち受けを無効化
        nn::hid::DisableAnyPalmaConnection();

#endif
        NN_RESULT_SUCCESS;
    }

#endif
    const SubCommand g_SubCommands[] =
    {
        { "count-pairing", ControllerCountPairingCommand },
        { "delete-all-pairing", ControllerDeleteAllPairingCommand },
        { "button-pairing", ControllerButtonPairingCommand },
        { "connect-all", ControllerConnectAllCommand },
        { "disconnect-all", ControllerDisconnectAllCommand },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "list", ControllerListCommand },
        { "update-color", ControllerUpdateColorCommand },
        { "palma-pairing", ControllerPairingPalmaCommand },
#endif
    };
} // namespace

Result ControllerCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu controller command. See '" DEVMENUCOMMAND_NAME " controller --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
