﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>
#include <algorithm>
#include <cctype>
#include <cstdlib>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <malloc>
#endif
#include <random>
#include <string>
#include <sstream>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/ec/system/ec_TicketApi.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Config.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/ns_ApplicationControlDataSystemApi.h>
#include <nn/ns/ns_ApplicationContentMetaApi.h>
#include <nn/ns/ns_ApplicationCopyIdentifierApi.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_ApplicationRightsApi.h>
#include <nn/ns/ns_ApplicationRightsSystemApi.h>
#include <nn/ns/ns_ApplicationVersionSystemApi.h>
#include <nn/ns/ns_ApplicationVerificationApi.h>
#include <nn/ns/ns_ApplicationViewSystemApi.h>
#include <nn/ns/ns_Async.h>
#include <nn/ns/ns_AsyncProgress.h>
#include <nn/ns/ns_ContentDeliveryApi.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_DownloadTaskApi.h>
#include <nn/ns/ns_DownloadTaskSystemApi.h>
#include <nn/ns/ns_DynamicRightsApi.h>
#include <nn/ns/ns_DynamicRightsSystemApi.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/ns/ns_NotificationSystemApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_TerminateResultApi.h>
#include <nn/ns/ns_TicketApi.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Drm.h>
#include <nn/time.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Span.h>
#include <nn/idle/idle_SystemApi.h>

#include "DevMenuCommand_ErrorContextUtil.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_InstallUtil.h"
#include "DevMenuCommand_ApplicationCommand.h"
#include "DevMenuCommand_ApplicationCommandData.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_HashUtil.h"
#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Language.h"
#include "DevMenuCommand_PrintUtil.h"
#include "DevMenuCommand_ShopUtil.h"
#include "DevMenuCommand_StorageId.h"
#include "DevMenuCommand_Stopwatch.h"

using namespace nn;
using namespace nne;
using namespace devmenuUtil;

namespace
{
    const size_t ListBufferSize = 8192;
    const size_t VerifyWorkBufferSize = 16 * 1024 * 1024;
    const size_t VerifyWorkBufferAlign = 4 * 1024;

    const char* ToString(ns::ApplicationEvent event)
    {
        switch (event)
        {
        case ns::ApplicationEvent::Launched: return "Launched";
        case ns::ApplicationEvent::LocalInstalled: return "LocalInstalled";
        case ns::ApplicationEvent::DownloadStarted: return "DownloadStarted";
        case ns::ApplicationEvent::GameCardInserted: return "GameCardInserted";
        case ns::ApplicationEvent::Touched: return "Touched";
        case ns::ApplicationEvent::GameCardRemoved: return "GameCardRemoved";
        case ns::ApplicationEvent::AttributeUpdated: return "AttributeUpdated";
        case ns::ApplicationEvent::DownloadTaskCommitted: return "DownloadTaskCommitted";
        case ns::ApplicationEvent::DownloadTaskCommittedPartially: return "DownloadTaskCommittedPartially";
        case ns::ApplicationEvent::ReceiveTaskCommitted: return "ReceiveTaskCommitted";
        case ns::ApplicationEvent::ApplyDeltaTaskCommitted: return "ApplyDeltaTaskCommitted";
        case ns::ApplicationEvent::Deleted: return "Deleted";
        case ns::ApplicationEvent::PropertyUpdated : return "PropertyUpdated";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* ToString(ns::ApplicationDownloadState state)
    {
        switch (state)
        {
        case ns::ApplicationDownloadState::Fatal: return "Fatal";
        case ns::ApplicationDownloadState::Finished: return "Finished";
        case ns::ApplicationDownloadState::NotEnoughSpace: return "NotEnoughSpace";
        case ns::ApplicationDownloadState::Runnable: return "Runnable";
        case ns::ApplicationDownloadState::Suspended: return "Suspended";
        case ns::ApplicationDownloadState::SystemUpdateRequired: return "SystemUpdateRequired";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    const char* ToString(ns::ApplicationApplyDeltaState state)
    {
        switch (state)
        {
        case ns::ApplicationApplyDeltaState::Applying: return "Applying";
        case ns::ApplicationApplyDeltaState::Suspended: return "Suspended";
        case ns::ApplicationApplyDeltaState::NotEnoughSpace: return "NotEnoughSpace";
        case ns::ApplicationApplyDeltaState::Fatal: return "Fatal";
        case ns::ApplicationApplyDeltaState::NoTask: return "NoTask";
        case ns::ApplicationApplyDeltaState::WaitApply: return "WaitApply";
        case ns::ApplicationApplyDeltaState::Applied: return "Applied";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* ToString(ncm::ContentInstallType type)
    {
        switch (type)
        {
        case ncm::ContentInstallType::Full: return "Full";
        case ncm::ContentInstallType::FragmentOnly: return "FragmentOnly";
        case ncm::ContentInstallType::Unknown: return "Unknown";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* ToString(nim::ApplyDownloadedDeltaProgressState state)
    {
        switch (state)
        {
        case nim::ApplyDownloadedDeltaProgressState::NotRunning: return "NotRunning";
        case nim::ApplyDownloadedDeltaProgressState::InitialPrepared: return "InitialPrepared";
        case nim::ApplyDownloadedDeltaProgressState::Applying: return "Applying";
        case nim::ApplyDownloadedDeltaProgressState::AllApplied: return "AllApplied";
        case nim::ApplyDownloadedDeltaProgressState::Commited: return "Commited";
        case nim::ApplyDownloadedDeltaProgressState::Suspended: return "Suspended";
        case nim::ApplyDownloadedDeltaProgressState::NotEnoughSpace: return "NotEnoughSpace";
        case nim::ApplyDownloadedDeltaProgressState::Fatal: return "Fatal";

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* ToString(ns::DownloadTaskStatusDetail type)
    {
        switch (type)
        {
        case ns::DownloadTaskStatusDetail::Created: return "Created";
        case ns::DownloadTaskStatusDetail::Added: return "Added";
        case ns::DownloadTaskStatusDetail::AlreadyExists: return "AlreadyExists";
        case ns::DownloadTaskStatusDetail::Failed: return "Failed";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* ToString(ns::ContentMetaRightsCheck check)
    {
        switch (check)
        {
        case ns::ContentMetaRightsCheck::NotChecked: return "NotChecked";
        case ns::ContentMetaRightsCheck::NotNeeded: return "NotNeeded";
        case ns::ContentMetaRightsCheck::CommonRights: return "CommonRights";
        case ns::ContentMetaRightsCheck::PersonalizedRights: return "PersonalizedRights";
        case ns::ContentMetaRightsCheck::NoRights: return "NoRights";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* ToString(ns::NotificationInfo::NotificationType type)
    {
        switch (type)
        {
        case ns::NotificationInfo::NotificationType::SystemUpdate: return "SystemUpdate";
        case ns::NotificationInfo::NotificationType::VersionList: return "VersionList";
        case ns::NotificationInfo::NotificationType::DownloadTaskList: return "DownloadTaskList";
        case ns::NotificationInfo::NotificationType::ETicketAvailable: return "ETicketAvailable";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* ToString(err::ErrorContextType type)
    {
        switch (type)
        {
        case err::ErrorContextType::None: return "None";
        case err::ErrorContextType::Http: return "Http";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* ToString(ncm::ContentType type)
    {
        switch (type)
        {
        case ncm::ContentType::Control: return "Control";
        case ncm::ContentType::Data: return "Data";
        case ncm::ContentType::DeltaFragment: return "DeltaFragment";
        case ncm::ContentType::HtmlDocument: return "HtmlDocument";
        case ncm::ContentType::LegalInformation: return "LegalInformation";
        case ncm::ContentType::Meta: return "Meta";
        case ncm::ContentType::Program: return "Program";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* ToString(nn::ns::ApplicationContentType type) NN_NOEXCEPT
    {
        switch (type)
        {
        case ns::ApplicationContentType::Application: return "Application";
        case ns::ApplicationContentType::AddOnContent: return "AddOnContent";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

#endif

    std::vector<ns::ApplicationRecord> ListApplicationRecord()
    {
        std::vector<ns::ApplicationRecord> list(ListBufferSize);
        auto count = ns::ListApplicationRecord(list.data(), ListBufferSize, 0);
        list.resize(static_cast<size_t>(count));
        return list;
    }

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    std::vector<ncm::ApplicationId> ListApplicationId()
    {
        std::vector<ncm::ApplicationId> idList;
        {
            auto list = ListApplicationRecord();
            for (auto& record : list)
            {
                idList.push_back(record.id);
            }
        }

        return idList;
    }

    std::vector<ns::ApplicationView> ListApplicationView()
    {
        auto idList = ListApplicationId();
        std::vector<ns::ApplicationView> viewList(idList.size());
        NN_ABORT_UNLESS_RESULT_SUCCESS(ns::GetApplicationView(viewList.data(), idList.data(), static_cast<int>(idList.size())));
        return viewList;
    }
#endif

    Result Install(bool* outValue, const Option& option)
    {
        // key に収まらなくてもインストールはされるので、とりあえず 1 個
        // TODO: StorageContentMetaKey に nullptr を指定したらそれで動くようにできるとよい
        // 互換性のために、どのコンテントメタタイプでもインストールできるように unknown を指定する
        ncm::StorageContentMetaKey key;
        int outCount;
        NN_RESULT_DO(devmenuUtil::InstallCommandCommon(&key, &outCount, 1, outValue, option, ncm::ContentMetaType::Unknown));
        NN_RESULT_SUCCESS;
    }

    void MakeNxArgsFilePath(char* outPath, size_t size, const ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        const char* DirectoryPath = "NintendoSDK/ApplicationLaunchParameters";
        const char* Extention = "nxargs";

        int count = nn::util::SNPrintf(outPath, size, "%s:/%s/%016llX.%s", SdCardMountName, DirectoryPath, applicationId.value, Extention);
        NN_SDK_ASSERT_LESS(count, static_cast<int>(size));
        NN_UNUSED(count);
    }

    Result InstallNxArgs(bool* outValue, const Option& option)
    {
        *outValue = false;

        auto nxargsPath = option.GetTarget();
        if (!devmenuUtil::IsAbsolutePath(nxargsPath))
        {
            DEVMENUCOMMAND_LOG("'%s' is not an absolute path.\n", nxargsPath);
            NN_RESULT_SUCCESS;
        }

        if (!option.HasKey("--application-id"))
        {
            DEVMENUCOMMAND_LOG("Please specify application id.\n");
            NN_RESULT_SUCCESS;
        }
        ncm::ApplicationId applicationId = { std::strtoull(option.GetValue("--application-id"), NULL, 16) };

        auto force = option.HasKey("--force");

        const size_t BufferSize = devmenu::LaunchParameterSizeMax;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);

        size_t fileSize;
        NN_RESULT_TRY(ReadFile(&fileSize, nxargsPath, buffer.get(), BufferSize))
            NN_RESULT_CATCH(nn::fs::ResultInvalidSize)
            {
                DEVMENUCOMMAND_LOG("An nxargs file must be less than or equal to %zu bytes\n", devmenu::LaunchParameterSizeMax);
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_TRY(nn::fs::MountSdCardForDebug(SdCardMountName))
            NN_RESULT_CATCH(nn::fs::ResultPortSdCardNoDevice)
            {
                DEVMENUCOMMAND_LOG("Please insert SD card\n");
                DEVMENUCOMMAND_LOG("An nxargs file will be installed into SD card\n");
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(SdCardMountName);
        };

        char path[80];
        MakeNxArgsFilePath(path, sizeof(path), applicationId);

        NN_RESULT_DO(devmenuUtil::CreateDirectoryRecursively(path, true));

        NN_RESULT_TRY(WriteFile(path, buffer.get(), fileSize, force))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
                DEVMENUCOMMAND_LOG("An nxargs file is already installed\n");
                DEVMENUCOMMAND_LOG("To overwrite, please use --force option\n");
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UninstallNxArgs(bool* outValue, const Option& option)
    {
        *outValue = false;

        auto applicationId = devmenuUtil::GetApplicationIdTarget(option);

        NN_RESULT_TRY(nn::fs::MountSdCardForDebug(SdCardMountName))
            NN_RESULT_CATCH(nn::fs::ResultPortSdCardNoDevice)
            {
                DEVMENUCOMMAND_LOG("SD card is not inserted\n");
                DEVMENUCOMMAND_LOG("An nxargs file is installed into SD card\n");
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(SdCardMountName);
        };

        char path[80];
        MakeNxArgsFilePath(path, sizeof(path), applicationId);

        NN_RESULT_TRY(nn::fs::DeleteFile(path))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    std::vector<std::string> Split(const std::string& input, char delimiter)
    {
        std::istringstream stream(input);

        std::string field;
        std::vector<std::string> result;
        while (std::getline(stream, field, delimiter)) {
            result.push_back(field);
        }
        return result;
    }

    Result WaitDownload(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        os::TimerEvent timer(os::EventClearMode_ManualClear);
        auto timeoutMs  = option.GetValue("--timeout");
        if (timeoutMs)
        {
            timer.StartOneShot(TimeSpan::FromMilliSeconds(std::strtoull(timeoutMs, nullptr, 10)));
        }

        int64_t downloadedTarget = 0;
        int64_t totalTarget = 0;
        auto downloadProgressOption = option.GetValue("--download-progress");
        if (downloadProgressOption)
        {
            auto splitted = Split(std::string(downloadProgressOption), '/');
            if (splitted.size() != 2)
            {
                DEVMENUCOMMAND_LOG("--download-progress option must be specified as <int64_t>/<int64_t>\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            downloadedTarget = std::stoll(splitted[0]);
            totalTarget = std::stoll(splitted[1]);
        }

        os::SystemEvent recordEvent;
        ns::GetApplicationRecordUpdateSystemEvent(&recordEvent);
        while (!timer.TryWait())
        {
            ns::ApplicationView view;
            NN_RESULT_TRY(ns::GetApplicationView(&view, &appId, 1))
                NN_RESULT_CATCH(ns::ResultApplicationRecordNotFound)
                {
                    DEVMENUCOMMAND_LOG("Application record 0x%016llx not found. Waiting download task registered...\n", appId.value);
                    os::WaitAny(timer.GetBase(), recordEvent.GetBase());
                    continue;
                }
            NN_RESULT_END_TRY;

            break;
        }

        uint32_t requestVersion = option.HasKey("--version") ? std::strtoul(option.GetValue("--version"), nullptr, 10) : 0;
        uint32_t currentVersion = 0;
        do
        {
            while (!timer.TryWait())
            {
                ns::ApplicationView view;
                NN_RESULT_DO(ns::GetApplicationView(&view, &appId, 1));
                currentVersion = view.version;
                if (!view.IsDownloading() && !view.IsWaitingCommit())
                {
                    break;
                }

                if (option.HasKey("--not-enough-space") && view.progress.state == ns::ApplicationDownloadState::NotEnoughSpace)
                {
                    *outValue = true;
                    NN_RESULT_SUCCESS;
                }
                if (view.IsDownloading() && view.progress.state != ns::ApplicationDownloadState::Runnable)
                {
                    DEVMENUCOMMAND_LOG("Download suspended as 0x%08x\n", view.progress.lastResult.GetInnerValueForDebug());
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }

                DEVMENUCOMMAND_LOG("%12lld / %lld, remainingTime: %lldmsec\n", view.progress.downloaded, view.progress.total, view.progress.remainingTime.GetMilliSeconds());
                if (downloadProgressOption)
                {
                    if ((view.progress.total >= totalTarget) &&
                        (view.progress.downloaded >= downloadedTarget))
                    {
                        *outValue = true;
                        NN_RESULT_SUCCESS;
                    }
                }
                if (view.IsWaitingCommit())
                {
                    if (option.HasKey("--wait-commit"))
                    {
                        *outValue = true;
                        NN_RESULT_SUCCESS;
                    }

                    DEVMENUCOMMAND_LOG("Wait commit unexpectedly\n");
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }

                os::TimedWaitAny(TimeSpan::FromSeconds(1), timer.GetBase());
            }

            while (!timer.TryWait())
            {
                ns::ApplicationView view;
                NN_RESULT_DO(ns::GetApplicationView(&view, &appId, 1));
                currentVersion = view.version;
                if (!view.IsApplyingDelta())
                {
                    break;
                }
                if (view.applyProgress.state != ns::ApplicationApplyDeltaState::Applying)
                {
                    DEVMENUCOMMAND_LOG("Apply suspended, state is %s\n", ToString(view.applyProgress.state));
                    if (view.applyProgress.state == ns::ApplicationApplyDeltaState::Fatal)
                    {
                        DEVMENUCOMMAND_LOG("Fatal: %08x\n", view.applyProgress.lastResult.GetInnerValueForDebug());
                    }
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
                DEVMENUCOMMAND_LOG("[Apply] %12lld / %lld, remainingTime: %lldmsec\n", view.applyProgress.applied, view.applyProgress.total, view.applyProgress.remainingTime.GetMilliSeconds());
                os::TimedWaitAny(TimeSpan::FromSeconds(1), timer.GetBase());
            }

            if (!timer.TryWait())
            {
                if (currentVersion >= requestVersion)
                {
                    break;
                }
                os::TimedWaitAny(TimeSpan::FromSeconds(1), timer.GetBase());
            }
        } while (!timer.TryWait());

        if (timer.TryWait())
        {
            DEVMENUCOMMAND_LOG("Timed out.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)
    Result WaitDownloadWithoutCommit(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        os::TimerEvent timer(os::EventClearMode_ManualClear);
        auto timeoutMs  = option.GetValue("--timeout");
        if (timeoutMs)
        {
            timer.StartOneShot(TimeSpan::FromMilliSeconds(std::strtoull(timeoutMs, nullptr, 10)));
        }
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network not available\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        {
            // requrest server は止める
            ns::RequestServerStopper stopper;
            nn::ns::GetRequestServerStopper(&stopper);

            // nim から直接ダウンロードタスク情報を取得する
            nim::NetworkInstallTaskId taskId;
            auto count = nim::ListApplicationNetworkInstallTask(&taskId, 1, appId);
            if (count != 1)
            {
                DEVMENUCOMMAND_LOG("Task not found for %016llx\n", appId.value);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            nim::AsyncResult async;
            NN_RESULT_DO(nim::RequestNetworkInstallTaskRun(&async, taskId));
            while (!timer.TryWait())
            {
                if (async.TryWait())
                {
                    auto result = async.Get();
                    // ref: ns_InstallUtil.cpp
                    if ((result.IsSuccess()
                         || result <= nim::ResultHttpConnectionCanceled()
                         || result <= nim::ResultHttpConnectionRetryableTimeout()
                         || result <= ncm::ResultInstallTaskCancelled()))
                    {
                        async.Finalize();
                        NN_RESULT_DO(nim::RequestNetworkInstallTaskRun(&async, taskId));
                    }
                    else
                    {
                        NN_RESULT_THROW(result);
                    }
                }
                nim::NetworkInstallTaskInfo info;
                NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, taskId));
                if (info.progress.state == ncm::Downloaded)
                {
                    DEVMENUCOMMAND_LOG("Downloaded without commit\n");
                    *outValue = true;
                    NN_RESULT_SUCCESS;
                }
                else if (info.progress.state == ncm::Fatal)
                {
                    Result result = util::Get(info.progress.lastResult);
                    DEVMENUCOMMAND_LOG("Task failed %08x\n", result.GetInnerValueForDebug());
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
                else
                {
                    DEVMENUCOMMAND_LOG("%12lld / %lld\n", info.progress.installedSize, info.progress.totalSize);
                }
#ifdef NN_BUILD_CONFIG_OS_HORIZON
                nn::idle::ReportUserIsActive();
#endif
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
            DEVMENUCOMMAND_LOG("Timed out\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
    }

    struct ApplicationDownloadSummary
    {
        ncm::ApplicationId              id;
        uint32_t                        version;
        Result                          cause;      // ns::ApplicationApplyDeltaProgress::lastResult
        ns::ApplicationDownloadState    state;      // ns::ApplicationApplyDeltaProgress::state

        class Collection : public std::vector<ApplicationDownloadSummary>
        {
        public:
            const_iterator FindByApplicationView(const ns::ApplicationView& view) const
            {
                return std::find_if(cbegin(), cend(), [&view](const ApplicationDownloadSummary& item)
                {
                    return (item.id == view.id && item.version == view.version);
                });
            }

            void ExclusivePushBack(const ns::ApplicationView& view)
            {
                if (cend() == FindByApplicationView(view))
                {
                    push_back(ApplicationDownloadSummary
                    {
                        view.id,
                        view.version,
                        view.progress.lastResult,
                        view.progress.state
                    });
                }
            }

            void Reject(const ns::ApplicationView& view)
            {
                auto found = FindByApplicationView(view);
                if ( cend() != found )
                {
                    erase( found );
                }
            }

            rapidjson::StringBuffer& GenerateJson(rapidjson::StringBuffer& buffer, bool withState = false)
            {
                rapidjson::Document document;
                document.SetArray();
                auto& allocator = document.GetAllocator();

                for (auto& entry : *this)
                {
                    rapidjson::Value item(rapidjson::kObjectType);
                    item.AddMember("applicationId", rapidjson::Value(devmenuUtil::GetBit64String(entry.id.value).data, allocator), allocator);
                    item.AddMember("version", rapidjson::Value(entry.version), allocator);
                    if (withState)
                    {
                        item.AddMember("state", rapidjson::Value(ToString(entry.state), allocator), allocator);
                        item.AddMember("lastResult", rapidjson::Value(devmenuUtil::GetBit32String(entry.cause.GetInnerValueForDebug()).data, allocator), allocator);
                    }
                    document.PushBack(item, allocator);
                }
                buffer.Clear();
                nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
                document.Accept(writer);
                return buffer;
            }
        };
    };

    Result WaitDownloadAll(bool* outValue, const Option& option)
    {
        os::TimerEvent timer(os::EventClearMode_ManualClear);
        auto timeoutMs = option.GetValue("--timeout");
        if (timeoutMs)
        {
            timer.StartOneShot(TimeSpan::FromMilliSeconds(std::strtoull(timeoutMs, nullptr, 10)));
        }

        size_t prePassCount = static_cast<size_t>(-1);
        size_t preFailCount = static_cast<size_t>(-1);
        ApplicationDownloadSummary::Collection order;
        ApplicationDownloadSummary::Collection failures;
        while (!timer.TryWait())
        {
            auto list = ListApplicationView();
            auto finished = std::all_of(list.rbegin(), list.rend(), [&order, &failures](decltype(list[0])& view)
            {
                if (!view.IsDownloading() && !view.IsApplyingDelta())
                {
                    failures.Reject(view);
                    order.ExclusivePushBack(view);
                    return true;
                }
                else if (ns::ApplicationDownloadState::NotEnoughSpace == view.progress.state
                    || ns::ApplicationDownloadState::Suspended == view.progress.state
                    || ns::ApplicationDownloadState::Fatal == view.progress.state)
                {
                    failures.ExclusivePushBack(view);
                    return true;
                }
                else
                {
                    return false;
                }
            });
            const auto passCount = order.size();
            const auto failCount = failures.size();
            if (prePassCount != passCount || preFailCount != failCount)
            {
                prePassCount = passCount;
                preFailCount = failCount;
                const auto doneCount = passCount + failCount;
                DEVMENUCOMMAND_LOG("%4lld / %lld ( OK : %4lld, FAIL : %4lld )\n", doneCount, list.size(), passCount, failCount);
            }
            if ( finished )
            {
                break;
            }
            os::TimedWaitAny(TimeSpan::FromSeconds(1), timer.GetBase());
        }

        if (timer.TryWait())
        {
            DEVMENUCOMMAND_LOG("Timed out.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nne::rapidjson::StringBuffer buffer;
        DEVMENUCOMMAND_LOG("%s\n", order.GenerateJson(buffer).GetString());
        DEVMENUCOMMAND_LOG("Failed application download requests %s\n", failures.GenerateJson(buffer, true).GetString());
        *outValue = ( 0 == failures.size() );
        NN_RESULT_SUCCESS;
    }

    Result CancelDownload(bool* outValue, const Option& option)
    {
        NN_RESULT_DO(ns::CancelApplicationDownload(devmenuUtil::GetApplicationIdTarget(option)));
        NN_RESULT_DO(ns::CancelApplicationApplyDelta(devmenuUtil::GetApplicationIdTarget(option)));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ResumeDownload(bool* outValue, const Option& option)
    {
        if (option.HasKey("--all"))
        {
            ns::ResumeAll();
            *outValue = true;
        }
        else
        {
            // TODO: ここではアドホックに、適用タスクの再開でタスクが見つからない場合は無視するが、
            //       ダウンロードでも見つからなかった場合はエラーを返している
            NN_RESULT_TRY(ns::ResumeApplicationApplyDelta(devmenuUtil::GetApplicationIdTarget(option)))
                NN_RESULT_CATCH(ns::ResultApplicationNotApplying) {}
            NN_RESULT_END_TRY

            NN_RESULT_DO(ns::ResumeApplicationDownload(devmenuUtil::GetApplicationIdTarget(option)));
            *outValue = true;
        }

        NN_RESULT_SUCCESS;
    }
    Result SetTaskState(bool* outValue, const Option& option)
    {
        auto id = devmenuUtil::GetApplicationIdTarget(option);

        if (!option.HasKey("--state"))
        {
            *outValue = false;
            DEVMENUCOMMAND_LOG("--state option should be specified.\n");
            NN_RESULT_SUCCESS;
        }
        std::string stateString = option.GetValue("--state");
        ns::ApplicationDownloadState state;
        if (stateString == "Suspended")
        {
            state = ns::ApplicationDownloadState::Suspended;
        }
        else if (stateString == "NotEnoughSpace")
        {
            state = ns::ApplicationDownloadState::NotEnoughSpace;
        }
        else
        {
            *outValue = false;
            DEVMENUCOMMAND_LOG("--state option should be either Suspended or NotEnoughSpace.\n");
            NN_RESULT_SUCCESS;
        }

        nim::NetworkInstallTaskId task;
        auto count = nim::ListApplicationNetworkInstallTask(&task, 1, id);
        if (count > 0)
        {
            ns::RequestServerStopper stopper;
            ns::GetRequestServerStopper(&stopper);

            NN_RESULT_DO(nim::SetNetworkInstallTaskAttribute(task, static_cast<Bit64>(state)));
            *outValue = true;
        }
        else
        {
            DEVMENUCOMMAND_LOG("Task of 0x%016llx not found.\n", id);
            *outValue = false;
        }

        NN_RESULT_SUCCESS;
    }

    Result Download(bool* outValue, const Option& option)
    {
        auto storage = option.HasKey("-s") ? ToInstallStorage(option.GetValue("-s")) : ToInstallStorage("builtin");
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        if (*storage == ncm::StorageId::SdCard)
        {
            NN_RESULT_DO(nn::ns::CheckSdCardMountStatus());
        }

        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);
        uint32_t version = option.HasKey("--version") ? std::strtoul(option.GetValue("--version"), nullptr, 10) : 0;

        NN_RESULT_DO(ns::BeginInstallApplication(appId, 0, *storage));
        DEVMENUCOMMAND_LOG("Waiting applicationId 0x%016llx version %u downloaded...\n", appId.value, version);

        NN_RESULT_DO(WaitDownload(outValue, option));

        NN_RESULT_SUCCESS;
    }


    rapidjson::StringBuffer GenerateSimpleDownloadTaskList(ncm::ApplicationId applicationId, ncm::ContentMetaType type, Bit64 contentMetaId, uint32_t version)
    {
        rapidjson::Document document;
        document.SetObject();

        rapidjson::Value task(rapidjson::kObjectType);
        util::Uuid uuid = util::GenerateUuid();
        char uuidStr[util::Uuid::StringSize];
        task.AddMember("id", rapidjson::StringRef(uuid.ToString(uuidStr, sizeof(uuidStr))), document.GetAllocator());
        char appIdStr[20];
        util::TSNPrintf(appIdStr, sizeof(appIdStr), "0x%016llx", applicationId.value);
        task.AddMember("owner_application", rapidjson::StringRef(appIdStr), document.GetAllocator());

        rapidjson::Value title(rapidjson::kObjectType);
        char cmetaIdStr[20];
        util::TSNPrintf(cmetaIdStr, sizeof(cmetaIdStr), "0x%016llx", contentMetaId);
        title.AddMember("id", rapidjson::StringRef(cmetaIdStr), document.GetAllocator());
        title.AddMember("version", rapidjson::Value(version), document.GetAllocator());
        title.AddMember("type", rapidjson::StringRef(devmenuUtil::GetContentMetaTypeString(type)), document.GetAllocator());

        rapidjson::Value titleList(rapidjson::kArrayType);
        titleList.PushBack(title, document.GetAllocator());

        task.AddMember("titles", titleList, document.GetAllocator());

        rapidjson::Value taskList(rapidjson::kArrayType);
        taskList.PushBack(task, document.GetAllocator());

        document.AddMember("tasks", taskList, document.GetAllocator());

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    Result CreateDownloadTask(bool* outValue, const Option& option)
    {
        auto appId = devmenuUtil::GetApplicationIdTarget(option);

        auto typeOption = option.GetValue("--type");
        auto idOption = option.GetValue("--id");
        auto versionOption = option.GetValue("--version");

        if (typeOption || idOption)
        {
            if (!typeOption || !idOption)
            {
                DEVMENUCOMMAND_LOG("--type and --id option must be used together\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        ncm::ContentMetaType type = ncm::ContentMetaType::Application;
        if (typeOption)
        {
            auto parsed = devmenuUtil::ParseDownloadableContentMetaType(typeOption);
            if (!parsed)
            {
                DEVMENUCOMMAND_LOG("%s is not downloadable content meta type.\n", typeOption);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            type = *parsed;
        }

        Bit64 id = appId.value;
        if (idOption)
        {
            char* endPtr;
            id = std::strtoull(idOption, &endPtr, 16);
            if (*endPtr != '\0')
            {
                DEVMENUCOMMAND_LOG("%s is not valid content meta id.\n", idOption);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        uint32_t version = 0;
        if (versionOption)
        {
            char* endPtr;
            version = std::strtoul(versionOption, &endPtr, 10);
            if (*endPtr != '\0')
            {
                DEVMENUCOMMAND_LOG("%s is not valid content meta version.\n", versionOption);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        auto json = GenerateSimpleDownloadTaskList(appId, type, id, version);
        DEVMENUCOMMAND_LOG("%s\n", json.GetString());

        NN_RESULT_DO(ns::PushDownloadTaskList(json.GetString(), json.GetSize()));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result PushDownloadTaskList(bool* outValue, const Option& option)
    {
        auto dtlPath = option.GetTarget();
        if (!devmenuUtil::IsAbsolutePath(dtlPath))
        {
            DEVMENUCOMMAND_LOG("'%s' is not an absolute path.\n", dtlPath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, dtlPath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, file));

        auto dtlSize = static_cast<size_t>(fileSize);
        std::unique_ptr<char[]> buffer(new char[dtlSize]);

        NN_RESULT_DO(fs::ReadFile(file, 0, buffer.get(), dtlSize));

        NN_RESULT_DO(nn::ns::PushDownloadTaskList(buffer.get(), dtlSize));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RequestDownloadTaskList(bool* outValue, const Option&)
    {
        ns::RequestDownloadTaskList();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RequestEnsureDownloadTask(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("Waiting network request accepted...\n");
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network not available\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ns::AsyncResult async;
        ns::RequestEnsureDownloadTask(&async);
        NN_RESULT_DO(Get(&async));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RequestDownloadTaskListData(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("Waiting network request accepted...\n");
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network not available\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ns::AsyncDownloadTaskListData async;
        NN_RESULT_DO(ns::RequestDownloadTaskListData(&async));

        std::unique_ptr<ns::DownloadTaskListData> data(new ns::DownloadTaskListData());
        NN_RESULT_TRY(Get(data.get(), &async))
            NN_RESULT_CATCH(nim::ResultDownloadTaskListNotFound)
            {
                DEVMENUCOMMAND_LOG("{}\n");
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        nne::rapidjson::Document document;
        if (document.ParseInsitu(data->data).HasParseError())
        {
            DEVMENUCOMMAND_LOG("Invalid download task list json\n");
        }
        rapidjson::StringBuffer buffer;
        rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    rapidjson::StringBuffer GenerateDownloadTaskStatusJson(const std::vector<ns::DownloadTaskStatus>& list)
    {
        rapidjson::Document document;
        document.SetArray();

        for (auto& status : list)
        {
            rapidjson::Value item(rapidjson::kObjectType);
            char uuidString[util::Uuid::StringSize];
            item.AddMember("uuid", rapidjson::Value(status.uuid.ToString(uuidString, sizeof(uuidString)), document.GetAllocator()), document.GetAllocator());
            item.AddMember("applicationId", rapidjson::Value(devmenuUtil::GetBit64String(status.applicationId.value).data, document.GetAllocator()), document.GetAllocator());
            item.AddMember("detail", rapidjson::StringRef(ToString(status.detail)), document.GetAllocator());
            item.AddMember("result", rapidjson::Value(devmenuUtil::GetBit32String(status.result).data, document.GetAllocator()), document.GetAllocator());

            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    Result ListDownloadTaskStatus(bool*outValue, const Option&)
    {
        std::vector<ns::DownloadTaskStatus> list(64);
        auto count = ns::ListDownloadTaskStatus(list.data(), 64);
        list.resize(static_cast<size_t>(count));

        auto buffer = GenerateDownloadTaskStatusJson(list);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearTaskStatusList(bool*outValue, const Option&)
    {
        nn::ns::ClearTaskStatusList();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DownloadControl(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        DEVMENUCOMMAND_LOG("Waiting network request accepted...\n");
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network not available\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ns::AsyncResult async;
        NN_RESULT_DO(ns::RequestDownloadApplicationControlData(&async, appId));
        DEVMENUCOMMAND_LOG("Waiting applicationId 0x%016llx control downloaded...\n", appId.value);
        NN_RESULT_DO(Get(&async));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckLaunchRights(bool* outValue, const Option& option)
    {
        NN_RESULT_TRY(ns::CheckApplicationLaunchRights(devmenuUtil::GetApplicationIdTarget(option)))
            NN_RESULT_CATCH(ns::ResultApplicationTicketNotFound)
            {
                DEVMENUCOMMAND_LOG("RightsNotFound\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultInactiveNintendoAccount)
            {
                DEVMENUCOMMAND_LOG("InactiveNintendoAccount\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultApplicationRecordNotFound)
            {
                DEVMENUCOMMAND_LOG("RecordNotFound\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultApplicationContentNotFound)
            {
                DEVMENUCOMMAND_LOG("ContentNotFound\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultApplicationPrepurchased)
            {
                DEVMENUCOMMAND_LOG("ContentPrepurchased\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        DEVMENUCOMMAND_LOG("Available\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetApplicationRightsOnClient(bool* outValue, const Option& option)
    {
        rapidjson::Document document;
        document.SetArray();

        std::array<ns::ApplicationRightsOnClient, 2> rightsList;
        int count;

        if (option.HasKey("--account-index"))
        {
            account::InitializeForAdministrator();
            util::optional<account::Uid> uid;
            NN_RESULT_DO(GetUidFromIndex(&uid, static_cast<int>(std::strtoul(option.GetValue("--account-index"), nullptr, 10))));
            if (!uid)
            {
                DEVMENUCOMMAND_LOG("Account not found.\n");
                NN_RESULT_SUCCESS;
            }

            NN_RESULT_DO(ns::GetApplicationRightsOnClient(&count, rightsList.data(), static_cast<int>(rightsList.size()), devmenuUtil::GetApplicationIdTarget(option), *uid));
        }
        else
        {
            NN_RESULT_DO(ns::GetApplicationRightsOnClient(&count, rightsList.data(), static_cast<int>(rightsList.size()), devmenuUtil::GetApplicationIdTarget(option)));
        }

        for (auto rights : util::MakeSpan(rightsList.data(), count))
        {
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("id", rapidjson::Value(devmenuUtil::GetBit64String(rights.id.value).data, document.GetAllocator()), document.GetAllocator());
            char uidString[16 * 2 + 2] = {};
            util::TSNPrintf(uidString, sizeof(uidString), "%016llx_%016llx", rights.uid._data[0], rights.uid._data[1]);
            item.AddMember("uid", rapidjson::Value(uidString, document.GetAllocator()), document.GetAllocator());
            item.AddMember("type", rapidjson::Value(ToString(rights.type), document.GetAllocator()), document.GetAllocator());
            item.AddMember("hasAvailableRights", rapidjson::Value(rights.HasAvailableRights()), document.GetAllocator());
            item.AddMember("hasUnavailableRights", rapidjson::Value(rights.HasUnavailableRights()), document.GetAllocator());
            item.AddMember("hasAccountRestrictedRights", rapidjson::Value(rights.HasAccountRestrictedRights()), document.GetAllocator());
            item.AddMember("recommendInquireServer", rapidjson::Value(rights.RecommendInquireServer()), document.GetAllocator());
            item.AddMember("hasPrepurchasedRights", rapidjson::Value(rights.HasPrepurchasedRights()), document.GetAllocator());

            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckResumeRights(bool* outValue, const Option& option)
    {
        NN_RESULT_TRY(ns::CheckApplicationResumeRights(devmenuUtil::GetApplicationIdTarget(option)))
            NN_RESULT_CATCH(ns::ResultApplicationTicketNotFound)
            {
                DEVMENUCOMMAND_LOG("RightsNotFound\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultInactiveNintendoAccount)
            {
                DEVMENUCOMMAND_LOG("InactiveNintendoAccount\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultApplicationRecordNotFound)
            {
                DEVMENUCOMMAND_LOG("RecordNotFound\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultApplicationContentNotFound)
            {
                DEVMENUCOMMAND_LOG("ContentNotFound\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UpdateVersionList(bool* outValue, const Option& option)
    {
        auto versionListPath = option.GetTarget();
        if (!devmenuUtil::IsAbsolutePath(versionListPath))
        {
            DEVMENUCOMMAND_LOG("'%s' is not an absolute path.\n", versionListPath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, versionListPath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, file));

        std::unique_ptr<char[]> versionListBuffer(new char[static_cast<size_t>(fileSize)]);
        NN_RESULT_THROW_UNLESS(versionListBuffer.get() != nullptr, nn::ns::ResultBufferNotEnough());
        NN_RESULT_DO(nn::fs::ReadFile(file, 0, versionListBuffer.get(), static_cast<size_t>(fileSize)));
        NN_RESULT_DO(nn::ns::UpdateVersionList(versionListBuffer.get(), static_cast<size_t>(fileSize)));

        if (!option.HasKey("--no-perform-update"))
        {
            nn::ns::PerformAutoUpdate();
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DumpRequiredVersion(bool* outValue, const Option&)
    {
        const int MaxEntryCount = 16 * 1024;
        std::unique_ptr<nn::ns::RequiredVersionEntry[]> entryBuffer(new nn::ns::RequiredVersionEntry[MaxEntryCount]);
        NN_RESULT_THROW_UNLESS(entryBuffer, nn::ns::ResultBufferNotEnough());
        auto out = entryBuffer.get();

        auto outCount = nn::ns::ListRequiredVersion(out, MaxEntryCount);
        DEVMENUCOMMAND_LOG("Dump required version\n");
        DEVMENUCOMMAND_LOG("Total count: %d\n", outCount);
        DEVMENUCOMMAND_LOG("\n");
        DEVMENUCOMMAND_LOG("ApplicationId         RequiredVersion\n");
        //      0x0123456789abcdef    0
        DEVMENUCOMMAND_LOG("-------------------------------------\n");
        for (int i = 0; i < outCount; ++i)
        {
            auto& entry = out[i];
            DEVMENUCOMMAND_LOG("0x%016llx     %14d\n", entry.id, entry.requiredVersion);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetRequiredVersion(bool* outValue, const Option& option)
    {
        const char* RequiredVersionKey = "-v";
        if (!option.HasKey(RequiredVersionKey))
        {
            DEVMENUCOMMAND_LOG("'-v' option is not specified.\n");
            DEVMENUCOMMAND_LOG("Example:\n");
            DEVMENUCOMMAND_LOG("application set-required-version 0005000c10000000 -v 1\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        uint32_t reqVersion = std::strtoul(option.GetValue(RequiredVersionKey), nullptr, 10);
        nn::ns::PushLaunchVersion(appId, reqVersion);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetRequiredVersionCommand(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        auto requiredVersion = ns::GetLaunchRequiredVersion(appId);
        DEVMENUCOMMAND_LOG("%d\n", requiredVersion);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UpgradeRequiredVersionCommand(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        if (!option.HasKey("-v"))
        {
            DEVMENUCOMMAND_LOG("'-v' option is not specified.\n");
            DEVMENUCOMMAND_LOG("Example:\n");
            DEVMENUCOMMAND_LOG("application upgrade-required-version 0x0005000c10000000 -v 1\n");
            NN_RESULT_SUCCESS;
        }
        auto version = std::strtoul(option.GetValue("-v"), NULL, 10);

        ns::UpgradeLaunchRequiredVersion(appId, version);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    Result ResetRequiredVersion(bool* outValue, const Option& option)
    {
        bool all = option.HasKey("--all");
        ncm::ApplicationId applicationId = devmenuUtil::GetApplicationIdTarget(option);
        bool found = false;

        const int MaxEntryCount = 16 * 1024;
        std::unique_ptr<nn::ns::RequiredVersionEntry[]> entryBuffer(new nn::ns::RequiredVersionEntry[MaxEntryCount]);
        NN_RESULT_THROW_UNLESS(entryBuffer, nn::ns::ResultBufferNotEnough());
        auto out = entryBuffer.get();
        auto outCount = nn::ns::ListRequiredVersion(out, MaxEntryCount);
        for (int i = 0; i < outCount; ++i)
        {
            auto& entry = out[i];
            ncm::ApplicationId id = {entry.id};
            if (all || applicationId == id)
            {
                DEVMENUCOMMAND_LOG("Reset: 0x%016llx\n", id.value);
                nn::ns::PushLaunchVersion(id, 0);
                found = true;
            }
        }
        if (!all && !found)
        {
            DEVMENUCOMMAND_LOG("ID not found: 0x%016llx\n", applicationId.value);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM

    Result CheckLaunchVersion(bool* outValue, const Option& option)
    {
        NN_RESULT_TRY(nn::ns::CheckApplicationLaunchVersion(devmenuUtil::GetApplicationIdTarget(option)))
            NN_RESULT_CATCH(ns::ResultSystemUpdateRequired)
            {
                DEVMENUCOMMAND_LOG("System update required.\n");
            }
            NN_RESULT_CATCH(ns::ResultApplicationUpdateRequired)
            {
                DEVMENUCOMMAND_LOG("Application update required.\n");
                if (ns::ResultApplicationUpdateRequiredByRequiredVersion::Includes(NN_RESULT_CURRENT_RESULT))
                {
                    DEVMENUCOMMAND_LOG("Required version is not satisfied.\n");
                }
                else if (ns::ResultApplicationUpdateRequiredByRequiredApplicationVersion::Includes(NN_RESULT_CURRENT_RESULT))
                {
                    DEVMENUCOMMAND_LOG("Required Application version is not satisfied.\n");
                }
                else
                {
                    *outValue = false;
                    NN_RESULT_RETHROW;
                }
            }
            NN_RESULT_CATCH(ns::ResultApplicationUpdateRecommended)
            {
                DEVMENUCOMMAND_LOG("Application update recommended.\n");
            }
            NN_RESULT_CATCH_ALL
            {
                *outValue = false;
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY;

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RequestVersionList(bool* outValue, const Option&)
    {
        ns::RequestVersionList();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RequestVersionListData(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("Waiting network request accepted...\n");
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network not available\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ns::AsyncVersionListData async;
        NN_RESULT_DO(ns::RequestVersionListData(&async));

        auto data = std::unique_ptr<ns::VersionListData>(new ns::VersionListData());
        NN_RESULT_DO(Get(data.get(), &async));

        DEVMENUCOMMAND_LOG("%s\n", data.get());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result PerformAutoUpdate(bool* outValue, const Option&)
    {
        ns::PerformAutoUpdate();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result VersionList(bool* outValue, const Option&)
    {
        std::vector<ns::VersionListEntry> list(ListBufferSize);
        auto count = ns::ListVersionList(list.data(), ListBufferSize);
        list.resize(static_cast<size_t>(count));
        if (!list.empty())
        {
            DEVMENUCOMMAND_LOG("ID                  Recommended     Notified\n");
            //      0x0123456789abcdef   0123456789   0123456789
        }
        for (auto& entry : list)
        {
            DEVMENUCOMMAND_LOG("0x%016llx   %10u   %10u\n", entry.id, entry.recommendedVersion, entry.notifiedVersion);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    rapidjson::StringBuffer GenerateControlMetaStatusJson(const std::vector<ns::ApplicationContentMetaStatus>& list)
    {
        rapidjson::Document document;
        document.SetArray();

        for (auto& status : list)
        {
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("id", rapidjson::Value(devmenuUtil::GetBit64String(status.id).data, document.GetAllocator()), document.GetAllocator());
            item.AddMember("type", rapidjson::Value(devmenuUtil::GetContentMetaTypeString(status.type), document.GetAllocator()), document.GetAllocator());
            item.AddMember("version", rapidjson::Value(status.version), document.GetAllocator());
            item.AddMember("installedStorage", rapidjson::Value(devmenuUtil::GetStorageIdString(status.installedStorage), document.GetAllocator()), document.GetAllocator());
            item.AddMember("rightsCheck", rapidjson::Value(ToString(status.rightsCheck), document.GetAllocator()), document.GetAllocator());

            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    Result ContentMetaStatus(bool* outValue, const Option& option)
    {
        auto idList = option.HasKey("--all") ? ListApplicationId() : std::vector<ncm::ApplicationId>(1, devmenuUtil::GetApplicationIdTarget(option));
        auto checkRights = option.HasKey("--check-rights");

        std::vector<ns::ApplicationContentMetaStatus> list;
        for (auto id : idList)
        {
            int count;
            NN_RESULT_DO(ns::CountApplicationContentMeta(&count, id));

            std::vector<ns::ApplicationContentMetaStatus> listOne(static_cast<size_t>(count));
            NN_RESULT_DO(checkRights ? ns::ListApplicationContentMetaStatusWithRightsCheck(&count, listOne.data(), count, id, 0) :
                ns::ListApplicationContentMetaStatus(&count, listOne.data(), count, id, 0));

            std::copy(listOne.begin(), listOne.end(), std::back_inserter(list));
        }

        DEVMENUCOMMAND_LOG("%s\n", GenerateControlMetaStatusJson(list).GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    rapidjson::StringBuffer GenerateControlCacheJson(const std::vector<ns::ApplicationControlCacheEntryInfo>& list)
    {
        rapidjson::Document document;
        document.SetArray();

        for (auto& entry : list)
        {
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("applicationId", rapidjson::Value(devmenuUtil::GetBit64String(entry.applicationId.value).data, document.GetAllocator()), document.GetAllocator());
            item.AddMember("version", rapidjson::Value(entry.version), document.GetAllocator());
            item.AddMember("language", rapidjson::Value(entry.language.string, document.GetAllocator()), document.GetAllocator());
            item.AddMember("iconSize", rapidjson::Value(entry.iconSize), document.GetAllocator());

            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    Result ListControlCache(bool* outValue, const Option&)
    {
        std::vector<ns::ApplicationControlCacheEntryInfo> list(4096);
        auto count = ns::ListApplicationControlCacheEntryInfo(list.data(), 4096);
        list.resize(static_cast<size_t>(count));

        auto buffer = GenerateControlCacheJson(list);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ShowControl(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        ns::ApplicationControlSource source;
        if (option.HasKey("--cache-only"))
        {
            source = ns::ApplicationControlSource::CacheOnly;
        }
        else if(option.HasKey("--storage-only"))
        {
            source = ns::ApplicationControlSource::StorageOnly;
        }
        else
        {
            source = ns::ApplicationControlSource::Storage;
        }

        auto buffer = std::unique_ptr<char[]>(new char[256 * 1024]);

        size_t size;
        NN_RESULT_DO(ns::GetApplicationControlData(&size, buffer.get(), 256 * 1024, source, appId));
        ns::ApplicationControlDataAccessor accessor(buffer.get(), size);
        DEVMENUCOMMAND_LOG("Size: %u KB\n", size / 1024);
        DEVMENUCOMMAND_LOG("Name: %s\n", accessor.GetProperty().GetDefaultTitle().name);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result WaitControl(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        os::TimerEvent timer(os::EventClearMode_ManualClear);
        auto timeoutMs = option.GetValue("--timeout");
        if (timeoutMs)
        {
            timer.StartOneShot(TimeSpan::FromMilliSeconds(std::strtoull(timeoutMs, nullptr, 10)));
        }

        auto buffer = std::unique_ptr<char[]>(new char[256 * 1024]);
        while (!timer.TryWait())
        {
            size_t size;
            if (ns::GetApplicationControlData(&size, buffer.get(), 256 * 1024, ns::ApplicationControlSource::CacheOnly, appId).IsSuccess())
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }

            DEVMENUCOMMAND_LOG("Waiting...\n");
            os::TimedWaitAny(TimeSpan::FromSeconds(1), timer.GetBase());
        }

        DEVMENUCOMMAND_LOG("Timed out %s ms\n", timeoutMs);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    rapidjson::StringBuffer GenerateRecordDetailJson(std::vector<std::pair<ns::ApplicationRecord, ns::ApplicationRecordProperty>> list)
    {
        rapidjson::Document document;
        document.SetArray();
        auto& allocator = document.GetAllocator();

        for (auto& item : list)
        {
            auto& record = item.first;
            auto& property = item.second;

            rapidjson::Value obj(rapidjson::kObjectType);
            obj.AddMember("id", rapidjson::Value(devmenuUtil::GetBit64String(record.id.value).data, allocator), allocator);
            obj.AddMember("lastEvent", rapidjson::StringRef(ToString(record.lastEvent)), allocator);
            obj.AddMember("isAutoDeleteDisabled", rapidjson::Value(ns::GetApplicationRecordAttribute(record, ns::ApplicationRecordAttribute::AutoDeleteDisabled)), allocator);
            obj.AddMember("isAutoUpdateEnabled", rapidjson::Value(ns::GetApplicationRecordAttribute(record, ns::ApplicationRecordAttribute::AutoUpdateEnabled)), allocator);
            obj.AddMember("lastUpdated", rapidjson::Value(record.lastUpdated), allocator);

            rapidjson::Value propertyObj(rapidjson::kObjectType);
            propertyObj.AddMember("terminateResult", rapidjson::Value(devmenuUtil::GetBit32String(property.terminateResult).data, allocator), allocator);

            if (ns::GetApplicationRecordAttribute(record, ns::ApplicationRecordAttribute::AutoUpdateEnabled))
            {
                rapidjson::Value autoUpdateInfoObj(rapidjson::kObjectType);
                autoUpdateInfoObj.AddMember("patchId", rapidjson::Value(devmenuUtil::GetBit64String(property.autoUpdateInfo.patchId.value).data, allocator), allocator);
                autoUpdateInfoObj.AddMember("currentVersion", rapidjson::Value(property.autoUpdateInfo.currentVersion), allocator);

                propertyObj.AddMember("autoUpdateInfo", autoUpdateInfoObj, allocator);
            }

            if (property.requestUpdateInfo.needsUpdate)
            {
                rapidjson::Value requestUpdateInfoObj(rapidjson::kObjectType);
                requestUpdateInfoObj.AddMember("reasonResult", rapidjson::Value(devmenuUtil::GetBit32String(property.requestUpdateInfo.reasonResultInnerValue).data, allocator), allocator);

                propertyObj.AddMember("requestUpdateInfo", requestUpdateInfoObj, allocator);
            }

            obj.AddMember("property", propertyObj, allocator);

            document.PushBack(obj, allocator);
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    Result ListRecordDetail(bool* outValue, const Option& option)
    {
        std::vector<std::pair<ns::ApplicationRecord, ns::ApplicationRecordProperty>> detailList;
        auto appId = devmenuUtil::GetApplicationIdTarget(option);
        auto recordList = ListApplicationRecord();
        for (auto& record : recordList)
        {
            if (appId == ncm::ApplicationId::GetInvalidId() || appId == record.id)
            {
                ns::ApplicationRecordProperty property;
                NN_RESULT_DO(ns::GetApplicationRecordProperty(&property, record.id));
                detailList.push_back(std::make_pair(record, property));
            }
        }

        DEVMENUCOMMAND_LOG("%s\n", GenerateRecordDetailJson(detailList).GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableAutoUpdate(bool* outValue, const Option& option)
    {
        NN_RESULT_DO(ns::EnableApplicationAutoUpdate(devmenuUtil::GetApplicationIdTarget(option)));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableAutoUpdate(bool* outValue, const Option& option)
    {
        NN_RESULT_DO(ns::DisableApplicationAutoUpdate(devmenuUtil::GetApplicationIdTarget(option)));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result IsUpdateRequested(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);
        util::optional<Result> updateReason;
        NN_RESULT_DO(ns::IsApplicationUpdateRequested(&updateReason, appId));
        if( updateReason )
        {
            DEVMENUCOMMAND_LOG("Update is requested : 0x%08x (%03u-%04u)\n", (*updateReason).GetInnerValueForDebug(), (*updateReason).GetModule(), (*updateReason).GetDescription());
        }
        else
        {
            DEVMENUCOMMAND_LOG("Update is not requested.\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RequestUpdate(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);
        if( !option.HasKey("--reason") )
        {
            DEVMENUCOMMAND_LOG("Use --reason option (hex value of a Result) to specify update request reason.");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto reasonValue = std::strtoul(option.GetValue("--reason"), NULL, 16);
        auto reasonResult = result::detail::ConstructResult(static_cast<uint32_t>(reasonValue));
        NN_RESULT_DO(ns::RequestApplicationUpdate(appId, reasonResult));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result WithdrawUpdateRequest(bool* outValue, const Option& option)
    {
        NN_RESULT_DO(ns::WithdrawApplicationUpdateRequest(devmenuUtil::GetApplicationIdTarget(option)));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    rapidjson::StringBuffer GenerateStorageContentMetaJson(const std::vector<ncm::StorageContentMetaKey>& list)
    {
        rapidjson::Document document;
        document.SetArray();

        for (auto& key : list)
        {
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("id", rapidjson::Value(devmenuUtil::GetBit64String(key.key.id).data, document.GetAllocator()), document.GetAllocator());
            item.AddMember("version", rapidjson::Value(key.key.version), document.GetAllocator());
            item.AddMember("type", rapidjson::StringRef(devmenuUtil::GetContentMetaTypeString(key.key.type)), document.GetAllocator());
            item.AddMember("storage", rapidjson::StringRef(devmenuUtil::GetStorageIdString(key.storageId)), document.GetAllocator());
            if (key.key.installType != ncm::ContentInstallType::Full)
            {
                item.AddMember("installType", rapidjson::StringRef(ToString(key.key.installType)), document.GetAllocator());
            }
            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    Result ListRecord(bool* outValue, const Option& option)
    {
        std::vector<ncm::StorageContentMetaKey> list(static_cast<size_t>(2048));
        int count;

        if (option.HasKey("--installed"))
        {
            NN_RESULT_DO(ns::ListApplicationRecordInstalledContentMeta(&count, list.data(), static_cast<int>(list.size()), devmenuUtil::GetApplicationIdTarget(option), 0));;
        }
        else
        {
            NN_RESULT_DO(ns::ListApplicationRecordContentMeta(&count, list.data(), static_cast<int>(list.size()), devmenuUtil::GetApplicationIdTarget(option), 0));;
        }
        list.resize(static_cast<int>(count));

        auto buffer = GenerateStorageContentMetaJson(list);
        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ListDownloadingContentMeta(bool* outValue, const Option& option)
    {
        std::vector<ncm::StorageContentMetaKey> list(static_cast<size_t>(2048));
        int count;

        NN_RESULT_DO(ns::ListApplicationDownloadingContentMeta(&count, list.data(), static_cast<int>(list.size()), devmenuUtil::GetApplicationIdTarget(option), 0));;
        list.resize(static_cast<int>(count));

        auto buffer = GenerateStorageContentMetaJson(list);
        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    rapidjson::Value CreateErrorContextJsonObject(const err::ErrorContext& context, rapidjson::Document* document)
    {
        rapidjson::Value object(rapidjson::kObjectType);
        object.AddMember("type", rapidjson::StringRef(ToString(context.type)), document->GetAllocator());
        if (context.type == err::ErrorContextType::Http)
        {
            object.AddMember("fqdn", rapidjson::Value(context.http.fqdn, document->GetAllocator()), document->GetAllocator());
            object.AddMember("ip", rapidjson::Value(context.http.ip, document->GetAllocator()), document->GetAllocator());
        }
        return object;
    }

    rapidjson::StringBuffer GenerateApplicationViewJson(const std::vector<ns::ApplicationView>& list)
    {
        rapidjson::Document document;
        document.SetArray();

        for (auto& view : list)
        {
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("id", rapidjson::Value(devmenuUtil::GetBit64String(view.id.value).data, document.GetAllocator()), document.GetAllocator());
            item.AddMember("version", rapidjson::Value(view.version), document.GetAllocator());
            item.AddMember("hasRecord", rapidjson::Value(view.HasRecord()), document.GetAllocator());
            item.AddMember("hasMainRecord", rapidjson::Value(view.HasMainRecord()), document.GetAllocator());
            item.AddMember("hasPatchRecord", rapidjson::Value(view.HasPatchRecord()), document.GetAllocator());
            item.AddMember("hasAddOnContentRecord", rapidjson::Value(view.HasAddOnContentRecord()), document.GetAllocator());
            item.AddMember("hasMainInstallRecord", rapidjson::Value(view.HasMainInstallRecord()), document.GetAllocator());
            item.AddMember("isDownloading", rapidjson::Value(view.IsDownloading()), document.GetAllocator());
            item.AddMember("isGameCard", rapidjson::Value(view.IsGameCard()), document.GetAllocator());
            item.AddMember("hasGameCardEntity", rapidjson::Value(view.HasGameCardEntity()), document.GetAllocator());
            item.AddMember("isLaunchable", rapidjson::Value(view.IsLaunchable()), document.GetAllocator());
            item.AddMember("hasAllEntity", rapidjson::Value(view.HasAllEntity()), document.GetAllocator());
            item.AddMember("hasMainEntity", rapidjson::Value(view.HasMainEntity()), document.GetAllocator());
            item.AddMember("hasAllAddOnContentEntity", rapidjson::Value(view.HasAllAddOnContentEntity()), document.GetAllocator());
            item.AddMember("hasAnyAddOnContentEntity", rapidjson::Value(view.HasAnyAddOnContentEntity()), document.GetAllocator());
            item.AddMember("hasPatchEntity", rapidjson::Value(view.HasPatchEntity()), document.GetAllocator());
            item.AddMember("maybeCorrupted", rapidjson::Value(view.MaybeCorrupted()), document.GetAllocator());
            item.AddMember("isWaitingCommit", rapidjson::Value(view.IsWaitingCommit()), document.GetAllocator());
            item.AddMember("isWaitingApplicationCommit", rapidjson::Value(view.IsWaitingApplicationCommit()), document.GetAllocator());
            item.AddMember("isWaitingAocCommit", rapidjson::Value(view.IsWaitingAocCommit()), document.GetAllocator());
            item.AddMember("isWaitingPatchInstall", rapidjson::Value(view.IsWaitingPatchInstall()), document.GetAllocator());
            item.AddMember("isAutoDeleteDisabled", rapidjson::Value(view.IsAutoDeleteDisabled()), document.GetAllocator());
            item.AddMember("isApplyingDelta", rapidjson::Value(view.IsApplyingDelta()), document.GetAllocator());
            item.AddMember("isCleanupAddOnContentWithNoRightsRecommended", rapidjson::Value(view.IsCleanupAddOnContentWithNoRightsRecommended()), document.GetAllocator());
            item.AddMember("isSystemUpdateRequiredToCommit", rapidjson::Value(view.IsSystemUpdateRequiredToCommit()), document.GetAllocator());
            item.AddMember("isPreInstalledApplication", rapidjson::Value(view.IsPreInstalledApplication()), document.GetAllocator());

            if (view.IsDownloading())
            {
                rapidjson::Value progress(rapidjson::kObjectType);
                progress.AddMember("downloaded", rapidjson::Value(view.progress.downloaded), document.GetAllocator());
                progress.AddMember("total", rapidjson::Value(view.progress.total), document.GetAllocator());
                progress.AddMember("lastResult", rapidjson::Value(devmenuUtil::GetBit32String(view.progress.lastResult.GetInnerValueForDebug()).data, document.GetAllocator()), document.GetAllocator());
                progress.AddMember("state", rapidjson::StringRef(ToString(view.progress.state)), document.GetAllocator());
                progress.AddMember("isRunning", rapidjson::Value(view.progress.isRunning), document.GetAllocator());
                progress.AddMember("remainingTime", rapidjson::Value(view.progress.remainingTime.GetSeconds()), document.GetAllocator());
                item.AddMember("progress", progress, document.GetAllocator());

                if (view.progress.lastResult.IsFailure() && (view.progress.state == ns::ApplicationDownloadState::Suspended || view.progress.state == ns::ApplicationDownloadState::Fatal))
                {
                    err::ErrorContext context;
                    NN_ABORT_UNLESS_RESULT_SUCCESS(ns::GetApplicationViewDownloadErrorContext(&context, view.id));
                    item.AddMember("errorContext", CreateErrorContextJsonObject(context, &document), document.GetAllocator());
                }
            }

            if (view.IsApplyingDelta())
            {
                rapidjson::Value applyProgress(rapidjson::kObjectType);
                applyProgress.AddMember("applied", rapidjson::Value(view.applyProgress.applied), document.GetAllocator());
                applyProgress.AddMember("total", rapidjson::Value(view.applyProgress.total), document.GetAllocator());
                applyProgress.AddMember("lastResult", rapidjson::Value(devmenuUtil::GetBit32String(view.applyProgress.lastResult.GetInnerValueForDebug()).data, document.GetAllocator()), document.GetAllocator());
                applyProgress.AddMember("state", rapidjson::StringRef(ToString(view.applyProgress.state)), document.GetAllocator());
                applyProgress.AddMember("remainingTime", rapidjson::Value(view.applyProgress.remainingTime.GetSeconds()), document.GetAllocator());
                item.AddMember("applyProgress", applyProgress, document.GetAllocator());
            }

            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    Result ListView(bool* outValue, const Option&)
    {
        DEVMENUCOMMAND_LOG("%s\n", GenerateApplicationViewJson(ListApplicationView()).GetString());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    rapidjson::StringBuffer GenerateNetworkInstallTaskInfoJson(const std::vector<nim::NetworkInstallTaskInfo>& list)
    {
        rapidjson::Document document;
        document.SetArray();

        for (const auto& info : list)
        {
            rapidjson::Value item(rapidjson::kObjectType);

            item.AddMember("ApplicationId", rapidjson::Value(devmenuUtil::GetBit64String(info.applicationId.value).data, document.GetAllocator()), document.GetAllocator());

            // nim の Attribute が ns の ApplicationDownloadState である
            // という知識をここで使ってしまっている
            // Attribute <-> ApplicationDownloadState の関係が変わったときには要修正
            item.AddMember("state", rapidjson::StringRef(ToString(static_cast<ns::ApplicationDownloadState>(info.attribute))), document.GetAllocator());

            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    rapidjson::StringBuffer GenerateApplyDeltaTaskInfoJson(const std::vector<nim::ApplyDeltaTaskInfo>& list)
    {
        rapidjson::Document document;
        document.SetArray();

        for (const auto& info : list)
        {
            rapidjson::Value item(rapidjson::kObjectType);

            item.AddMember("ApplicationId", rapidjson::Value(devmenuUtil::GetBit64String(info.applicationId.value).data, document.GetAllocator()), document.GetAllocator());

            item.AddMember("state", rapidjson::StringRef(ToString(info.progress.state)), document.GetAllocator());

            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    Result ListTask(bool* outValue, const Option&)
    {
        ns::RequestServerStopper stopper;
        ns::GetRequestServerStopper(&stopper);

        rapidjson::StringBuffer downloadTaskJson, applyDeltaTaskJson;
        {
            nim::NetworkInstallTaskId ids[nim::MaxNetworkInstallTaskCount];
            int count = nim::ListNetworkInstallTask(ids, nim::MaxNetworkInstallTaskCount);

            std::vector<nim::NetworkInstallTaskInfo> infos(static_cast<size_t>(count));
            for (int i = 0; i < count; ++i)
            {
                NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&(infos[i]), ids[i]));
            }
            downloadTaskJson = GenerateNetworkInstallTaskInfoJson(infos);
        }
        {
            nim::ApplyDeltaTaskId ids[nim::MaxApplyDeltaTaskCount];
            int count = nim::ListApplyDeltaTask(ids, nim::MaxApplyDeltaTaskCount);

            std::vector<nim::ApplyDeltaTaskInfo> infos(static_cast<size_t>(count));
            for (int i = 0; i < count; ++i)
            {
                NN_RESULT_DO(nim::GetApplyDeltaTaskInfo(&(infos[i]), ids[i]));
            }
            applyDeltaTaskJson = GenerateApplyDeltaTaskInfoJson(infos);
        }
        DEVMENUCOMMAND_LOG(
            "{\n"
            "\"DownloadTask\": %s,\n"
            "\"ApplyDeltaTask\": %s\n"
            "}\n",
            downloadTaskJson.GetString(),
            applyDeltaTaskJson.GetString()
        );

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ListContentMetaDatabase(bool* outValue, const Option& option)
    {
        auto storage = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : ToStorageId("builtin");
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, *storage));

        std::vector<ncm::ContentMetaKey> keyList(8192);
        auto listCount = db.ListContentMeta(keyList.data(), static_cast<int>(keyList.size()));
        keyList.resize(static_cast<size_t>(listCount.listed));

        rapidjson::Document document;
        document.SetArray();

        for (auto& key : keyList)
        {
            rapidjson::Value contentMeta(rapidjson::kObjectType);
            contentMeta.AddMember("id", rapidjson::Value(devmenuUtil::GetBit64String(key.id).data, document.GetAllocator()), document.GetAllocator());
            contentMeta.AddMember("version", rapidjson::Value(key.version), document.GetAllocator());
            contentMeta.AddMember("type", rapidjson::StringRef(devmenuUtil::GetContentMetaTypeString(key.type)), document.GetAllocator());

            rapidjson::Value contentInfoList(rapidjson::kArrayType);
            std::vector<ncm::ContentInfo> infoList(256);
            int count;
            NN_RESULT_DO(db.ListContentInfo(&count, infoList.data(), static_cast<int>(infoList.size()), key, 0));
            infoList.resize(static_cast<size_t>(count));
            for (auto& info : infoList)
            {
                rapidjson::Value contentInfo(rapidjson::kObjectType);
                contentInfo.AddMember("id", rapidjson::Value(ncm::GetContentIdString(info.GetId()).data, document.GetAllocator()), document.GetAllocator());
                contentInfo.AddMember("type", rapidjson::StringRef(ToString(info.GetType())), document.GetAllocator());
                contentInfo.AddMember("size", rapidjson::Value(info.GetSize()), document.GetAllocator());

                contentInfoList.PushBack(contentInfo, document.GetAllocator());
            }
            contentMeta.AddMember("contentInfoList", contentInfoList, document.GetAllocator());

            document.PushBack(contentMeta, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ListContentStorage(bool* outValue, const Option& option)
    {
        auto storageId = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : ToStorageId("builtin");
        if (!storageId)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, *storageId));

        std::vector<ncm::ContentId> list(8192);
        int count;
        NN_RESULT_DO(storage.ListContentId(&count, list.data(), static_cast<int>(list.size()), 0));
        list.resize(static_cast<size_t>(count));

        rapidjson::Document document;
        document.SetArray();

        for (auto& id : list)
        {
            document.PushBack(rapidjson::Value(ncm::GetContentIdString(id).data, document.GetAllocator()), document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    util::optional<ncm::StorageId> GetPlaceHolderCommandStorage(const Option& option)
    {
        auto storageId = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : ToStorageId("builtin");
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        if (!storageId && option.HasKey("-s"))
        {
            storageId = ToSystemStorage(option.GetValue("-s"));
        }
#endif
        return storageId;

    }

    Result ListPlaceHolder(bool* outValue, const Option& option)
    {
        auto storageId = GetPlaceHolderCommandStorage(option);
        if (!storageId)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, *storageId));

        std::vector<ncm::PlaceHolderId> list(8192);
        int count;
        NN_RESULT_DO(storage.ListPlaceHolder(&count, list.data(), static_cast<int>(list.size())));
        list.resize(static_cast<size_t>(count));

        rapidjson::Document document;
        document.SetArray();

        for (auto& id : list)
        {
            char uuidString[util::Uuid::StringSize];
            document.PushBack(rapidjson::Value(id.uuid.ToString(uuidString, sizeof(uuidString)), document.GetAllocator()), document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    ncm::ContentId GenerateRandomContentId()
    {
        std::random_device random;
        ncm::ContentId contentId;
        for (auto& bit : contentId.data)
        {
            bit = static_cast<Bit8>(random());
        }
        return contentId;
    }

    Result CreatePlaceHolder(bool* outValue, const Option& option)
    {
        auto storageId = GetPlaceHolderCommandStorage(option);
        if (!storageId)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, *storageId));

        auto sizeOption = option.GetValue("--size");
        int64_t size = sizeOption ? std::strtoll(sizeOption, nullptr, 10) : 1024;

        auto contentIdOption = option.GetValue("--content-id");
        ncm::ContentId contentId;
        if (contentIdOption)
        {
            auto id = ncm::GetContentIdFromString(contentIdOption, std::strlen(contentIdOption));
            if (!id)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            contentId = *id;
        }
        else
        {
            contentId = GenerateRandomContentId();
        }

        auto placeHolderId = storage.GeneratePlaceHolderId();
        NN_RESULT_DO(storage.CreatePlaceHolder(placeHolderId, contentId, size));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeletePlaceHoldersAllImpl(ncm::ContentStorage* pStorage, bool isTargetLockedErrorIgnored)
    {
        int holderCount = 0;
        const int MaxPlaceHolderCount = 256;
        std::unique_ptr<ncm::PlaceHolderId[]> placeHolders(new ncm::PlaceHolderId[MaxPlaceHolderCount]);
        NN_RESULT_DO(pStorage->ListPlaceHolder(&holderCount, placeHolders.get(), MaxPlaceHolderCount));

        if (holderCount == 0)
        {
            DEVMENUCOMMAND_LOG("Place holder is not found.\n");
            NN_RESULT_SUCCESS;
        }

        for (int i = 0; i < holderCount; i++)
        {
            char uuidString[util::Uuid::StringSize];
            placeHolders[i].uuid.ToString(uuidString, sizeof(uuidString));
            DEVMENUCOMMAND_LOG("Place Holder Id: %s\n", uuidString);
            NN_RESULT_TRY(pStorage->DeletePlaceHolder(placeHolders[i]))
                NN_RESULT_CATCH(fs::ResultTargetLocked)
                {
                    DEVMENUCOMMAND_LOG("Target place holder is locked\n");
                    if (!isTargetLockedErrorIgnored)
                    {
                        NN_RESULT_RETHROW;
                    }
                }
            NN_RESULT_END_TRY;
        }
        NN_RESULT_SUCCESS;
    }

    Result DeletePlaceHolderImpl(ncm::ContentStorage *pStorage, const char *inputPlaceHolderId)
    {
        auto placeHolderIdLength = util::Strnlen(inputPlaceHolderId, util::Uuid::StringSize);
        if (placeHolderIdLength != util::Uuid::StringSize - 1)
        {
            DEVMENUCOMMAND_LOG("Please input place holder id.\n");
            NN_RESULT_SUCCESS;
        }

        ncm::PlaceHolderId placeHolderId;
        placeHolderId.uuid.FromString(inputPlaceHolderId);

        bool hasPlaceHolder = false;
        NN_RESULT_DO(pStorage->HasPlaceHolder(&hasPlaceHolder, placeHolderId));
        if (!hasPlaceHolder)
        {
            DEVMENUCOMMAND_LOG("Place holder id %s is not found.\n", inputPlaceHolderId);
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(pStorage->DeletePlaceHolder(placeHolderId));
        NN_RESULT_SUCCESS;
    }

    Result DeletePlaceHolder(bool* outValue, const Option& option)
    {
        *outValue = false;
        auto storageId = GetPlaceHolderCommandStorage(option);
        if (!storageId)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, *storageId));

        if (option.HasKey("--all"))
        {
            auto isTargetLockedErrorIgnored = option.HasKey("--ignore-target-locked-error");
            NN_RESULT_DO(DeletePlaceHoldersAllImpl(&storage, isTargetLockedErrorIgnored));
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
        else
        {
            auto inputPlaceHolderId = option.GetTarget();
            NN_RESULT_DO(DeletePlaceHolderImpl(&storage, inputPlaceHolderId));
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
    }

    Result DeleteEntity(bool* outValue, const Option& option)
    {
        auto storage = option.HasKey("-s") ? ToInstallStorage(option.GetValue("-s")) : ToInstallStorage("auto");
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        ns::ApplicationEntityFlag flags;
        flags.Reset();

        if (option.HasKey("--app"))
        {
            flags.Set<ns::ApplicationEntityFlag_Application>();
        }
        if (option.HasKey("--aoc"))
        {
            flags.Set<ns::ApplicationEntityFlag_AddOnContent>();
        }
        if (option.HasKey("--patch"))
        {
            flags.Set<ns::ApplicationEntityFlag_Patch>();
        }
        if (!(flags & ns::ApplicationEntityFlag_All).IsAnyOn())
        {
            flags = ns::ApplicationEntityFlag_All;
        }

        if (option.HasKey("--ignore-running-check"))
        {
            flags = flags | ns::ApplicationEntityFlag_SkipRunningCheck::Mask;
        }

        auto appId = devmenuUtil::GetApplicationIdTarget(option);
        NN_RESULT_DO(ns::DeleteApplicationContentEntities(appId, flags, *storage));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Touch(bool* outValue, const Option& option)
    {
        NN_RESULT_DO(ns::TouchApplication(devmenuUtil::GetApplicationIdTarget(option)));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UpdateInfo(bool* outValue, const Option& option)
    {
        auto idString = option.GetTarget();
        ncm::ApplicationId appId = { std::strtoull(idString, NULL, 16) };

        DEVMENUCOMMAND_LOG("Waiting network request accepted...\n");
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network not available\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ns::AsyncApplicationUpdateInfo asyncUpdateInfo;
        NN_RESULT_DO(ns::RequestApplicationUpdateInfo(&asyncUpdateInfo, appId));

        ns::ApplicationUpdateInfo updateInfo;
        NN_RESULT_DO(Get(&updateInfo, &asyncUpdateInfo));
        if (updateInfo == ns::ApplicationUpdateInfo::UpToDate)
        {
            DEVMENUCOMMAND_LOG("UpToDate\n");
        }
        else if (updateInfo == ns::ApplicationUpdateInfo::Updatable)
        {
            DEVMENUCOMMAND_LOG("Updatable\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Update(bool* outValue, const Option& option)
    {
        auto idString = option.GetTarget();
        ncm::ApplicationId appId = { std::strtoull(idString, NULL, 16) };

        DEVMENUCOMMAND_LOG("Waiting network request accepted...\n");
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network not available\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ns::AsyncResult asyncUpdateResult;
        NN_RESULT_DO(ns::RequestUpdateApplication(&asyncUpdateResult, appId));
        NN_RESULT_DO(Get(&asyncUpdateResult));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableAutoUpdateSetting(bool* outValue, const Option&)
    {
        settings::system::SetAutoUpdateEnabled(true);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableAutoUpdateSetting(bool* outValue, const Option&)
    {
        settings::system::SetAutoUpdateEnabled(false);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Corrupt(bool* outValue, const Option& option)
    {
        auto storage = option.HasKey("-s") ? ToInstallStorage(option.GetValue("-s")) : ToInstallStorage("auto");
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        ncm::ApplicationId appId = { std::strtoull(option.GetTarget(), NULL, 16) };
        ns::CorruptContentFlag flags;
        flags.Reset();

        if (option.HasKey("--app"))
        {
            flags.Set<ns::CorruptContentFlag_Application>();
        }
        if (option.HasKey("--patch"))
        {
            flags.Set<ns::CorruptContentFlag_Patch>();
        }
        if (option.HasKey("--aoc"))
        {
            flags.Set<ns::CorruptContentFlag_AddOnContent>();
        }
        if (option.HasKey("--code"))
        {
            flags.Set<ns::CorruptContentFlag_Code>();
        }
        if (option.HasKey("--sdcard"))
        {
            flags.Set<ns::CorruptContentFlag_SdCardEncryption>();
        }

        if (!(flags & ns::CorruptContentFlag_All).IsAnyOn())
        {
            flags |= ns::CorruptContentFlag_All;
        }

        NN_RESULT_DO(ns::CorruptApplicationForDebug(appId, flags, *storage));
        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result NotificationInfo(bool* outValue, const Option&)
    {
        time::Initialize();
        NN_UTIL_SCOPE_EXIT{ time::Finalize(); };

        auto isSetupCompleted = ns::IsNotificationSetupCompleted();
        auto lastCount = ns::GetLastNotificationInfoCount();

        std::vector<ns::NotificationInfo> list(256);
        auto count = ns::ListLastNotificationInfo(list.data(), static_cast<int>(list.size()));
        list.resize(static_cast<size_t>(count));

        rapidjson::Document document(rapidjson::kObjectType);
        document.AddMember("isSetupCompleted", rapidjson::Value(isSetupCompleted), document.GetAllocator());
        document.AddMember("lastCount", rapidjson::Value(lastCount), document.GetAllocator());

        rapidjson::Value infoList(rapidjson::kArrayType);
        for (auto& info : list)
        {
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("type", rapidjson::StringRef(ToString(info.type)), document.GetAllocator());

            auto utc = time::ToCalendarTimeInUtc(info.time);
            item.AddMember("UTC", rapidjson::Value(devmenuUtil::GetCalendarTimeString(utc).data, document.GetAllocator()), document.GetAllocator());
            item.AddMember("eTag", rapidjson::Value(reinterpret_cast<char*>(info.eTag.data), document.GetAllocator()), document.GetAllocator());


            switch (info.type)
            {
            case ns::NotificationInfo::NotificationType::SystemUpdate:
                {
                    item.AddMember("titleId", rapidjson::Value(devmenuUtil::GetBit64String(info.nupInfo.titleId).data, document.GetAllocator()), document.GetAllocator());
                    item.AddMember("titleVersion", rapidjson::Value(info.nupInfo.titleVersion), document.GetAllocator());
                    item.AddMember("requiredTitleId", rapidjson::Value(devmenuUtil::GetBit64String(info.nupInfo.requiredTitleId).data, document.GetAllocator()), document.GetAllocator());
                    item.AddMember("requriedVersion", rapidjson::Value(info.nupInfo.requiredVersion), document.GetAllocator());
                }
                break;
            case ns::NotificationInfo::NotificationType::VersionList:
                {
                }
                break;
            case ns::NotificationInfo::NotificationType::DownloadTaskList:
                {
                }
                break;
            case ns::NotificationInfo::NotificationType::ETicketAvailable:
                {
                    // この値はクライアントで解釈しておらず、常に 0 が入るのでコメントアウトしておく
                    //item.AddMember("eciAccountId", rapidjson::Value(info.eTicketAvailableInfo.accountId), document.GetAllocator());
                    item.AddMember("titleIds", rapidjson::Value(devmenuUtil::GetBit64String(info.eTicketAvailableInfo.titleId).data, document.GetAllocator()), document.GetAllocator());
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
            infoList.PushBack(item, document.GetAllocator());
        }
        document.AddMember("infoList", infoList, document.GetAllocator());

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const char* GetNotificationTaskType(const ns::AsyncTask& task)
    {
        switch (task.info.type)
        {
        case ns::NotificationInfo::NotificationType::SystemUpdate:
            return "SystemUpdate";
        case ns::NotificationInfo::NotificationType::DownloadTaskList:
            return "DownloadTaskList";
        case ns::NotificationInfo::NotificationType::VersionList:
            return "VersionList";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    rapidjson::StringBuffer GenerateNotificationTaskList(const std::vector<ns::AsyncTask>& list)
    {
        rapidjson::Document document;
        document.SetArray();

        auto currentTick = os::GetSystemTick();
        for (auto& task : list)
        {
            auto elapsedTick = currentTick - os::Tick(task.registeredTime);

            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("type", rapidjson::Value(GetNotificationTaskType(task) , document.GetAllocator()), document.GetAllocator());
            item.AddMember("elapsed", rapidjson::Value(elapsedTick.ToTimeSpan().GetSeconds()), document.GetAllocator());
            item.AddMember("limit", rapidjson::Value(task.waitingLimit), document.GetAllocator());
            item.AddMember("registered", rapidjson::Value(task.registeredTime), document.GetAllocator());
            item.AddMember("updated", rapidjson::Value(task.lastUpdatedTime), document.GetAllocator());
            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }

    Result ListNotificationTask(bool* outValue, const Option&)
    {
        const int MaxTaskCount = static_cast<int>(ns::NotificationInfo::NotificationType::NumberOfNotificationType);
        std::vector<ns::AsyncTask> tasks(MaxTaskCount);

        auto taskCount = ns::ListNotificationTask(&tasks[0], MaxTaskCount);
        tasks.resize(taskCount);
        auto buffer = GenerateNotificationTaskList(tasks);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteRedundantEntity(bool* outValue, const Option&)
    {
        NN_RESULT_DO(ns::DeleteRedundantApplicationEntity());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result InstallRedundantEntity(bool* outValue, const Option& option)
    {
        const size_t InstallBufferSize = 16 * 1024 * 1024;
        auto storage = option.HasKey("-s") ? ToInstallStorage(option.GetValue("-s")) : ToInstallStorage("builtin");
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        auto nspPath = option.GetTarget();
        if (!devmenuUtil::IsAbsolutePath(nspPath))
        {
            DEVMENUCOMMAND_LOG("'%s' is not an absolute path.\n", nspPath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, nspPath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        std::unique_ptr<char[]> buffer(new char[InstallBufferSize]);
        ncm::SubmissionPackageInstallTask task;
        NN_RESULT_DO(task.Initialize(file, *storage, buffer.get(), InstallBufferSize));

        DEVMENUCOMMAND_LOG("Preparing to install %s...\n", nspPath);
        NN_RESULT_DO(task.Prepare());
        NN_RESULT_DO(devmenuUtil::WaitInstallTaskExecute(&task));
        NN_RESULT_DO(task.Commit());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearTerminateResult(bool* outValue, const Option& option)
    {
        auto appId = devmenuUtil::GetApplicationIdTarget(option);

        NN_RESULT_DO(nn::ns::ClearApplicationTerminateResult(appId));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result TryCommitCurrentApplicationDownloadTask(bool* outValue, const Option&)
    {
        NN_RESULT_DO(ns::TryCommitCurrentApplicationDownloadTask());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableAutoCommit(bool* outValue, const Option&)
    {
        ns::EnableAutoCommit();
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableAutoCommit(bool* outValue, const Option&)
    {
        ns::DisableAutoCommit();
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    Result Uninstall(bool* outValue, const Option& option)
    {
        ns::DeleteRedundantApplicationEntity();

        ns::ApplicationEntityFlag flags = ns::ApplicationEntityFlag_All;

        flags.Set<ns::ApplicationEntityFlag_SkipRunningCheck>(option.HasKey("--ignore-running-check"));

        if (option.HasKey("--all"))
        {
            // GameCard のアプリでも削除する
            flags.Set<ns::ApplicationEntityFlag_SkipGameCardCheck>();
            auto list = ListApplicationRecord();
            for (auto& record : list)
            {
                NN_RESULT_DO(ns::DeleteApplicationCompletelyForDebug(record.id, flags));
            }
        }
        else
        {
            flags.Set<ns::ApplicationEntityFlag_SkipGameCardCheck>(option.HasKey("--ignore-gamecard-check"));
            auto appId = devmenuUtil::GetApplicationIdTarget(option);
            NN_RESULT_DO(ns::DeleteApplicationCompletelyForDebug(appId, flags));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Launch(bool* outValue, const Option& option)
    {
        os::ProcessId processId;
        ncm::ApplicationId id = devmenuUtil::GetApplicationIdTarget(option);

        auto appStorage = option.HasKey("--application-storage") ? ToStorageId(option.GetValue("--application-storage")) : util::optional<ncm::StorageId>();
        auto patchStorage = option.HasKey("--patch-storage") ? ToStorageId(option.GetValue("--patch-storage")) : util::optional<ncm::StorageId>();
        auto appStorageId = appStorage ? *appStorage : ncm::StorageId::Any;
        auto patchStorageId = patchStorage ? *patchStorage : ncm::StorageId::Any;
        NN_RESULT_DO(devmenuUtil::VerifyApplicationAndPatchCombination(id, appStorageId, patchStorageId));

        NN_RESULT_TRY(ns::CheckApplicationLaunchVersion(id))
            NN_RESULT_CATCH(ns::ResultApplicationUpdateRequiredByRequiredVersion)
            {
                DEVMENUCOMMAND_LOG("Need to reset the required version of the application to launch.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultApplicationUpdateRequiredByRequiredApplicationVersion)
            {
                DEVMENUCOMMAND_LOG("Application required version of add-on content is not satisfied. Please install new application or patch.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultApplicationUpdateRecommended)
            {
                // 起動可能なので、launch コマンドでは何もしない
            }
            NN_RESULT_CATCH_ALL
            {
                *outValue = false;
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY;
        if (!appStorage && !patchStorage)
        {
            ns::ApplicationLaunchInfo info;
            NN_RESULT_DO(ns::GetApplicationLaunchInfo(&info, id));
            NN_RESULT_DO(ns::LaunchApplicationForDevelop(&processId, id, info.launchFlags));
        }
        else
        {
            const int32_t flags = ns::LaunchProgramFlags_NotifyStart | ns::LaunchProgramFlags_NotifyExit | ns::LaunchProgramFlags_NotifyDebug;
            NN_RESULT_DO(ns::LaunchApplicationWithStorageIdForDevelop(&processId, id, flags, appStorageId, patchStorageId));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    void ShowApplicationRightsUnavailableReason(const char* headerMessage, const ns::ApplicationRightsOnServer& rights) NN_NOEXCEPT
    {
        if (rights.HasUnavailableRights())
        {
            DEVMENUCOMMAND_LOG("[%s] Unavailable rights found.\n", headerMessage);
            if (rights.reason.NoRights())
            {
                DEVMENUCOMMAND_LOG("unavialable reason: There is no rights of the content.\n");
            }
            if (rights.reason.SystemUpdateRequired())
            {
                DEVMENUCOMMAND_LOG("unavialable reason: System update is required.\n");
            }
            if (rights.reason.HasDeviceLinkedRightsOnlyContent())
            {
                DEVMENUCOMMAND_LOG("unavialable reason: The content has only device linked rights and this device is not device linked one.\n");
            }
            if (rights.reason.NotReleased())
            {
                DEVMENUCOMMAND_LOG("unavialable reason: The content is not released.\n");
            }
            if (rights.reason.AssignableRightsLimitExceeded())
            {
                DEVMENUCOMMAND_LOG("unavialable reason: Assignable rights limit is exceeded.\n");
            }
        }
    }

    void ShowApplicationRightsOnServer(const ns::ApplicationRightsOnServer* pBegin, const ns::ApplicationRightsOnServer* pEnd) NN_NOEXCEPT
    {
        rapidjson::Document document;
        document.SetArray();

        std::for_each(pBegin, pEnd, [&document](const ns::ApplicationRightsOnServer& rights) NN_NOEXCEPT
        {
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("id", rapidjson::Value(devmenuUtil::GetBit64String(rights.id.value).data, document.GetAllocator()), document.GetAllocator());
            char uidString[16 * 2 + 2] = {};
            util::TSNPrintf(uidString, sizeof(uidString), "%016llx_%016llx", rights.uid._data[0], rights.uid._data[1]);
            item.AddMember("uid", rapidjson::Value(uidString, document.GetAllocator()), document.GetAllocator());
            item.AddMember("type", rapidjson::Value(ToString(rights.type), document.GetAllocator()), document.GetAllocator());
            item.AddMember("hasAvailableRights", rapidjson::Value(rights.HasAvailableRights()), document.GetAllocator());
            item.AddMember("hasUnavailableRights", rapidjson::Value(rights.HasUnavailableRights()), document.GetAllocator());
            item.AddMember("isAccountRestrictedRights", rapidjson::Value(rights.IsAccountRestrictedRights()), document.GetAllocator());
            item.AddMember("recommendSyncTicket", rapidjson::Value(rights.RecommendSyncTicket()), document.GetAllocator());
            item.AddMember("recommendAssignRights", rapidjson::Value(rights.RecommendAssignRights()), document.GetAllocator());
            item.AddMember("deviceLinkedPermanentLicense", rapidjson::Value(rights._flags.Test<ns::ApplicationRightsOnServerFlag_DeviceLinkedPermanentLicense>()), document.GetAllocator());
            item.AddMember("permanentLicense", rapidjson::Value(rights._flags.Test<ns::ApplicationRightsOnServerFlag_PermanentLicense>()), document.GetAllocator());
            item.AddMember("accountRestrictedPermanentLicense", rapidjson::Value(rights._flags.Test<ns::ApplicationRightsOnServerFlag_AccountRestrictivePermanentLicense>()), document.GetAllocator());
            item.AddMember("temporaryLicense", rapidjson::Value(rights._flags.Test<ns::ApplicationRightsOnServerFlag_TemporaryLicense>()), document.GetAllocator());
            document.PushBack(item, document.GetAllocator());
        });

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
    }

    Result CheckAndLaunchApplication(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        account::InitializeForAdministrator();

        // TORIAEZU: 問答無用で呼んでしまう
        ns::UnregisterAllUsersOfAccountRestrictiveRights();

        std::array<ns::ApplicationRightsOnClient, 2> rightsList;
        int count;
        NN_RESULT_DO(ns::GetApplicationRightsOnClient(&count, rightsList.data(), static_cast<int>(rightsList.size()), devmenuUtil::GetApplicationIdTarget(option)));
        auto hasUnavailableRights = std::any_of(rightsList.begin(), rightsList.begin() + count, [](const ns::ApplicationRightsOnClient& rights) NN_NOEXCEPT
        {
            return rights.HasUnavailableRights();
        });

        util::optional<account::Uid> uid;
        bool maybeDynamicRights = false;
        if (hasUnavailableRights)
        {
            NN_RESULT_DO(GetUidFromIndex(&uid, option.HasKey("--account-index") ? static_cast<int>(std::strtoul(option.GetValue("--account-index"), nullptr, 10)) : 0));
            if (!uid)
            {
                DEVMENUCOMMAND_LOG("Account not found.\n");
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_RESULT_DO(ns::GetApplicationRightsOnClient(&count, rightsList.data(), static_cast<int>(rightsList.size()), devmenuUtil::GetApplicationIdTarget(option), *uid));
                maybeDynamicRights = std::any_of(rightsList.begin(), rightsList.begin() + count, [](const ns::ApplicationRightsOnClient& rights) NN_NOEXCEPT
                {
                    return rights.RecommendInquireServer();
                });
            }
        }

        if (maybeDynamicRights)
        {
            std::array<ns::ApplicationRightsOnServer, 2> rightsOnServerList;
            int rightsCount;
            {
                ns::AsyncApplicationRightsOnServerList async;
                if (uid)
                {
                    NN_RESULT_DO(ns::RequestApplicationRightsOnServer(&async, devmenuUtil::GetApplicationIdTarget(option), *uid));
                }
                else
                {
                    NN_RESULT_DO(ns::RequestApplicationRightsOnServer(&async, devmenuUtil::GetApplicationIdTarget(option)));
                }
                NN_RESULT_DO(Get(&rightsCount, rightsOnServerList.data(), static_cast<int>(rightsOnServerList.size()), &async));
            }

            auto endList = rightsOnServerList.begin() + rightsCount;

            auto appRights = std::find_if(rightsOnServerList.begin(), endList, [](const ns::ApplicationRightsOnServer& rights) NN_NOEXCEPT
            {
                return rights.type == ns::ApplicationContentType::Application;
            });
            if (appRights == endList)
            {
                DEVMENUCOMMAND_LOG("Application rights on server not found.\n");
                NN_RESULT_SUCCESS;
            }
            auto aocRights = std::find_if(rightsOnServerList.begin(), endList, [](const ns::ApplicationRightsOnServer& rights) NN_NOEXCEPT
            {
                return rights.type == ns::ApplicationContentType::AddOnContent;
            });

            if (option.HasKey("--verbose"))
            {
                ShowApplicationRightsOnServer(&(*rightsOnServerList.begin()), &(*endList));
            }

            // アプリケーションの場合は有効な権利のみがある状態でないと起動できない
            if (!appRights->HasAvailableRights())
            {
                if (appRights->HasUnavailableRights())
                {
                    ShowApplicationRightsUnavailableReason("Application", *appRights);
                }
                else
                {
                    DEVMENUCOMMAND_LOG("Application content not found.\n");
                }

                NN_RESULT_SUCCESS;
            }
            if (aocRights != endList)
            {
                ShowApplicationRightsUnavailableReason("AddOnContent", *aocRights);
            }

            if (std::any_of(rightsOnServerList.begin(), endList, [](const ns::ApplicationRightsOnServer& rights) NN_NOEXCEPT { return rights.RecommendAssignRights(); }))
            {
                ns::AsyncResult asyncResult;
                NN_RESULT_DO(ns::RequestAssignRights(&asyncResult, rightsOnServerList.data(), rightsCount));
                NN_RESULT_DO(Get(&asyncResult));
            }
            if (std::any_of(rightsOnServerList.begin(), endList, [](const ns::ApplicationRightsOnServer& rights) NN_NOEXCEPT { return rights.RecommendSyncTicket(); }))
            {
                ec::system::AsyncProgressResult async;
                NN_RESULT_DO(ec::system::RequestSyncTicket(&async));
                NN_RESULT_DO(Get(&async));
            }

            // 再チェック
            if (uid)
            {
                NN_RESULT_DO(ns::GetApplicationRightsOnClient(&count, rightsList.data(), static_cast<int>(rightsList.size()), devmenuUtil::GetApplicationIdTarget(option), *uid));
            }
            else
            {
                NN_RESULT_DO(ns::GetApplicationRightsOnClient(&count, rightsList.data(), static_cast<int>(rightsList.size()), devmenuUtil::GetApplicationIdTarget(option)));
            }
            for (auto& rights : util::MakeSpan(rightsList.data(), count))
            {
                const char* msgHeader = rights.type == ns::ApplicationContentType::Application ? "[Error] " : "[Warning] ";
                if (rights.HasUnavailableRights())
                {
                    if (rights.HasPrepurchasedRights())
                    {
                        DEVMENUCOMMAND_LOG("%s Prepurchased rights is found.\n", msgHeader);
                    }
                    else
                    {
                        DEVMENUCOMMAND_LOG("%s There are contents that have no rights.\n", msgHeader);
                    }

                    if (rights.type == ns::ApplicationContentType::Application)
                    {
                        // アプリケーションの場合は起動させない
                        *outValue = false;
                        NN_RESULT_SUCCESS;
                    }
                }

            }

            if (std::any_of(rightsOnServerList.begin(), endList, [](const ns::ApplicationRightsOnServer& rights) NN_NOEXCEPT { return rights.IsAccountRestrictedRights(); }))
            {
                if (uid)
                {
                    NN_RESULT_DO(ns::RegisterUserOfAccountRestrictedRights(*uid));
                }
            }
        }

        NN_RESULT_DO(Launch(outValue, option));

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    Result GetTerminateResult(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        Result result;
        NN_RESULT_DO(ns::GetApplicationTerminateResult(&result, devmenuUtil::GetApplicationIdTarget(option)));
        DEVMENUCOMMAND_LOG("result = 0x%08x\n", result.GetInnerValueForDebug());
        NN_RESULT_SUCCESS;
    }
#endif // ~NN_TOOL_DEVMENUCOMMANDSYSTEM

    char ToChar(bool value)
    {
        return value ? '+' : '-';
    }

    Result GetApplicationControlProperty(ns::ApplicationControlProperty* outValue, ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        const size_t BufferSize = 128 * 1024;
        static uint8_t s_Buffer[BufferSize];

        size_t outSize = 0;
        NN_RESULT_DO(nn::ns::GetApplicationControlData(&outSize, s_Buffer, BufferSize, nn::ns::ApplicationControlSource::Storage, applicationId));

        ns::ApplicationControlDataAccessor accessor(s_Buffer, outSize);
        std::memcpy(outValue, &accessor.GetProperty(), sizeof(ns::ApplicationControlProperty));

        NN_RESULT_SUCCESS;
    }

    Result List(bool* outValue, const Option& option)
    {
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        auto listsDetail = option.HasKey("--detail");
#else
        NN_UNUSED(option);
        auto listsDetail = false;
#endif

        {
            std::unique_ptr<ns::ApplicationRecord[]> recordBuffer(new ns::ApplicationRecord[ListBufferSize]);
            auto recordList = recordBuffer.get();

            auto count = ns::ListApplicationRecord(recordList, ListBufferSize, 0);
            if (0 < count)
            {
                DEVMENUCOMMAND_LOG("%d application(s) found.\n", count);
                if (listsDetail)
                {
                    char flagLabel[ns::ApplicationViewFlagIndexEnd + 1] = "flag";
                    std::memset(flagLabel + std::strlen(flagLabel), ' ', sizeof(flagLabel) - std::strlen(flagLabel) - 1);
                    DEVMENUCOMMAND_LOG("id                          version  lastEvent         lastUpdated  %s    downloaded         total  downloadState  lastResult\n", flagLabel);
                    //                  0x0123456789abcdef  123456789012345  GameCardInserted   4294967295  (-) 4294967295KB  4294967295KB  Downloading    0x012345678
                }
                else
                {
                    DEVMENUCOMMAND_LOG("id                          version  name\n");
                    //                  0x0123456789abcdef  123456789012345  Name
                }

                for (int i = 0; i < count; i++)
                {
                    auto& record = recordList[i];
                    ns::ApplicationView view;
                    NN_RESULT_DO(ns::GetApplicationView(&view, &record.id, 1));

                    static ns::ApplicationControlProperty s_ApplicationControlProperty;
                    auto result = GetApplicationControlProperty(&s_ApplicationControlProperty, record.id);
                    const char* version = result.IsSuccess() ? s_ApplicationControlProperty.displayVersion : "";

                    if (listsDetail)
                    {
                        char flag[ns::ApplicationViewFlagIndexEnd + 1] = {};
                        for (int j = 0; j < ns::ApplicationViewFlagIndexEnd; j++)
                        {
                            flag[j] = ToChar(view.flag[j]);
                        }

                        DEVMENUCOMMAND_LOG("0x%016llx  %15s  %-16s   %10llu  %s  %10lluKB  %10lluKB  %s    0x%08x\n",
                                view.id, version, ToString(record.lastEvent), record.lastUpdated, flag,
                                view.progress.downloaded / 1024, view.progress.total / 1024,
                                ToString(view.progress.state), view.progress.lastResult.GetInnerValueForDebug());
                    }
                    else
                    {
                        const char *name = s_ApplicationControlProperty.GetDefaultTitle().name;
                        if (*name == '\0')
                        {
                            name = "Name is not set.";
                        }
                        DEVMENUCOMMAND_LOG("0x%016llx  %15s  %s\n", view.id, version, name);
                    }
                }
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    struct SpaceViewInfo
    {
        const double GigaBytes = 1024 * 1024 * 1024.0;
        const double MegaBytes = 1024 * 1024.0;
        const double KiloBytes = 1024.0;
        const double Bytes = 1.0;

        const char* GigaBytesView   = "GB";
        const char* MegaBytesView   = "MB";
        const char* KiloBytesView   = "KB";
        const char* BytesView       = "B";

        int64_t size;
        double unitSize;
        const char* unitString;

        explicit SpaceViewInfo(int64_t size) NN_NOEXCEPT
        {
            this->size = size;
            SetBytesView();
        }

        void SetGigaBytesView() NN_NOEXCEPT
        {
            unitSize = size / GigaBytes;
            unitString = GigaBytesView;
        }

        void SetMegaBytesView() NN_NOEXCEPT
        {
            unitSize = size / MegaBytes;
            unitString = MegaBytesView;
        }

        void SetKiloBytesView() NN_NOEXCEPT
        {
            unitSize = size / KiloBytes;
            unitString = KiloBytesView;
        }

        void SetBytesView() NN_NOEXCEPT
        {
            unitSize = static_cast<double>(size);
            unitString = BytesView;
        }

        void SetAutoView() NN_NOEXCEPT
        {
            if (size >= GigaBytes)
            {
                SetGigaBytesView();
            }
            else if (size >= MegaBytes)
            {
                SetMegaBytesView();
            }
            else if (size >= KiloBytes)
            {
                SetKiloBytesView();
            }
            else
            {
                SetBytesView();
            }
        }

        void Output() NN_NOEXCEPT
        {
            if (unitSize < 10.0)
            {
                DEVMENUCOMMAND_LOG("%.1f %s", unitSize, unitString);
            }
            else
            {
                DEVMENUCOMMAND_LOG("%lld %s", static_cast<int64_t>(unitSize), unitString);
            }
        }
    };

    void ShowSpaceSize(const char* header, int64_t size, const char* viewOption) NN_NOEXCEPT
    {
        SpaceViewInfo info(size);
        if (viewOption)
        {
            switch (std::tolower(*viewOption))
            {
            case 'g':
                info.SetGigaBytesView();
                break;
            case 'm':
                info.SetMegaBytesView();
                break;
            case 'k':
                info.SetKiloBytesView();
                break;
            case 'b':
                info.SetBytesView();
                break;
            case 'a':
                info.SetAutoView();
                break;
            default:
                info.SetBytesView();
                break;
            }
        }

        DEVMENUCOMMAND_LOG("%s\t", header);
        info.Output();
        DEVMENUCOMMAND_LOG("\n");
    }

    ncm::StorageId GetStorageId(const char* storageString)
    {
        if ( std::string(storageString) == "card" )
        {
            return ncm::StorageId::GameCard;
        }
        if ( std::string(storageString) == "sdcard" )
        {
            return ncm::StorageId::SdCard;
        }
        if ( std::string(storageString) == "builtin" )
        {
            return ncm::StorageId::BuildInUser;
        }

        return ncm::StorageId::None;
    }

    const char* GetStorageString(ncm::StorageId id)
    {
        switch ( id )
        {
        case ncm::StorageId::None : return "None";
        case ncm::StorageId::Host : return "Host";
        case ncm::StorageId::Card : return "Card";
        case ncm::StorageId::BuildInUser : return "Built-in";
        case ncm::StorageId::BuildInSystem : return "Built-in System";
        case ncm::StorageId::SdCard : return "SD Card";
        default : return "";
        }
    }

    Result StorageSize(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        int64_t freeSpaceSize;
        int64_t totalSpaceSize;

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        bool force = option.HasKey("--force");
#else
        bool force = false;
#endif // NN_TOOL_DEVMENUCOMMANDSYSTEM

        auto storageString = option.GetTarget();
        if ( std::string(storageString) == "" )
        {
            DEVMENUCOMMAND_LOG("You must specify a storage. Set one of [sdcard|builtin].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto storageId = GetStorageId( storageString );
        if ( storageId == ncm::StorageId::None )
        {
            DEVMENUCOMMAND_LOG("%s is a invalid storage. Set one of [sdcard|builtin].\n", storageString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (!force)
        {
            NN_RESULT_DO(nn::ns::GetFreeSpaceSize(&freeSpaceSize, storageId));
            NN_RESULT_DO(nn::ns::GetTotalSpaceSize(&totalSpaceSize, storageId));
        }
        else
        {
            NN_RESULT_DO(nn::ns::GetStorageSize(&totalSpaceSize, &freeSpaceSize, storageId));
        }

        ShowSpaceSize("Free space", freeSpaceSize, option.GetValue("-b"));
        ShowSpaceSize("Total space", totalSpaceSize, option.GetValue("-b"));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result OccupiedSize(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        auto target = option.GetTarget();
        if (std::strlen(target) == 0)
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " application occupied-size <application_id>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ns::ApplicationOccupiedSize occupiedSize;
        NN_RESULT_DO(ns::CalculateApplicationOccupiedSize( &occupiedSize, devmenuUtil::GetApplicationIdTarget(option)));

        for (int i = 0; i < ns::ApplicationOccupiedSize::MaxStorageCount; i++)
        {
            auto entity = occupiedSize.storage[i];

            auto storageId = entity.storageId;
            if ( storageId == ncm::StorageId::None || storageId == ncm::StorageId::Host || storageId == ncm::StorageId::BuildInSystem )
            {
                continue;
            }

            DEVMENUCOMMAND_LOG("----------------------------------------------------------\n");
            DEVMENUCOMMAND_LOG("[%s]\n", GetStorageString(storageId));
            ShowSpaceSize("Application   ", entity.appSize, option.GetValue("-b"));
            ShowSpaceSize("Patch         ", entity.patchSize, option.GetValue("-b"));
            ShowSpaceSize("Add-On Content", entity.aocSize, option.GetValue("-b"));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result InvalidateCache(bool* outValue, const Option& option)
    {
        auto targetCache = option.GetTarget();

        if(std::strcmp(targetCache, "application-control") == 0)
        {
            nn::ns::InvalidateAllApplicationControlCache();
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
        else
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " application invalidate-cache application-control\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
    }

    Result Verify(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        ns::AsyncVerifyApplicationResult asyncResult;
        *outValue = false;
        auto target = option.GetTarget();
        if (std::strlen(target) == 0)
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " application verify <application_id>\n");
            NN_RESULT_SUCCESS;
        }

#if defined(NN_BUILD_CONFIG_OS_WIN)
        auto buffer = _aligned_malloc(VerifyWorkBufferSize, VerifyWorkBufferAlign);
#else
        auto buffer = std::aligned_alloc(VerifyWorkBufferAlign, VerifyWorkBufferSize);
#endif
        NN_UTIL_SCOPE_EXIT
        {
            std::free(buffer);
        };

        if (option.HasKey("--app") || option.HasKey("--patch") || option.HasKey("--aoc"))
        {
            ns::VerifyContentFlag flags;
            flags.Reset();

            if (option.HasKey("--app"))
            {
                flags.Set<ns::VerifyContentFlag_Application>();
            }
            if (option.HasKey("--patch"))
            {
                flags.Set<ns::VerifyContentFlag_Patch>();
            }
            if (option.HasKey("--aoc"))
            {
                flags.Set<ns::VerifyContentFlag_AddOnContent>();
            }

            NN_RESULT_DO(nn::ns::RequestVerifyApplication(&asyncResult, devmenuUtil::GetApplicationIdTarget(option), flags, buffer, VerifyWorkBufferSize));
        }
        else
        {
            NN_RESULT_DO(nn::ns::RequestVerifyApplication(&asyncResult, devmenuUtil::GetApplicationIdTarget(option), buffer, VerifyWorkBufferSize));
        }

        for(;;)
        {
            ns::AsyncVerifyApplicationProgress progress;
            asyncResult.GetProgress(&progress);
            DEVMENUCOMMAND_LOG("Current Progress: %d %%\n", static_cast<int>(progress.currentSize * 100.0f / progress.totalSize));
            if (asyncResult.TryWait())
            {
                auto result = asyncResult.Get();
                if (result.IsSuccess())
                {
                    DEVMENUCOMMAND_LOG("No corrupted contents.\n");
                    *outValue = true;
                    NN_RESULT_SUCCESS;
                }
                else
                {
                    DEVMENUCOMMAND_LOG("The application is corrupted.\n");
                    DEVMENUCOMMAND_LOG("Result: module = %d, description = %d\n", result.GetModule(), result.GetDescription());
                    auto detailResult = asyncResult.GetDetailResult();
                    DEVMENUCOMMAND_LOG("DetailResult: module = %d, description = %d\n", detailResult.GetModule(), detailResult.GetDescription());
                    NN_RESULT_THROW(result);
                }
            }

#ifdef NN_BUILD_CONFIG_OS_HORIZON
            nn::idle::ReportUserIsActive();
#endif
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result GetDesiredLanguage (bool* outValue, const Option& option)
    {
        auto applicationId = devmenuUtil::GetApplicationIdTarget(option);
        ns::ApplicationControlProperty property;
        NN_RESULT_DO(GetApplicationControlProperty(&property, applicationId));
        auto code = ns::GetApplicationSupportedLanguage(property.supportedLanguageFlag);
        auto languageString = devmenuUtil::language::ConvertLanguageCodeToString(code);

        DEVMENUCOMMAND_LOG("Language: %s\n", languageString);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ForceApplicationPrepurchased(bool* outValue, const Option& option)
    {
        es::RightsIdIncludingKeyId rightsId = devmenuUtil::GetRightsIdTarget(option);

        es::TicketInfo info;
        int infoCount = es::GetTicketInfo(&info, 1, &rightsId, 1);

        if (infoCount == 0)
        {
            DEVMENUCOMMAND_LOG("RightsId Not Found.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        es::DeleteTicket(&rightsId, 1);

        es::PrepurchaseRecord record = { info.rightsId, info.ticketId, info.accountId };
        NN_RESULT_DO(es::ImportPrepurchaseRecord(record));

        *outValue = true;
        NN_RESULT_SUCCESS;

    }

    Result DownloadApplicationPrepurchasedRights(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ns::AsyncResult async;
        NN_RESULT_DO(ns::RequestDownloadApplicationPrepurchasedRights(&async, appId));

        NN_RESULT_TRY(async.Get())
            NN_RESULT_CATCH(ns::ResultPrepurchasedContentStillUnavailable)
            {
                DEVMENUCOMMAND_LOG("ContentStillUnavailable\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultPrepurchasedContentPartiallyStillUnavailable)
            {
                DEVMENUCOMMAND_LOG("ContentPartiallyStillUnavailable\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultPrepurchasedRightsLost)
            {
                DEVMENUCOMMAND_LOG("ContentRightsLost\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(ns::ResultTryLater)
            {
                DEVMENUCOMMAND_LOG("TryLater\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        DEVMENUCOMMAND_LOG("Available\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result TicketInfo(bool* outValue, const Option& option)
    {
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);

        ns::ApplicationTicketInfo info;
        NN_RESULT_DO(ns::GetApplicationTicketInfo(&info, appId));

        switch (info.status)
        {
        case ns::ApplicationTicketInfo::Status::Owned:
            DEVMENUCOMMAND_LOG("Owned\n");
            break;
        case ns::ApplicationTicketInfo::Status::PrepurchasedAndMaybeReleased:
            DEVMENUCOMMAND_LOG("PrepurchasedAndMaybeReleased\n");
            break;
        case ns::ApplicationTicketInfo::Status::PrepurchasedButMaybeNotReleased:
            DEVMENUCOMMAND_LOG("PrepurchasedButMaybeNotReleased\n");
            break;
        case ns::ApplicationTicketInfo::Status::NotOwned:
            DEVMENUCOMMAND_LOG("NotOwned\n");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetPreInstalledApplication(bool* outValue, const Option& option)
    {
        *outValue = false;
        NN_RESULT_DO(ns::SetPreInstalledApplication(devmenuUtil::GetApplicationIdTarget(option)));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearPreInstalledApplication(bool* outValue, const Option& option)
    {
        *outValue = false;
        NN_RESULT_DO(ns::ClearPreInstalledApplicationFlag(devmenuUtil::GetApplicationIdTarget(option)));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CopyIdentifier(bool* outValue, const Option& option)
    {
        auto appId = devmenuUtil::GetApplicationIdTarget(option);

        auto storageString = option.GetValue("--storage");
        if (!storageString)
        {
            DEVMENUCOMMAND_LOG("You must specify --storage option. Set one of [card|builtin|sdcard].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto storageId = GetStorageId(storageString);
        if ( storageId == ncm::StorageId::None )
        {
            DEVMENUCOMMAND_LOG("%s is a invalid storage. Set one of [card|builtin|sdcard].\n", storageString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if(storageId == ncm::StorageId::GameCard)
        {
            ns::GameCardApplicationCopyIdentifier identifier;
            NN_RESULT_DO(ns::GetGameCardApplicationCopyIdentifier(&identifier, appId));
            DEVMENUCOMMAND_LOG("GameCardApplicationCopyIdentifier.deviceId %u bytes\n", sizeof(identifier.deviceId));
            devmenuUtil::PrintHexDump(identifier.deviceId, sizeof(identifier.deviceId));
            DEVMENUCOMMAND_LOG("GameCardApplicationCopyIdentifier.certificate %u bytes\n", sizeof(identifier.certificate));
            devmenuUtil::PrintHexDump(identifier.certificate, sizeof(identifier.certificate));
        }
        else
        {
            ns::InstalledApplicationCopyIdentifier identifier;
            NN_RESULT_DO(ns::GetInstalledApplicationCopyIdentifier(&identifier, appId, storageId));
            DEVMENUCOMMAND_LOG("InstalledApplicationCopyIdentifier.ticketId %llud\n", identifier.ticketId);
            DEVMENUCOMMAND_LOG("InstalledApplicationCopyIdentifier.ticket %lld bytes\n", identifier.ticketSize);
            devmenuUtil::PrintHexDump(identifier.ticket, static_cast<size_t>(identifier.ticketSize));
            DEVMENUCOMMAND_LOG("InstalledApplicationCopyIdentifier.key %lld bytes\n", sizeof(identifier.key));
            devmenuUtil::PrintHexDump(identifier.key, sizeof(identifier.key));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " application install <absolute_nsp_path> [--force] [-s <sdcard|builtin|auto>] [--skip-patch-combination-check]\n"
        "       " DEVMENUCOMMAND_NAME " application install <absolute_directory_path> --all [--force] [-s <sdcard|builtin|auto>] [--skip-patch-combination-check]\n"
        "       " DEVMENUCOMMAND_NAME " application install-nxargs  <absolute_nxargs_path> --application-id <application_id> [--force]\n"
        "       " DEVMENUCOMMAND_NAME " application uninstall-nxargs <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application list\n"
        "       " DEVMENUCOMMAND_NAME " application launch <application_id> [--application-storage <card|sdcard|builtin>] [--patch-storage <card|sdcard|builtin>]\n"
        "       " DEVMENUCOMMAND_NAME " application uninstall <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application uninstall --all\n"
        "       " DEVMENUCOMMAND_NAME " application storage-size <sdcard|builtin> [-b <g, m, k, b, a>]\n"
        "       " DEVMENUCOMMAND_NAME " application occupied-size <application_id> [-b <g, m, k, b, a>]\n"
        "       " DEVMENUCOMMAND_NAME " application verify <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application reset-required-version <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application reset-required-version --all\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " application install <absolute_nsp_path> [--force] [-s <sdcard|builtin|auto>] [--detach-sdcard-after <milliseconds>]\n"
        "       " DEVMENUCOMMAND_NAME " application download <application_id> [--version <version>] \n"
        "       " DEVMENUCOMMAND_NAME " application download-control <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application wait-download <application_id> [--timeout <ms>] [--download--progress <downloaded>/<total>] [--version <version>]\n"
        "       " DEVMENUCOMMAND_NAME " application wait-download-without-commit <application_id> [--timeout <ms>]\n"
        "       " DEVMENUCOMMAND_NAME " application wait-download-all [--timeout]\n"
        "       " DEVMENUCOMMAND_NAME " application cancel-download <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application resume-download <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application set-task-state <application_id> --state <Suspended|NotEnoughSpace> \n"
        "       " DEVMENUCOMMAND_NAME " application create-download-task <application_id> [--type <type> --id <id>] [--version <version>]\n"
        "       " DEVMENUCOMMAND_NAME " application push-download-task-list <absolute_download_task_list_path>\n"
        "       " DEVMENUCOMMAND_NAME " application request-download-task-list\n"
        "       " DEVMENUCOMMAND_NAME " application request-ensure-download-task\n"
        "       " DEVMENUCOMMAND_NAME " application request-download-task-list-data\n"
        "       " DEVMENUCOMMAND_NAME " application list-download-task-status\n"
        "       " DEVMENUCOMMAND_NAME " application clear-task-status-list\n"
        "       " DEVMENUCOMMAND_NAME " application list-record <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application list-record-detail [<application_id>]\n"
        "       " DEVMENUCOMMAND_NAME " application list-downloading-content-meta <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application list-view\n"
        "       " DEVMENUCOMMAND_NAME " application list-task\n"
        "       " DEVMENUCOMMAND_NAME " application list-content-meta-database [-s <card|builtin|sdcard>]\n"
        "       " DEVMENUCOMMAND_NAME " application list-content-storage [-s <card|builtin|sdcard>]\n"
        "       " DEVMENUCOMMAND_NAME " application list-place-holder [-s <card|builtin|sdcard>]\n"
        "       " DEVMENUCOMMAND_NAME " application list [--detail]\n"
        "       " DEVMENUCOMMAND_NAME " application check-launch-rights <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application check-resume-rights <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application update-version-list <absolute_version_list_path> [--no-perform-update]\n"
        "       " DEVMENUCOMMAND_NAME " application dump-required-version\n"
        "       " DEVMENUCOMMAND_NAME " application set-required-version <application_id> -v <required_version>\n"
        "       " DEVMENUCOMMAND_NAME " application check-launch-version <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application request-version-list\n"
        "       " DEVMENUCOMMAND_NAME " application request-version-list-data\n"
        "       " DEVMENUCOMMAND_NAME " application perform-auto-update\n"
        "       " DEVMENUCOMMAND_NAME " application version-list\n"
        "       " DEVMENUCOMMAND_NAME " application content-meta-status <application_id> [--check-rights]\n"
        "       " DEVMENUCOMMAND_NAME " application content-meta-status --all [--check-rights]\n"
        "       " DEVMENUCOMMAND_NAME " application list-control-cache\n"
        "       " DEVMENUCOMMAND_NAME " application show-control <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application wait-control <application_id> [--timeout <ms>]\n"
        "       " DEVMENUCOMMAND_NAME " application show-record-property <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application enable-auto-update <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application disable-auto-update <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application is-update-requested <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application request-update <application_id> --reason <result_value>\n"
        "       " DEVMENUCOMMAND_NAME " application withdraw-update-request <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application delete-place-holder <placeholder_id> [--all] [--ignore-target-locked-error] [-s <sdcard|builtin|auto>]\n"
        "       " DEVMENUCOMMAND_NAME " application delete-entity <application_id> [--app] [--aoc] [--patch] [-s <sdcard|builtin|auto>]\n"
        "       " DEVMENUCOMMAND_NAME " application touch <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application update-info <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application update <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application enable-auto-update-setting\n"
        "       " DEVMENUCOMMAND_NAME " application disable-auto-update-setting\n"
        "       " DEVMENUCOMMAND_NAME " application corrupt <application_id> [--app] [--aoc] [--patch] [--code] [--sdcard] [-s <sdcard|builtin|auto>]\n"
        "       " DEVMENUCOMMAND_NAME " application notification-info\n"
        "       " DEVMENUCOMMAND_NAME " application list-notification-task\n"
        "       " DEVMENUCOMMAND_NAME " application delete-redundant-entity\n"
        "       " DEVMENUCOMMAND_NAME " application install-redundant-entity\n"
        "       " DEVMENUCOMMAND_NAME " application clear-terminate-result\n"
        "       " DEVMENUCOMMAND_NAME " application get-launch-required-version <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application upgrade-required-version <application_id> -v <required_version>\n"
        "       " DEVMENUCOMMAND_NAME " application try-commit-current-application-download-task\n"
        "       " DEVMENUCOMMAND_NAME " application enable-auto-commit\n"
        "       " DEVMENUCOMMAND_NAME " application disable-auto-commit\n"
        "       " DEVMENUCOMMAND_NAME " application get-desired-language <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application force-application-prepurchased <rights_id>\n"
        "       " DEVMENUCOMMAND_NAME " application download-application-prepurchased-rights <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application ticket-info <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application set-preinstalled-application <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application clear-preinstalled-application <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application check-and-launch <application_id> [--account-index <user_account_index>] [--verbose]\n"
        "       " DEVMENUCOMMAND_NAME " application get-terminate-result <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " application get-application-rights-on-client <application_id> [--account-index <user_account_index>]\n"
        "       " DEVMENUCOMMAND_NAME " application copy-identifier <application_id> --storage <card|sdcard|builtin>\n"
#endif
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    const SubCommand g_SubCommands[] =
    {
        { "install", Install },
        { "install-nxargs", InstallNxArgs },
        { "uninstall-nxargs", UninstallNxArgs },
        { "launch", Launch },
        { "list", List },
        { "uninstall", Uninstall },
        { "invalidate-cache", InvalidateCache },
        { "storage-size", StorageSize },
        { "occupied-size", OccupiedSize },
        { "verify", Verify },
        { "reset-required-version", ResetRequiredVersion },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "download", Download },
        { "download-control", DownloadControl },
        { "wait-download", WaitDownload },
        { "wait-download-without-commit", WaitDownloadWithoutCommit },
        { "wait-download-all", WaitDownloadAll },
        { "cancel-download", CancelDownload },
        { "resume-download", ResumeDownload },
        { "set-task-state", SetTaskState },
        { "create-download-task", CreateDownloadTask },
        { "push-download-task-list", PushDownloadTaskList },
        { "request-download-task-list", RequestDownloadTaskList },
        { "request-ensure-download-task", RequestEnsureDownloadTask },
        { "request-download-task-list-data", RequestDownloadTaskListData },
        { "list-download-task-status", ListDownloadTaskStatus },
        { "clear-task-status-list", ClearTaskStatusList },
        { "check-launch-rights", CheckLaunchRights },
        { "check-resume-rights", CheckResumeRights },
        { "update-version-list", UpdateVersionList },
        { "dump-required-version", DumpRequiredVersion },
        { "set-required-version", SetRequiredVersion },
        { "check-launch-version", CheckLaunchVersion },
        { "request-version-list", RequestVersionList },
        { "request-version-list-data", RequestVersionListData },
        { "version-list", VersionList },
        { "content-meta-status", ContentMetaStatus },
        { "list-control-cache", ListControlCache },
        { "show-control", ShowControl },
        { "wait-control", WaitControl },
        { "enable-auto-update", EnableAutoUpdate },
        { "disable-auto-update", DisableAutoUpdate },
        { "is-update-requested", IsUpdateRequested },
        { "request-update", RequestUpdate },
        { "withdraw-update-request", WithdrawUpdateRequest },
        { "list-record", ListRecord },
        { "list-record-detail", ListRecordDetail },
        { "list-downloading-content-meta", ListDownloadingContentMeta },
        { "list-view", ListView },
        { "list-task", ListTask },
        { "list-content-meta-database", ListContentMetaDatabase },
        { "list-content-storage", ListContentStorage },
        { "list-place-holder", ListPlaceHolder },
        { "create-place-holder", CreatePlaceHolder },
        { "delete-place-holder", DeletePlaceHolder },
        { "delete-entity", DeleteEntity },
        { "touch", Touch },
        { "update-info", UpdateInfo },
        { "update", Update },
        { "enable-auto-update-setting", EnableAutoUpdateSetting },
        { "disable-auto-update-setting", DisableAutoUpdateSetting },
        { "corrupt", Corrupt },
        { "notification-info", NotificationInfo },
        { "list-notification-task", ListNotificationTask },
        { "delete-redundant-entity", DeleteRedundantEntity },
        { "install-redundant-entity", InstallRedundantEntity },
        { "clear-terminate-result", ClearTerminateResult },
        { "get-required-version", GetRequiredVersionCommand },
        { "upgrade-required-version", UpgradeRequiredVersionCommand },
        { "try-commit-current-application-download-task", TryCommitCurrentApplicationDownloadTask },
        { "enable-auto-commit", EnableAutoCommit },
        { "disable-auto-commit", DisableAutoCommit },
        { "get-desired-language", GetDesiredLanguage },
        { "force-application-prepurchased", ForceApplicationPrepurchased },
        { "download-application-prepurchased-rights", DownloadApplicationPrepurchasedRights },
        { "perform-auto-update", PerformAutoUpdate },
        { "ticket-info", TicketInfo },
        { "set-preinstalled-application", SetPreInstalledApplication },
        { "clear-preinstalled-application", ClearPreInstalledApplication },
        { "check-and-launch", CheckAndLaunchApplication },
        { "get-terminate-result", GetTerminateResult },
        { "get-application-rights-on-client", GetApplicationRightsOnClient },
        { "copy-identifier", CopyIdentifier },
#endif
    };
}

namespace {

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    const auto IsDevMenuCommandSystem = true;
#else
    const auto IsDevMenuCommandSystem = false;
#endif

void ShowHelpMessage() NN_NOEXCEPT
{
    DEVMENUCOMMAND_LOG(HelpMessage);
#if !(defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM))
    // TORIAEZU: DevMenu 系では関連オブジェクトのリンクがされないため、一旦使用しない。
    for (auto&& e : nn::devmenucommand::application::GetApplicationSubCommands())
    {
        if (nn::devmenucommand::NeedsHelpMessage(e, IsDevMenuCommandSystem))
        {
            DEVMENUCOMMAND_LOG("       " DEVMENUCOMMAND_NAME " application %s %s\n", e.name.c_str(), e.argumentHelpMessage.c_str());
        }
    }
#endif
}

}

Result ApplicationCommand(bool* outValue, const Option& option)
{
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
    NN_UNUSED(HelpMessage);
#endif
    if (!option.HasSubCommand())
    {
        ShowHelpMessage();
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        ShowHelpMessage();
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // TORIAEZU : ns::InitializeForSystemUpdate() を呼び出しただけでは接続に成功して待たないため、
    // 動作上無意味な API を読んで初期化が終わるまで待つようにする
    ns::GetBackgroundNetworkUpdateState();

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }
#if !(defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM))
    // TORIAEZU: DevMenu 系では関連オブジェクトのリンクがされないため、一旦使用しない。
    for (auto&& e : nn::devmenucommand::application::GetApplicationSubCommands())
    {
        if (e.name == option.GetSubCommand())
        {
            if (nn::devmenucommand::IsAvailable(e, IsDevMenuCommandSystem))
            {
                return e.function(outValue, option);
            }
        }
    }
#endif
    NN_UNUSED(IsDevMenuCommandSystem);

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu application command. See '" DEVMENUCOMMAND_NAME " application --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
