﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs_Base.h>
#include "CalWriter_IFile.h"
#include <memory>

class NnFsFile : public IFile
{
public:
    virtual ~NnFsFile() {}

    virtual nn::Result OpenRead(const char *path);
    virtual nn::Result OpenWrite(const char *path);
    virtual void Close();
    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size, bool flush);
    virtual nn::Result Read(size_t* pOut, int64_t offset, void* buffer, size_t size);
    virtual nn::Result GetSize(int64_t* pOut);
    virtual nn::Result Flush();
    virtual bool IsValid();
    static nn::Result Exists(bool *pOut, const char *path);
    static nn::Result Create(const char *path);
private:
    nn::fs::FileHandle m_Handle;
};

nn::Result OpenNnFsFile(std::shared_ptr<IFile> *pOut, const char *path, int mode)
{
    *pOut = std::shared_ptr<IFile>(new NnFsFile());

    if (mode & nn::fs::OpenMode_Write)
    {
        NN_RESULT_DO((*pOut)->OpenWrite(path));
    }
    else
    {
        NN_RESULT_DO((*pOut)->OpenRead(path));
    }

    NN_RESULT_SUCCESS;
}

nn::Result NnFsFile::OpenRead(const char *path)
{
    NN_RESULT_DO(
        nn::fs::OpenFile(&m_Handle, path, nn::fs::OpenMode_Read));

    NN_RESULT_SUCCESS;
}

nn::Result NnFsFile::OpenWrite(const char *path)
{
    NN_RESULT_DO(
        nn::fs::OpenFile(&m_Handle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

    NN_RESULT_SUCCESS;
}

void NnFsFile::Close()
{
    nn::fs::CloseFile(m_Handle);
}

nn::Result NnFsFile::Write(int64_t offset, const void* buffer, size_t size, bool flush)
{
    nn::fs::WriteOption option = {};
    if (flush)
    {
        option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush);
    }

    NN_RESULT_DO(
        nn::fs::WriteFile(m_Handle, offset, buffer, size, option));

    NN_RESULT_SUCCESS;
}

nn::Result NnFsFile::Read(size_t* pOut, int64_t offset, void* buffer, size_t size)
{
    NN_RESULT_DO(
        nn::fs::ReadFile(pOut, m_Handle, offset, buffer, size));

    NN_RESULT_SUCCESS;
}

nn::Result NnFsFile::GetSize(int64_t* pOut)
{
    NN_RESULT_DO(
        nn::fs::GetFileSize(pOut, m_Handle));

    NN_RESULT_SUCCESS;
}

nn::Result NnFsFile::Flush()
{
    NN_RESULT_DO(
        nn::fs::FlushFile(m_Handle));
    NN_RESULT_SUCCESS;
}


bool NnFsFile::IsValid()
{
    return m_Handle.handle != 0;
}

nn::Result NnFsFile::Exists(bool *pOut, const char *name)
{
    nn::fs::FileHandle handle;

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, name, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            *pOut = false;

            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    *pOut = true;

    NN_RESULT_SUCCESS;
}

nn::Result NnFsFile::Create(const char *path)
{
    NN_RESULT_DO(
        nn::fs::CreateFile(path, 0));

    NN_RESULT_SUCCESS;
}
