﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <curl/curl.h>
#include <cstdlib>
#include <string>
#include <vector>
#include <unordered_map>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/os.h>
#include <nn/init.h>

#include <nn/apm/apm_Lib.h>
#include <nn/idle/idle_SystemApi.h>

#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ApplyDeltaTask.h>

#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_ApplicationContentMetaApi.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SdCardSystemApi.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_InitializationApi.h>

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "../DevMenuCommand/DevMenuCommand_Common.h"
#include "../DevMenuCommand/DevMenuCommand_InstallUtil.h"

using namespace nn;

namespace {

const size_t HeapByteSize = 20 * 1024 * 1024;
const size_t MallocBytesSize = 20 * 1024 * 1024;

const size_t CacheBufferSize = 1 * 1024 * 1024;
uint8_t s_CacheBuffer[CacheBufferSize];

nn::Result SelfUninstall()
{
    nn::Bit64 id = NN_TOOL_APPLICATIONINSTALLER_PROGRAM_ID;
    nn::ncm::ContentMetaDatabase db;
    nn::ncm::ContentStorage storage;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, nn::ncm::StorageId::BuildInSystem));
    NN_RESULT_DO(nn::ncm::OpenContentStorage(&storage, nn::ncm::StorageId::BuildInSystem));
    nn::ncm::ContentManagerAccessor accessor(&db, &storage);
    NN_RESULT_DO(accessor.DeleteAll(id));
    NN_RESULT_DO(db.Commit());

    NN_RESULT_SUCCESS;
}


nn::Result InstallFromRomFs(ncm::StorageContentMetaKey* outInstalledKey, int* outCount, int maxOutCount, bool* outValue, ncm::ContentMetaType type)
{
    auto storage = ncm::StorageId::BuildInUser;
    auto onlyPush = false;
    auto nspPath = "rom:/";

    NN_RESULT_DO(
        nn::fs::MountRom("rom", s_CacheBuffer, sizeof(s_CacheBuffer)));
    NN_UTIL_SCOPE_EXIT {
        nn::fs::Unmount("rom");
    };

    NN_RESULT_DO(nn::ns::DeleteRedundantApplicationEntity());

    bool forceUpdate = true;
    bool skipInvalidateCache = false;
    int64_t detachTime = 0;

    {
        nn::fs::DirectoryEntryType entryType;
        NN_RESULT_DO(nn::fs::GetEntryType(&entryType, nspPath));
        if (entryType != nn::fs::DirectoryEntryType_Directory)
        {
            NN_LOG("%s is not directory path. --all option must used with directory.\n", nspPath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int readCount = 0;
        nn::fs::DirectoryHandle dir;
        NN_RESULT_DO(nn::fs::OpenDirectory(&dir, nspPath, nn::fs::OpenDirectoryMode::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT{
            nn::fs::CloseDirectory(dir);
        };


        while (NN_STATIC_CONDITION(true))
        {
            int64_t count;
            nn::fs::DirectoryEntry entry;
            NN_RESULT_DO(nn::fs::ReadDirectory(&count, &entry, dir, 1));
            if (count == 0)
            {
                break;
            }

            auto pathString = std::string(nspPath) + "/" + entry.name;
            auto path = pathString.c_str();
            if (entry.directoryEntryType != nn::fs::DirectoryEntryType_File)
            {
                NN_LOG("Skip %s since it is not file.\n", path);
                continue;
            }
            if (!devmenuUtil::IsNspFile(path))
            {
                NN_LOG("Skip %s since it is not .nsp file.\n", path);
                continue;
            }

            int readCountOne;
            bool success;
            NN_RESULT_DO(devmenuUtil::InstallCommonOne(&success, &outInstalledKey[readCount], &readCountOne, maxOutCount - readCount, path, storage, type, onlyPush, forceUpdate, skipInvalidateCache, detachTime));
            if (!success)
            {
                NN_LOG("Aborted since failed to install %s.\n", path);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        *outCount = readCount;
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

}

extern "C" void nninitStartup()
{
    auto result = nn::os::SetMemoryHeapSize( HeapByteSize );
    if (result.IsFailure())
    {
        NN_LOG("Unable to allocate heap memory (%zu bytes).\nPlease restart your target and try again.\n", HeapByteSize);
        NN_ABORT("");
    }

    uintptr_t address;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::AllocateMemoryBlock( &address, MallocBytesSize ) );
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MallocBytesSize );
}

extern "C" void nnMain()
{
    fs::InitializeWithMultiSessionForTargetTool();
    fs::SetEnabledAutoAbort(false);
    ncm::Initialize();
    ns::Initialize();

    bool outValue;
    ncm::StorageContentMetaKey key;
    int outCount;

    auto result = InstallFromRomFs(&key, &outCount, 1, &outValue, ncm::ContentMetaType::Unknown);

    if (result.IsSuccess())
    {
        NN_LOG("[SUCCESS]\n");
    }
    else
    {
        NN_LOG("[FAILURE]: result 0x%08x\n", result.GetInnerValueForDebug());
        NN_ABORT();
    }

    result = SelfUninstall();

    if (result.IsSuccess())
    {
        NN_LOG("Succeeded to uninstall self.\n");
    }
    else
    {
        NN_LOG("Failed to uninstall self.: result 0x%08x\n", result.GetInnerValueForDebug());
        NN_ABORT();
    }

    bool needsBooting = false;
    settings::fwdbg::SetSettingsItemValue("boot", "boot_devmenuapp_installer", &needsBooting, sizeof(needsBooting));
}
