﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include "wlan_VieManager.h"

namespace nn { namespace wlan {

VieManager::VieManager() NN_NOEXCEPT
{
    m_totalLengh = 0;
    m_pIeHead = NULL;
    m_pIeTail = NULL;
    m_ieCount = 0;
}

VieManager::~VieManager() NN_NOEXCEPT
{
    // 保有IEの全削除
}

bool VieManager::CreateIe(uint32_t* pOutIndex, WlanIeContainer* ie) NN_NOEXCEPT
{
    // flagやbodyの正当性はimplにて確認済みとする

    NN_SDK_REQUIRES_NOT_NULL(pOutIndex);
    NN_SDK_REQUIRES_NOT_NULL(ie);
    if( m_totalLengh + ie->length > VieLengthMax )
    {
        NN_SDK_ASSERT(false, "Total IE length exceeds VieLengthMax\n");
    }

    WlanIeContainer* pIe = new WlanIeContainer;
    NN_SDK_ASSERT_NOT_NULL(pIe);

    pIe->flag = ie->flag;
    pIe->length = ie->length;
    std::memcpy(pIe->oui, ie->oui, 3);
    std::memcpy(pIe->body, ie->body, ie->length - 3);

    // 管理情報更新
    m_ieCount++;
    m_totalLengh += ie->length;


    // 初回追加の場合先頭に追加
    if( m_pIeHead == NULL )
    {
        m_pIeHead = pIe;
        m_pIeTail = pIe;
        pIe->prev = NULL;
        pIe->next = NULL;
        pIe->index = 1;
        *pOutIndex = 1;
        return true;
    }

    // indexを割り振る
    // 空き番のindexを探し出してその最小値を割り振る
    // index値でソート済みなので、連番になっていない部分だけを探せばよい
    WlanIeContainer* tmp;
    uint32_t emptyIdx = 0;
    for( tmp = m_pIeHead; tmp != NULL; tmp = tmp->next )
    {
        if( tmp->index - emptyIdx > 1 )
        {
            break;
        }
        emptyIdx = tmp->index;
    }
    emptyIdx++;
    WLAN_LOG_DEBUG("New Vie Index is %d\n", emptyIdx);
    pIe->index = emptyIdx;
    *pOutIndex = emptyIdx;

    // 抜け番地にIEを追加
    // リストの最後に追加の場合
    if( pIe->index > m_pIeTail->index )
    {
        m_pIeTail->next = pIe;
        pIe->prev = m_pIeTail;
        pIe->next = m_pIeHead;
        m_pIeTail = pIe;
    }
    else
    {
        // tmp->prevとtmpの間
        tmp->prev->next = pIe;
        tmp->prev = pIe;
        pIe->prev = tmp->prev;
        pIe->next = tmp;
    }

    if( nn::wlan::WlanLogLevel >= nn::wlan::LogLevel_Debug )
    {
        DumpList();
    }
    return true;
}

bool VieManager::RemoveIe(uint32_t index) NN_NOEXCEPT
{
    // 該当するIEを探す
    WlanIeContainer* tmp = NULL;
    GetWlanIeContainer(&tmp, index);
    if( tmp == NULL )
    {
        WLAN_LOG_ERROR("[WLAN]There is no IE which has given index.\n");
        return false;
    }

    // 管理情報の更新
    m_ieCount--;
    m_totalLengh -= tmp->length;

    // 最後の1個だった場合
    if( m_ieCount == 0 )
    {
        m_pIeHead = NULL;
        m_pIeTail = NULL;
        m_totalLengh = 0;
    }
    else
    {
        // リストの付け替え
        // 該当IEが先頭または最後の場合、その情報も更新しておく。
        if( tmp == m_pIeHead )
        {
            m_pIeHead = tmp->next;
        }
        else if( tmp == m_pIeTail )
        {
            m_pIeTail = tmp->prev;
        }
        tmp->prev->next = tmp->next;
        tmp->next->prev = tmp->prev;
    }

    // 該当IEを削除
    delete tmp;

    if( nn::wlan::WlanLogLevel >= nn::wlan::LogLevel_Debug )
    {
        DumpList();
    }

    return true;
}

void VieManager::GetWlanIeContainer(WlanIeContainer** pOut, uint32_t index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_ieCount != 0, "There is no IE which has given index.\n" );

    // 該当するindex値をもつIEを探す
    WlanIeContainer* tmp;
    for( tmp = m_pIeHead; ; tmp = tmp->next )
    {
        if( tmp->index == index )
        {
            break;
        }

        if( tmp == m_pIeTail )
        {
            // 該当するindex値が見つからなかった
            WLAN_LOG_ERROR("[WLAN]There is no IE which has given index.\n");
            *pOut = NULL;
            return;
        }
    }

    *pOut = tmp;
}

void VieManager::DumpList() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[WLAN]Dump VendorIe List ----\n");

    WLAN_LOG_DEBUG("*** IE containers ***\n");
    WlanIeContainer* tmp = m_pIeHead;
    for( uint32_t i = 0; i < m_ieCount; i++ )
    {
        WLAN_LOG_DEBUG("[%d] index : %d length : %d\n", i, tmp->index, tmp->length);
        tmp = tmp->next;
    }
}

}}

