﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "wlan_RxEntries.h"
#include "wlan_Util.h"
#include "driver/wlan_Driver.h"

namespace nn { namespace wlan {


RxEntries::RxEntries() NN_NOEXCEPT
{
    nn::os::InitializeMutex(&m_Mutex, false, 0);
    m_LastRxId = 0;
    m_bMute = false;
}

RxEntries::~RxEntries() NN_NOEXCEPT
{
    RemoveAll();
    nn::os::FinalizeMutex(&m_Mutex);
}


Result RxEntries::Add(uint32_t* pOutRxId, uint32_t capacity, const uint16_t types[], uint32_t count, bool bOverWrite) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRxId);

    // capacity の上下限チェック
    NN_SDK_REQUIRES_RANGE(capacity, 1, RxCapacityMax + 1);

    // Ethertype数の上下限チェック
    NN_SDK_REQUIRES_RANGE(count, 1, EthertypeCountMax + 1);

    // Ethertypeの範囲チェック
    for( uint32_t i = 0; i < count; i++ )
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(types[i], 0x0600);
    }

    nn::os::LockMutex(&m_Mutex);

    // RxEntriesの数が超過していないか
    if (m_RingBuffers.size() >= RxEntriesCountMax)
    {
        WLAN_LOG_ERROR("Number of Registered RxEntries is full (=%d).\n", m_RingBuffers.size());
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryCountIsFull();
    }

    // 重複登録をチェック
    const RingBufferMap::iterator itEnd = m_RingBuffers.end();
    for( uint32_t i = 0; i < count; ++i )
    {
        if( m_RingBuffers.find(types[i]) != itEnd )
        {
            WLAN_LOG_ERROR("Protocol 0x%04x is already registered.\n", types[i]);
            nn::os::UnlockMutex(&m_Mutex);
            return ResultDuplicateRxEntry();
        }
    }

    // リングバッファの生成
    RingBuffer* pRingBuffer = new RingBuffer(capacity, bOverWrite);
    NN_SDK_ASSERT_NOT_NULL(pRingBuffer);

    // リングバッファをEthertype と関連付ける
    // 複数のEthertype が選ばれていてもリングバッファは RX エントリ一つにつき一つ
    for( uint32_t i = 0; i < count; ++i )
    {
        std::pair<RingBufferMap::iterator, bool> ret;
        ret = m_RingBuffers.insert(RingBufferPair(types[i], pRingBuffer));
        NN_SDK_ASSERT(ret.second == true);
        WLAN_LOG_DEBUG("Added 0x%04x\n", types[i]);
    }

    // リングバッファの ID を作って返す
    *pOutRxId = ++m_LastRxId;
    std::pair<EntryMap::iterator, bool> ret;
    ret = m_Entries.insert(EntryPair(*pOutRxId, pRingBuffer));
    NN_SDK_ASSERT(ret.second == true);
    WLAN_LOG_DEBUG("Created RxId 0x%08x\n", *pOutRxId);

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntries::Remove(uint32_t rxId) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    // 削除すべきリングバッファを得る
    RingBuffer* pRingBuffer = FindRingBufferByRxId(rxId);
    if ( pRingBuffer == nullptr )
    {
        WLAN_LOG_ERROR("rxid(%d) is not found.\n", rxId);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    // Ethertype との関連付けを削除
    RingBufferMap::iterator it;
    for( it = m_RingBuffers.begin();
        it != m_RingBuffers.end(); )
    {
        if (it->second == pRingBuffer)
        {
            WLAN_LOG_DEBUG("Delete protocol 0x%04x\n", it->first);
            m_RingBuffers.erase(it++);
        }
        else
        {
            ++it;
        }
    }

    // RX エントリを削除
    m_Entries.erase(rxId);

    // 新たに受信フレームが届くことはないはずなのでここでキャンセル
    pRingBuffer->Cancel();

    // リングバッファを削除
    if( m_bMute != true )
    {
        WLAN_LOG_DEBUG("RingBuffer may be referenced.\n");
    }
    delete pRingBuffer;

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntries::RemoveType(uint32_t* pOutRxId, uint16_t type) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    // Protocol IDの範囲チェック
    NN_SDK_ASSERT(type >= 0x0600);

    RingBufferMap::iterator it = m_RingBuffers.find(type);
    if ( it == m_RingBuffers.end() )
    {
        WLAN_LOG_ERROR("protocol 0x%04x is not registered.\n", type);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    nn::Result result = FindRxIdFromRingBuffer(pOutRxId, it->second);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);

    m_RingBuffers.erase(it);

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntries::AddType(uint32_t rxId, uint16_t type) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    // Ethertypeの範囲チェック
    NN_SDK_ASSERT(type >= 0x0600);

    // 重複登録をチェック
    const RingBufferMap::iterator itEnd = m_RingBuffers.end();
    if( m_RingBuffers.find(type) != itEnd )
    {
        WLAN_LOG_ERROR("Protocol 0x%04x is already registered.\n", type);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultDuplicateRxEntry();
    }

    // Protocol ID数の上限チェック
    if( m_RingBuffers.size() >= EthertypeCountMax )
    {
        WLAN_LOG_ERROR("The number of Protocol ID must be (%d) or less.\n", EthertypeCountMax);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryCountIsFull();
    }

    RingBuffer* pRingBuffer = FindRingBufferByRxId(rxId);
    if ( pRingBuffer == nullptr )
    {
        WLAN_LOG_ERROR("rxid(%d) is not found.\n", rxId);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    std::pair<RingBufferMap::iterator, bool> ret;
    ret = m_RingBuffers.insert(RingBufferPair(type, pRingBuffer));
    NN_SDK_ASSERT(ret.second == true);

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntries::AddMatchingInfo(uint32_t rxId, const ReceivedDataMatchInfo& pMatchDataInfo) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    RingBuffer* pRingBuffer = FindRingBufferByRxId(rxId);
    if ( pRingBuffer == nullptr )
    {
        WLAN_LOG_ERROR("rxid(%d) is not found.\n", rxId);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    if( pRingBuffer->SetMatchInfo(pMatchDataInfo) != true )
    {
        nn::os::UnlockMutex(&m_Mutex);
        return ResultDataMatchInfoError();
    }

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntries::RemoveMatchInfo(uint32_t rxId, const ReceivedDataMatchInfo& pMatchDataInfo) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    RingBuffer* pRingBuffer = FindRingBufferByRxId(rxId);
    if ( pRingBuffer == nullptr )
    {
        WLAN_LOG_ERROR("rxid(%d) is not found.\n", rxId);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    pRingBuffer->RemoveMatchInfo(pMatchDataInfo);

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

bool RxEntries::IsMatchDataInfoRegistered(uint32_t rxId) NN_NOEXCEPT
{
    bool ret = false;
    nn::os::LockMutex(&m_Mutex);

    RingBuffer* pRingBuffer = FindRingBufferByRxId(rxId);
    if ( pRingBuffer == nullptr )
    {
        WLAN_LOG_ERROR("rxid is not found.\n");
        nn::os::UnlockMutex(&m_Mutex);
        return false;
    }

    ret = pRingBuffer->IsMatchDataInfoRegistered();

    nn::os::UnlockMutex(&m_Mutex);

    return ret;
}


void RxEntries::RemoveAll() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    EntryMap::iterator it;
    EntryMap::iterator itNext;
    for(it = m_Entries.begin();
        it != m_Entries.end();)
    {
        itNext = it;
        ++itNext;
        Remove(it->first);
        it = itNext;
    }

    nn::os::UnlockMutex(&m_Mutex);
}

Result RxEntries::PushBuffer(nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pMbuf);

    nn::os::LockMutex(&m_Mutex);

    if( m_bMute == true )
    {
        WLAN_LOG_INFO("RxEntries is muted.\n");
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryIsMuted();
    }

    // 受信バッファからEthertypeを取り出す。
    // 受信データはDA SA PIDというDIX形式なので、13オクテット目からPIDが入っている
    nn::Bit8 *pData = reinterpret_cast<nn::Bit8*>(nn::mbuf::MbufTod(pMbuf));
    NN_SDK_ASSERT_NOT_NULL(pData);
    nn::Bit16 *pEthertype = reinterpret_cast<nn::Bit16*>(&pData[12]);
    uint16_t ethertype= ReverseEndian16(static_cast<uint16_t>(*pEthertype));

    WLAN_LOG_DEBUG("Ethertype is %04x @ %s\n", ethertype, __FUNCTION__);

    RingBuffer* pRingBuffer = FindRingBufferByType(ethertype);
    if ( pRingBuffer == nullptr )
    {
        WLAN_LOG_DEBUG("Ethertype 0x%04x is not registered.\n", ethertype);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    // リングバッファに強制的にエンキューする
    // あふれたバッファは内部で自動的に破棄される
    pRingBuffer->ForceEnqueue(pMbuf);

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntries::PushBuffer(uint32_t rxId, nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    if( m_bMute == true )
    {
        WLAN_LOG_DEBUG("RxEntries is muted.\n");
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryIsMuted();
    }

    RingBuffer* pRingBuffer = FindRingBufferByRxId(rxId);
    if ( pRingBuffer == nullptr )
    {
        WLAN_LOG_ERROR("Cannot find rxid(%d) RxEntry.\n", rxId);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    // リングバッファに強制的にエンキューする
    // あふれたバッファは内部で自動的に破棄される
    pRingBuffer->ForceEnqueue(pMbuf);

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntries::PullBuffer(nn::mbuf::Mbuf** pOutMbuf, uint32_t rxId) NN_NOEXCEPT
{
    RingBuffer* pRingBuffer = nullptr;
    {
        nn::os::LockMutex(&m_Mutex);

        if( m_bMute == true)
        {
            WLAN_LOG_INFO("RxEntries is muted.\n");
            nn::os::UnlockMutex(&m_Mutex);
            return ResultRxEntryIsMuted();
        }

        pRingBuffer = FindRingBufferByRxId(rxId);
        if ( pRingBuffer == nullptr )
        {
            WLAN_LOG_ERROR("Cannot find rxid(%d) RxEntry.\n", rxId);
            nn::os::UnlockMutex(&m_Mutex);
            return ResultRxEntryNotFound();
        }
        WLAN_LOG_DEBUG("RxEntries: Found ringbuffer for rxid[%d]\n", rxId);
        nn::os::UnlockMutex(&m_Mutex);
    }

    *pOutMbuf = pRingBuffer->Dequeue();

    if( *pOutMbuf == NULL )
    {
        // Cancelによる特殊データが入っていた
        WLAN_LOG_DEBUG("[RxEntries]Cancel data was pulled out.\n");
        return ResultGetFrameCancelled();
    }

    return nn::ResultSuccess();
}

Result RxEntries::PullBufferEx(nn::mbuf::Mbuf** pOutMbuf, uint32_t rxId, bool block) NN_NOEXCEPT
{
    *pOutMbuf = nullptr;

    RingBuffer* pRingBuffer = nullptr;
    {
        nn::os::LockMutex(&m_Mutex);

        if( m_bMute == true && block == false )
        {
            WLAN_LOG_INFO("RxEntries is muted.\n");
            nn::os::UnlockMutex(&m_Mutex);
            return ResultRxEntryIsMuted();
        }

        pRingBuffer = FindRingBufferByRxId(rxId);
        if ( pRingBuffer == nullptr )
        {
            WLAN_LOG_ERROR("Cannot find rxid(%d) RxEntry.\n", rxId);
            nn::os::UnlockMutex(&m_Mutex);
            return ResultRxEntryNotFound();
        }
        WLAN_LOG_DEBUG("RxEntries: Found ringbuffer for rxid[%d]\n", rxId);
        nn::os::UnlockMutex(&m_Mutex);
    }

    if (block)
    {
        *pOutMbuf = pRingBuffer->Dequeue();
    }
    else
    {
        bool ret;
        ret = pRingBuffer->TryDequeue(pOutMbuf);
        if( ret == false )
        {
            // 受信データが空
            return ResultNoRxData();
        }
    }

    if( *pOutMbuf == NULL )
    {
        // Cancelによる特殊データが入っていた
        WLAN_LOG_DEBUG("[RxEntries]Cancel data was pulled out.\n");
        return ResultGetFrameCancelled();
    }

    return nn::ResultSuccess();
}

void RxEntries::SetMute(bool bMute) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    if( bMute == m_bMute )
    {
        nn::os::UnlockMutex(&m_Mutex);
        return;
    }

    m_bMute = bMute;
    if( m_bMute )
    {
        Cancel();
    }
    else
    {
        Cleanup();
    }

    nn::os::UnlockMutex(&m_Mutex);
}

Result RxEntries::Cancel(uint32_t rxId) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    RingBuffer* pRingBuffer = FindRingBufferByRxId(rxId);
    if( pRingBuffer == nullptr )
    {
        return ResultRxEntryNotFound();
    }
    pRingBuffer->Cancel();

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

void RxEntries::Cancel() NN_NOEXCEPT
{
    EntryMap::iterator it;
    for( it = m_Entries.begin(); it != m_Entries.end(); ++it )
    {
        RingBuffer* pRingBuffer = it->second;
        // すれちがい用のRingBufferの場合キャンセルはしない
        if( pRingBuffer->IsForDetect() == false )
        {
            pRingBuffer->Cancel();
        }
        else
        {
            WLAN_LOG_DEBUG("This RingBuffer is for Detect. Not canceled.\n");
        }
    }
}

void RxEntries::Cleanup() NN_NOEXCEPT
{
    EntryMap::iterator it;
    for(it = m_Entries.begin();
        it != m_Entries.end();
        ++it)
    {
        RingBuffer* pRingBuffer = it->second;

        // リングバッファの中身を空にする
        // すれちがい用のRingBufferの場合クリアはしない
        if( pRingBuffer->IsForDetect() == false )
        {
            pRingBuffer->Cleanup();
        }
        else
        {
            WLAN_LOG_DEBUG("This RingBuffer is for Detect. Not cleaned up.\n");
        }
    }
}

RingBuffer* RxEntries::FindRingBufferByRxId(uint32_t rxId) NN_NOEXCEPT
{
    EntryMap::iterator it;
    it = m_Entries.find(rxId);
    if (it == m_Entries.end())
    {
        WLAN_LOG_ERROR("RxId %d not found.\n", rxId);
        return nullptr;
    }
    NN_SDK_ASSERT_NOT_NULL(it->second);
    return it->second;
}

RingBuffer* RxEntries::FindRingBufferByType(uint16_t type) NN_NOEXCEPT
{
    RingBufferMap::iterator it;
    it = m_RingBuffers.find(type);
    if( it == m_RingBuffers.end() )
    {
        WLAN_LOG_DEBUG("Type %04X not found.\n", type);
        return nullptr;
    }
    NN_SDK_ASSERT_NOT_NULL(it->second);

    return it->second;
}

Result RxEntries::FindRxIdFromRingBuffer(uint32_t* pOutRxId, const RingBuffer* pRingBuffer) const NN_NOEXCEPT
{
    EntryMap::const_iterator itStart = m_Entries.begin();
    EntryMap::const_iterator itEnd = m_Entries.end();

    for( EntryMap::const_iterator it = itStart; it != itEnd; ++it )
    {
        if( it->second == pRingBuffer )
        {
            *pOutRxId = it->first;
            return nn::ResultSuccess();
        }
    }

    return ResultRxEntryNotFound();
}



/*
 * RxEntriesForActionFrame
 */

Result RxEntriesForActionFrame::Add(uint32_t* pOutRxId, uint32_t capacity, const uint16_t types[], uint32_t count, bool bOverWrite) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRxId);

    // capacity の上下限チェック
    NN_SDK_REQUIRES_RANGE(capacity, 1, RxCapacityMax + 1);

    // Subtype数の上下限チェック
    NN_SDK_REQUIRES_RANGE(count, 1, ActionFrameTypeCountMax + 1);

    // Subtypeの範囲チェック
    for( uint32_t i = 0; i < count; i++ )
    {
        NN_SDK_REQUIRES_RANGE(types[i], ActionFrameType_Beacon, ActionFrameType_End);
    }

    nn::os::LockMutex(&m_Mutex);

    // RxEntriesの数が超過していないか
    if (m_RingBuffers.size() >= RxEntriesCountMax)
    {
        WLAN_LOG_ERROR("Number of Registered RxEntries is full (=%d).\n", m_RingBuffers.size());
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryCountIsFull();
    }

    // 重複登録をチェック
    const RingBufferMap::iterator itEnd = m_RingBuffers.end();
    for( uint32_t i = 0; i < count; ++i )
    {
        if( m_RingBuffers.find(types[i]) != itEnd )
        {
            WLAN_LOG_ERROR("Subtype %d is already registered.\n", types[i]);
            nn::os::UnlockMutex(&m_Mutex);
            return ResultDuplicateRxEntry();
        }
    }

    // リングバッファの生成
    RingBuffer* pRingBuffer = new RingBuffer(capacity, bOverWrite);
    NN_SDK_ASSERT_NOT_NULL(pRingBuffer);

    // リングバッファをSubtype と関連付ける
    // 複数のSubtype が選ばれていてもリングバッファは RX エントリ一つにつき一つ
    for( uint32_t i = 0; i < count; ++i )
    {
        std::pair<RingBufferMap::iterator, bool> ret;
        ret = m_RingBuffers.insert(RingBufferPair(types[i], pRingBuffer));
        NN_SDK_ASSERT(ret.second == true);
        WLAN_LOG_DEBUG("Added %d\n", types[i]);

        // すれちがい用のRxEntryかどうかチェック
        if( types[i] == ActionFrameType_Detect )
        {
            // すれちがい用なので、記録しておく
            WLAN_LOG_DEBUG("%s: This RxEntry is for Detect.\n", __FUNCTION__);
            pRingBuffer->SetParamForDetect(true);
        }
    }

    // リングバッファの ID を作って返す
    *pOutRxId = ++m_LastRxId;
    std::pair<EntryMap::iterator, bool> ret;
    ret = m_Entries.insert(EntryPair(*pOutRxId, pRingBuffer));
    NN_SDK_ASSERT(ret.second == true);
    WLAN_LOG_DEBUG("Created RxId %d\n", *pOutRxId);

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntriesForActionFrame::RemoveType(uint32_t* pOutRxId, uint16_t type) NN_NOEXCEPT
{
    // Subtypeの範囲チェック
    NN_SDK_REQUIRES_RANGE(type, ActionFrameType_Beacon, ActionFrameType_End);

    nn::os::LockMutex(&m_Mutex);

    RingBufferMap::iterator it = m_RingBuffers.find(type);
    if ( it == m_RingBuffers.end() )
    {
        WLAN_LOG_ERROR("Subtype %d is not registered.\n", type);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    nn::Result result = FindRxIdFromRingBuffer(pOutRxId, it->second);
    if ( result.IsFailure() )
    {
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    // すれちがい用のTypeかどうかチェック
    if( type == ActionFrameType_Detect )
    {
        // すれちがい用RingBufferであることを解除しておく
        RingBuffer* pRingBuffer = it->second;
        pRingBuffer->SetParamForDetect(false);
    }
    m_RingBuffers.erase(it);

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntriesForActionFrame::AddType(uint32_t rxId, uint16_t type) NN_NOEXCEPT
{
    // Subtypeの範囲チェック
    NN_SDK_REQUIRES_RANGE(type, ActionFrameType_Beacon, ActionFrameType_End);

    nn::os::LockMutex(&m_Mutex);

    // 重複登録をチェック
    const RingBufferMap::iterator itEnd = m_RingBuffers.end();
    if( m_RingBuffers.find(type) != itEnd )
    {
        WLAN_LOG_ERROR("Subtype %d is already registered.\n", type);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultDuplicateRxEntry();
    }

    // Subtype数の上限チェック
    if( m_RingBuffers.size() >= ActionFrameTypeCountMax )
    {
        WLAN_LOG_ERROR("The number of Subtype must be (%d) or less.\n", ActionFrameTypeCountMax);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryCountIsFull();
    }

    RingBuffer* pRingBuffer = FindRingBufferByRxId(rxId);
    if ( pRingBuffer == nullptr )
    {
        WLAN_LOG_ERROR("rxid(%d) is not found.\n", rxId);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    std::pair<RingBufferMap::iterator, bool> ret;
    ret = m_RingBuffers.insert(RingBufferPair(type, pRingBuffer));
    NN_SDK_ASSERT(ret.second == true);

    // すれちがい用のRxEntryかどうかチェック
    if( type == ActionFrameType_Detect )
    {
        // すれちがい用なので、記録しておく
        WLAN_LOG_DEBUG("%s: This RxEntry is for Detect.\n", __FUNCTION__);
        pRingBuffer->SetParamForDetect(true);
    }

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

Result RxEntriesForActionFrame::PushBuffer(nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT
{
    return PushBufferCore(pMbuf, false);
}

Result RxEntriesForActionFrame::PushBufferEx(nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT
{
    return PushBufferCore(pMbuf, true);
}

Result RxEntriesForActionFrame::PushBufferCore(nn::mbuf::Mbuf* pMbuf, bool isEx) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pMbuf);

    nn::os::LockMutex(&m_Mutex);

    if( m_bMute == true )
    {
        WLAN_LOG_DEBUG("RxEntries is muted.\n");
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryIsMuted();
    }

    // 受信バッファからSubtypeを取り出す。
    uint8_t *pData = reinterpret_cast<uint8_t*>(nn::mbuf::MbufTod(pMbuf));
    NN_SDK_ASSERT_NOT_NULL(pData);
    uint8_t subtype;
    if( isEx == true )
    {
        subtype = pData[driver::offsetToActionFrameSubtypeEx];
    }
    else
    {
        subtype = pData[driver::offsetToActionFrameSubtype];
    }

    WLAN_LOG_DEBUG("Subtype is %d @ %s\n", subtype, __FUNCTION__);

    RingBuffer* pRingBuffer = FindRingBufferByType(static_cast<uint16_t>(subtype));
    if ( pRingBuffer == nullptr )
    {
        WLAN_LOG_DEBUG("Subtype %d is not registered.\n", subtype);
        nn::os::UnlockMutex(&m_Mutex);
        return ResultRxEntryNotFound();
    }

    // リングバッファに強制的にエンキューする
    // あふれたバッファは内部で自動的に破棄される
    pRingBuffer->ForceEnqueue(pMbuf);

    nn::os::UnlockMutex(&m_Mutex);

    return nn::ResultSuccess();
}

}}

