﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/util/util_FormatString.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/TargetConfigs/build_Base.h>

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nn/gpio/gpio.h>
#endif

#include "server/wlan_HipcServer.h"

#include "wlan_DebugLog.h"
#include "wlan_MemoryInit.h"
#include "wlan_StateMachine.h"
#include "wlan_BaseFunctions.h"
#include "wlan_Mezcal.h"
#include "wlan_WpaSupplicant.h"
#include "wlan_Settings.h"

#ifdef USE_LOCALWL
#include "dbg/wlan_Htcs.h"
#include "dbg/wlan_DbgConsole.h"
#endif

namespace {

const size_t           ThreadStackSize = 8192 * 2;
NN_ALIGNAS(4096) char  g_StateMachineStack[ ThreadStackSize ];
#if defined(CHECK_HIGH_PRIORITY_THREAD)
NN_ALIGNAS(4096) char  g_DelayCheckStackHigh[ ThreadStackSize ];
#elif defined(CHECK_LOW_PRIORITY_THREAD)
NN_ALIGNAS(4096) char  g_DelayCheckStackLow[ ThreadStackSize ];
#endif
#if defined(ENABLE_BUFFER_STARVATION_CHECK)
NN_ALIGNAS(4096) char  g_bufferCheckStack[4096];
nn::os::ThreadType     g_bufferCheckThread;
bool                  g_isExitBufferCheck;
#endif

const int HipcServerThreadCount = 4;
const size_t HipcServerThreadStackSize = 8192 * 2;
NN_ALIGNAS(4096) char g_HipcServerThreadStacks[HipcServerThreadCount][HipcServerThreadStackSize];

#if defined(CHECK_HIGH_PRIORITY_THREAD)
nn::os::ThreadType  g_DelayCheckThreadHigh;
#elif defined(CHECK_LOW_PRIORITY_THREAD)
nn::os::ThreadType  g_DelayCheckThreadLow;
#endif
nn::os::ThreadType  g_StateMachineThread;
nn::os::ThreadType  g_HipcServerThreads[HipcServerThreadCount];

nn::wlan::StateMachine* g_pStateMachine;
nn::wlan::WlanBaseFunctions* g_pDriverFunctions;
nn::wlan::WpaSupplicant* g_pWpaSupplicantFunctions;
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
const nn::TimeSpan PadValueSwitchTime = nn::TimeSpan::FromMilliSeconds(5); // gpio (tentative)
#endif
}   // namespace


namespace nn { namespace wlan {

int WlanLogLevel = LogLevel_Fatal;

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
/* gpio (tentative) ------------------------------------------------ */
void InitializeGpioOutput(nn::gpio::GpioPadName pad) NN_NOEXCEPT
{
    nn::gpio::GpioPadSession session;
    nn::gpio::Initialize();
    nn::gpio::OpenSession(&session, pad);
    nn::gpio::SetDirection(&session, nn::gpio::Direction_Output);
    nn::gpio::SetValue(&session, nn::gpio::GpioValue_High);
    nn::os::SleepThread(PadValueSwitchTime);
    nn::gpio::CloseSession(&session);
    nn::gpio::Finalize();
}
#endif

/* WLANプロセスの初期化処理 ------------------------------------------------ */
void InitializeWlan() NN_NOEXCEPT
{
#if defined(NN_WLAN_DEBUG_LOG_LEVEL_FATAL)
    nn::wlan::WlanLogLevel = nn::wlan::LogLevel_Fatal;
#elif defined(NN_WLAN_DEBUG_LOG_LEVEL_ERROR)
    nn::wlan::WlanLogLevel = nn::wlan::LogLevel_Error;
#elif defined(NN_WLAN_DEBUG_LOG_LEVEL_INFO)
    nn::wlan::WlanLogLevel = nn::wlan::LogLevel_Info;
#elif defined(NN_WLAN_DEBUG_LOG_LEVEL_DEBUG)
    nn::wlan::WlanLogLevel = nn::wlan::LogLevel_Debug;
#endif

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    if( nn::wlan::FirmwareDebugSettings::IsSkipBoot() == false )
    {
        // WIFI_REG_ONをHに
        nn::wlan::InitializeGpioOutput(nn::gpio::GpioPadName_WifiReset); // gpio (tentative)
    }
#endif
    nn::wlan::InitializeMbuf();
}

/* WLANプロセスの終了処理 -------------------------------------------------- */
void FinalizeWlan() NN_NOEXCEPT
{
    nn::wlan::FinalizeMbuf();
    nn::wlan::FinalizeMemory();
}

void CreateStateMachineThread() NN_NOEXCEPT
{
    nn::Result result;

    /*
     * StateMachineスレッドの起動
     */

    WLAN_LOG_INFO("WLAN: DriverFunction class created\n");
    // Broadcomの無線ドライバを扱うクラスを生成
    nn::wlan::Mezcal* mezcal = new Mezcal();
    nn::wlan::WpaSupplicant* supplicant = new WpaSupplicant();

    // 汎用的な無線関数を扱うクラスにMezcalクラスを登録
    g_pDriverFunctions = reinterpret_cast<nn::wlan::WlanBaseFunctions*>(mezcal);
    g_pWpaSupplicantFunctions = reinterpret_cast<nn::wlan::WpaSupplicant*>(supplicant);

    WLAN_LOG_INFO("WLAN: StateMachine created\n");
    g_pStateMachine = new nn::wlan::StateMachine();

    g_pStateMachine->Initialize( nn::wlan::StateMachine::EventMessageCountMax,
                                 nn::wlan::StateMachine::CommandMessageCountMax,
                                 g_pDriverFunctions,
                                 g_pWpaSupplicantFunctions);

    // Mezcalクラスにステートマシーンを覚えさせておく
    nn::wlan::BindStateMachine( g_pStateMachine );
    // Bind StateMachine class to WpaSupplicant class
    nn::wlan::WpaSupplicant::BindStateMachineInstance( g_pStateMachine );

    result = nn::os::CreateThread( &g_StateMachineThread,
                                   StateMachineThreadFunc,
                                   g_pStateMachine,
                                   g_StateMachineStack,
                                   sizeof(g_StateMachineStack),
                                   NN_SYSTEM_THREAD_PRIORITY(wlan, StateMachine) );

    NN_SDK_ASSERT( result.IsSuccess(), "WLAN: Cannot create StateMachineThread." );

    nn::os::SetThreadNamePointer(&g_StateMachineThread, NN_SYSTEM_THREAD_NAME(wlan, StateMachine));
    WLAN_LOG_INFO("StateMachine starts\n");
    nn::os::StartThread( &g_StateMachineThread );

    // StateMachineの初期化完了まで待つ
    g_pStateMachine->WaitInitialization();
}

void LoopHipcServerThreadFunction(void* arg)
{
    nn::wlan::server::HipcServerManager* pManager = reinterpret_cast<nn::wlan::server::HipcServerManager*>(arg);

    // IPCリクエスト処理開始
    pManager->LoopAuto();

    // TODO サーバーの終了処理
}

void CreateHipcServerThreads(nn::wlan::server::HipcServerManager* pManager)
{
    nn::Result result;
    /*
     * HIPC Serverスレッドの起動
     */
    for (int i = 0; i < HipcServerThreadCount; ++i)
    {
        result = nn::os::CreateThread( &g_HipcServerThreads[i],
                                       LoopHipcServerThreadFunction,
                                       pManager,
                                       g_HipcServerThreadStacks[i],
                                       HipcServerThreadStackSize,
                                       NN_SYSTEM_THREAD_PRIORITY(wlan, IpcServer) );
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::SetThreadNamePointer(&g_HipcServerThreads[i], NN_SYSTEM_THREAD_NAME(wlan, IpcServer));
        WLAN_LOG_INFO("HipcServerThread %d starts\n", i);
        nn::os::StartThread( &g_HipcServerThreads[i] );
    }
}

void DestroyStateMachine()
{
    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_StateMachineThread );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_StateMachineThread );

    delete g_pDriverFunctions;
    delete g_pWpaSupplicantFunctions;
    delete g_pStateMachine;
}

void InitializeWlanDriver()
{
    // DHDの初期化を行う
    // StateMachineに初期化要求コマンドをポストする
    WLAN_LOG_INFO("Do : post initialize command\n");
#if defined(PRINT_INITIALIZATION_TIME)
    nn::os::Tick sysTick;
    sysTick = nn::os::GetSystemTick();
#endif

    WlanCommand* pcmdbuff;

    pcmdbuff = g_pStateMachine->WlanGetCommandBuff(0);
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
    pcmdbuff->id = WLAN_INITIALIZE;
    g_pStateMachine->PostCommandMessage(pcmdbuff);
    WLAN_LOG_INFO("Done : post initialize command\n");
    g_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);
#if defined(PRINT_INITIALIZATION_TIME)
    NN_SDK_LOG("Wlan initialize done. %lld[ms]\n",
            (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
#else
    NN_SDK_LOG("Wlan initialize done.\n");
#endif
}

void FinalizeWlanDriver()
{
    // DHDの初期化を行う
    // StateMachineに初期化要求コマンドをポストする
    WLAN_LOG_INFO("Do : post finalize command\n");
    WlanCommand* pcmdbuff;
    pcmdbuff = g_pStateMachine->WlanGetCommandBuff(0);
    if( pcmdbuff != NULL )
    {
        pcmdbuff->id = WLAN_FINALIZE;
        g_pStateMachine->PostCommandMessage(pcmdbuff);
        WLAN_LOG_INFO("Done : post finalize command\n");
    }
}

#if defined(CHECK_HIGH_PRIORITY_THREAD)
/* Delay Check Thread ------------------------------------------------ */
void DelayCheckThreadFuncHigh(void *pArg) NN_NOEXCEPT
{
    nn::os::Tick sleepBeforeTick;
    nn::os::Tick sleepAfterTick;
    nn::os::Tick enterTick;
    int sendCount;
    int timeSpan;
    int timeDelayMax;
    int timeDelayMin;
    double timeDelayAverage;

    NN_SDK_LOG("%s start\n", __FUNCTION__);
    enterTick = nn::os::GetSystemTick();
    sleepBeforeTick = enterTick;
    timeDelayMax = 0;
    timeDelayMin = 0;
    timeDelayAverage = 0;
    sendCount = 0;

    while (true) {
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(500));
        sleepAfterTick = nn::os::GetSystemTick();
        timeSpan = (sleepAfterTick - sleepBeforeTick).ToTimeSpan().GetMicroSeconds();
        if (timeSpan > timeDelayMax) {
            timeDelayMax = timeSpan;
        }
        if (timeDelayMin == 0) {
            timeDelayMin = timeSpan;
        } else if (timeSpan < timeDelayMin) {
            timeDelayMin = timeSpan;
        }
        timeDelayAverage = (timeSpan + timeDelayAverage * sendCount) / (sendCount + 1);
        sendCount++;
        if (timeSpan >= 100000) {
            NN_SDK_LOG("H.Detect delay %d us\n", timeSpan);
        }

        timeSpan = (sleepAfterTick - enterTick).ToTimeSpan().GetMicroSeconds();
        if (timeSpan >= 5000000) {
            NN_SDK_LOG("[H.sleep:%d]delay time[min:%d, max:%d, avg:%.3lf]\n",
                       sendCount, timeDelayMin, timeDelayMax, timeDelayAverage);
            enterTick = nn::os::GetSystemTick();
            timeDelayMax = 0;
            timeDelayMin = 0;
            timeDelayAverage = 0;
            sendCount = 0;
        }
        sleepBeforeTick = nn::os::GetSystemTick();
    }
}

void CreateDelayCheckThreadHigh() NN_NOEXCEPT
{
    nn::Result result;

    /*
     * DelayCheckスレッドの起動
     */
    result = nn::os::CreateThread( &g_DelayCheckThreadHigh,
                                   DelayCheckThreadFuncHigh,
                                   NULL,
                                   g_DelayCheckStackHigh,
                                   sizeof(g_DelayCheckStackHigh),
                                   NN_SYSTEM_THREAD_PRIORITY(wlan, DelayCheckHigh) );

    NN_SDK_ASSERT( result.IsSuccess(), "WLAN: Cannot create DelayCheckThread." );

    nn::os::SetThreadNamePointer(&g_DelayCheckThreadHigh, NN_SYSTEM_THREAD_NAME(wlan, DelayCheckHigh));
    NN_SDK_LOG("DelayCheckHigh starts\n");
    nn::os::StartThread( &g_DelayCheckThreadHigh );
}

void DestroyDelayCheckHigh()
{
    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_DelayCheckThreadHigh );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_DelayCheckThreadHigh );
}
#elif defined(CHECK_LOW_PRIORITY_THREAD)
/* Delay Check Thread ------------------------------------------------ */
void DelayCheckThreadFuncLow(void *pArg) NN_NOEXCEPT
{
    nn::os::Tick sleepBeforeTick;
    nn::os::Tick sleepAfterTick;
    nn::os::Tick enterTick;
    int sendCount;
    int timeSpan;
    int timeDelayMax;
    int timeDelayMin;
    double timeDelayAverage;

    NN_SDK_LOG("%s start\n", __FUNCTION__);
    enterTick = nn::os::GetSystemTick();
    sleepBeforeTick = enterTick;
    timeDelayMax = 0;
    timeDelayMin = 0;
    timeDelayAverage = 0;
    sendCount = 0;

    while (true) {
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(500));
        sleepAfterTick = nn::os::GetSystemTick();
        timeSpan = (sleepAfterTick - sleepBeforeTick).ToTimeSpan().GetMicroSeconds();
        if (timeSpan > timeDelayMax) {
            timeDelayMax = timeSpan;
        }
        if (timeDelayMin == 0) {
            timeDelayMin = timeSpan;
        } else if (timeSpan < timeDelayMin) {
            timeDelayMin = timeSpan;
        }
        timeDelayAverage = (timeSpan + timeDelayAverage * sendCount) / (sendCount + 1);
        sendCount++;
        if (timeSpan >= 100000) {
            NN_SDK_LOG("L.Detect delay %d us\n", timeSpan);
        }

        timeSpan = (sleepAfterTick - enterTick).ToTimeSpan().GetMicroSeconds();
        if (timeSpan >= 5000000) {
            NN_SDK_LOG("[L.sleep:%d]delay time[min:%d, max:%d, avg:%.3lf]\n",
                       sendCount, timeDelayMin, timeDelayMax, timeDelayAverage);
            enterTick = nn::os::GetSystemTick();
            timeDelayMax = 0;
            timeDelayMin = 0;
            timeDelayAverage = 0;
            sendCount = 0;
        }
        sleepBeforeTick = nn::os::GetSystemTick();
    }
}

void CreateDelayCheckThreadLow() NN_NOEXCEPT
{
    nn::Result result;

    /*
     * DelayCheckスレッドの起動
     */
    result = nn::os::CreateThread( &g_DelayCheckThreadLow,
                                   DelayCheckThreadFuncLow,
                                   NULL,
                                   g_DelayCheckStackLow,
                                   sizeof(g_DelayCheckStackLow),
                                   NN_SYSTEM_THREAD_PRIORITY(wlan, DelayCheckLow) );

    NN_SDK_ASSERT( result.IsSuccess(), "WLAN: Cannot create DelayCheckThread." );

    nn::os::SetThreadNamePointer(&g_DelayCheckThreadLow, NN_SYSTEM_THREAD_NAME(wlan, DelayCheckLow));
    NN_SDK_LOG("DelayCheckLow starts\n");
    nn::os::StartThread( &g_DelayCheckThreadLow );
}

void DestroyDelayCheckLow()
{
    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_DelayCheckThreadLow );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_DelayCheckThreadLow );
}
#endif

#if defined(ENABLE_BUFFER_STARVATION_CHECK)
void BufferCheckThreadFunc(void* arg) NN_NOEXCEPT
{
    while( g_isExitBufferCheck == false )
    {
        WLAN_LOG_FORCE("Normal buffer: %d[bytes] available\n", nnwlanGetTotalFreeSize());
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
    }
}

void StartBufferMonitor()
{
    nn::Result result;

    /*
     * DelayCheckスレッドの起動
     */
    result = nn::os::CreateThread( &g_bufferCheckThread,
            BufferCheckThreadFunc,
            NULL,
            g_bufferCheckStack,
            sizeof(g_bufferCheckStack),
            NN_SYSTEM_THREAD_PRIORITY(wlan, MainThread) );
    NN_SDK_ASSERT( result.IsSuccess(), "WLAN: Cannot create BufferCheckThread." );

    g_isExitBufferCheck = false;
    nn::os::StartThread( &g_bufferCheckThread );
}

void StopBufferMonitor()
{
    nn::Result result;
    g_isExitBufferCheck = true;
    nn::os::WaitThread( &g_bufferCheckThread );
    nn::os::DestroyThread(&g_bufferCheckThread);
}
#endif
}} // nn::wlan
/*
extern "C" void NNC_SDK_LOG(const char *pFormat, ...)
{
    char buffer[256];
    va_list ap;
    va_start(ap, pFormat);
    nn::util::VSNPrintf(buffer, sizeof(buffer), pFormat, ap);
    va_end(ap);
    NN_SDK_LOG(buffer);
}
*/

// 空の nndiagStartup() を定義
extern "C" void nndiagStartup()
{
}

//-----------------------------------------------------------------------------
//  スタートアップ関数
//-----------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    nn::wlan::InitializeMemory();
}


/* WLAN main ---------------------------------------------------------------- */
extern "C" void nnMain()
{
    WLAN_LOG_INFO("process starts\n");

    // wlanプロセスのスレッドを取得する
    nn::os::ThreadType* const pCurrentThread = ::nn::os::GetCurrentThread();

    // メインスレッドに優先度の設定
    nn::os::ChangeThreadPriority(pCurrentThread, NN_SYSTEM_THREAD_PRIORITY(wlan, Main));

    // メインスレッドにスレッド名の設定
    nn::os::SetThreadNamePointer(pCurrentThread, NN_SYSTEM_THREAD_NAME(wlan, Main));

    // wlanプロセスのデバッグセッティング読み込み
    nn::wlan::FirmwareDebugSettings::Initialize();

    // メモリアロケータの初期化
    nn::wlan::InitializeWlan();

    // StateMachineスレッドの起動
    nn::wlan::CreateStateMachineThread();

#if defined(CHECK_HIGH_PRIORITY_THREAD)
    nn::wlan::CreateDelayCheckThreadHigh();
#elif defined(CHECK_LOW_PRIORITY_THREAD)
    nn::wlan::CreateDelayCheckThreadLow();
#endif

#if defined(ENABLE_BUFFER_STARVATION_CHECK)
    nn::wlan::StartBufferMonitor();
#endif

    // IPCサーバーマネージャーの作成
    // コンストラクタ内でStart()も呼んでいる
    nn::wlan::server::HipcServerManager* pManager = new nn::wlan::server::HipcServerManager(g_pStateMachine);

    // 無線ドライバの初期化
    // LoopAutoを回すより先にMACアドレスをセットする必要がある
    nn::wlan::InitializeWlanDriver();

    // HipcServerスレッドの起動
    nn::wlan::CreateHipcServerThreads(pManager);

#ifdef USE_LOCALWL
    // wlandbg
    nn::wlan::dbg::wlan_DbgConsoleInit();
#endif

    // TODO Sleepや電源断などの割り込み受付

    // TORIAEZU:メインスレッドが終了しないように待機
    nn::os::WaitThread(&g_StateMachineThread);
    for (int i = 0; i < HipcServerThreadCount; ++i)
    {
        nn::os::WaitThread(&g_HipcServerThreads[i]);
    }

    // 無線ドライバの終了処理
    nn::wlan::FinalizeWlanDriver();

    // ステートマシーンスレッドの終了
    nn::wlan::DestroyStateMachine();

#if defined(ENABLE_BUFFER_STARVATION_CHECK)
    nn::wlan::StopBufferMonitor();
#endif

    // DelayCheckスレッドの終了
#if defined(CHECK_HIGH_PRIORITY_THREAD)
    nn::wlan::DestroyDelayCheckHigh();
#elif defined(CHECK_LOW_PRIORITY_THREAD)
    nn::wlan::DestroyDelayCheckLow();
#endif

    // メモリアロケータの終了処理
    nn::wlan::FinalizeWlan();

    WLAN_LOG_INFO("WALN: process ends\n\n\n");
}

