﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/wlan/wlan_MacAddress.h>
#include <nn/wlan/wlan_Types.h>
#include "wlan_CommonType.h"
#include "wlan_DebugLog.h"

namespace nn { namespace wlan {

class ClientStatusManager
{
public:
    ClientStatusManager() NN_NOEXCEPT;  // state = ConnectionState_Disconnectedの空リストを作成する
    ~ClientStatusManager() NN_NOEXCEPT;

    // ClientStatusの中身を初期状態にします。
    void Initialize() NN_NOEXCEPT;

    /*
     * リストを更新します。
     * リスト内に一致するBSSIDがあれば、内容を上書きします。
     * 一致するBSSIDが存在しない場合は、stateがDisconnectedかつリストの一番若いところに格納されているものを消して追加します。
     * 空き場所がなければ格納せず失敗とします。
     * stateがDisconnectedかつBSSIDがBroadcastアドレスの場合、Connected状態のものを全てDisconnected状態にします。
     * 追加に成功した場合、trueが、 失敗の場合、falseが返ります。
     */
    bool UpdateClientStatus(const ClientStatus& status) NN_NOEXCEPT;

    void GetClientStatusList(ClientStatus* pOutList, uint8_t count, Bit32* pOutBitmap, bool IsClear = true) NN_NOEXCEPT;

    /*
     * 入力したBSSIDに合致するClientStatusを取得します。該当BSSIDが見つかればtrueを、無ければfalseを返します。
     */
    bool GetClientStatusByMac(ClientStatus* pOutStatus, const MacAddress& mac) NN_NOEXCEPT;

    /*
     * ClientStatusBitMapを用いて、更新のあったClientの数を取得します。
     */
    const uint32_t GetClientCount() NN_NOEXCEPT;

    /*
     * 現在接続しているClientの数を取得します。
     */
    const int GetConnectedClientCount() NN_NOEXCEPT
    {
        int ret = 0;
        nn::os::LockMutex(&m_Mutex);
        ret = m_ConnectedClientCount;
        nn::os::UnlockMutex(&m_Mutex);
        return ret;
    }

    /*
     * 無線ドライバが保持しているClientのMACアドレスリストとClientManagerが保持しているMACアドレスリストを比較します
     * 完全一致でtrueが返ります。
     * 不完全一致でfalseが返ります。
     */
    bool VerifyClientList(const WlanMacAddressData* list, uint32_t count) NN_NOEXCEPT;
private:
    void EditUpdatedClientStatusBitMap(uint32_t index)
    {
        m_UpdatedClientStatusBitMap |= ((0x00000001 << index) & 0x000000FF); // プラットフォーム別に接続最大数に応じて変える必要がある
    }

private:
    nn::os::MutexType        m_Mutex;
    Bit32 m_UpdatedClientStatusBitMap;  // 更新があった場所をビットフラグで管理
    ClientStatus m_Status[ConnectableClientsCountMax];
    int m_ConnectedClientCount;
};

}}

