﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cctype>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_MacAddress.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_StateDefinition.h>
#include <nn/wlan/wlan_Result.h>

#include "wlan_InfraApiImpl.h"
#include "../wlan_MsgBuffer.h"
#include "../wlan_IeParser.h"
#include "../wlan_SignalStrength.h"

namespace nn { namespace wlan { namespace server
{
    Result InfraApiImpl::OpenMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_InfraIdle
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        return ChangeMode(WLAN_CHANGE_STATE_INFRA);
    }

    Result InfraApiImpl::CloseMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_InfraIdle,
                WdrvMainState_Sleep
        };
        nn::Result result = CheckState(states, 3);
        if( result.IsFailure() )
        {
            return result;
        }

        return ChangeMode(WLAN_CHANGE_STATE_READY);
    }

    nn::Result InfraApiImpl::StartScan(const nn::wlan::detail::SfdlScanParameters& scanParam) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // パラメータチェック
        if( !(scanParam.scanType == static_cast<uint32_t>(ScanType_Active) ||
                scanParam.scanType == static_cast<uint32_t>(ScanType_Passive)) )
        {
            NN_SDK_REQUIRES(false, "scanType is invalid");
            return ResultInvalidArgument();
        }
        if( !(scanParam.channelScanTime == -1 || (scanParam.channelScanTime >= 10 && scanParam.channelScanTime <= 3000)) )
        {
            NN_SDK_REQUIRES(false, "channelScanTime is invalid");
            return ResultInvalidArgument();
        }
        if( !(scanParam.homeChannelTime == -1 || (scanParam.homeChannelTime >= 0 && scanParam.homeChannelTime <= 500)) )
        {
            NN_SDK_REQUIRES(false, "homeChannelTime is invalid");
            return ResultInvalidArgument();
        }
        if( std::memcmp(&scanParam.bssid.data[0], MacAddress::CreateZeroMacAddress().GetMacAddressData(),
                MacAddress::MacAddressSize) == 0 )
        {
            NN_SDK_REQUIRES(false, "bssid must not be ZeroMacAddress");
            return ResultInvalidArgument();
        }
        // スキャンチャンネルチェック
        NN_RESULT_DO(m_pStateMachine->CheckChannels(scanParam.channelList.channel, scanParam.channelCount));

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;

        // I/FのUP要求
        // 既にUPされていたら、スルーされる
        WLAN_LOG_DEBUG("[%s] interface up\n", __FUNCTION__);

        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(uint32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SET_IFUPDOWN;
        *(reinterpret_cast<uint32_t*>(pcmdbuff->Args)) = 1;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            return result;
        }

        // scanリクエスト送る
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanScanParameters));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        pcmdbuff->id = WLAN_SCAN_REQUEST;
        WlanScanParameters* params = static_cast<WlanScanParameters*>(pcmdbuff->Args);

        // scanパラメータのコピー
        if( scanParam.channelCount > 0 )  // channelCountのコピー
        {
            for(int i = 0; i < scanParam.channelCount; i++)
            {
                params->channelList[i] = scanParam.channelList.channel[i];
            }
        }
        if( scanParam.ssidCount > 0 )  // SSIDリストのコピー
        {
            for(int i = 0; i < scanParam.ssidCount; i++)
            {
                params->ssidList[i].length = scanParam.ssidList[i].length;
                std::memcpy(&params->ssidList[i].ssid[0], &scanParam.ssidList[i].data[0], scanParam.ssidList[i].length);
            }
        }
        params->ssidCount = scanParam.ssidCount;
        std::memcpy(&params->bssid[0], &scanParam.bssid.data[0], MacAddress::MacAddressSize); // ZeroMacはShimではじく
        params->scanType = scanParam.scanType;
        params->channelScanTime = scanParam.channelScanTime;
        params->homeChannelTime = scanParam.homeChannelTime;
        params->channelCount = scanParam.channelCount;

        WLAN_LOG_DEBUG("[%s] scan request post.\n", __FUNCTION__);
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff); // コマンドバッファの削除

        return result;
    } //NOLINT(impl/function_size)


    nn::Result InfraApiImpl::GetScanResult(const nn::sf::OutBuffer& pOutScanBuffer) NN_NOEXCEPT
    {
        // パラメータチェック
        if( pOutScanBuffer.GetSize() <= sizeof(ScanResultHeader) )
        {
            NN_SDK_REQUIRES(false, "%s: scan buffer size must be larger than %d",
                    __FUNCTION__, sizeof(ScanResultHeader));
            return ResultInvalidArgument();
        }

        WlanCommand* pcmdbuff;

        /*
         * スキャン結果の取得
         */
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanBssInfo*));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SCANLIST_REQUEST;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff->Args);

        WlanBssInfo** pbsslist = static_cast<WlanBssInfo**>(pcmdbuff->Args);
        WlanBssInfo* pBssInfo = *pbsslist;

        uint8_t* pScanBuf = reinterpret_cast<uint8_t*>(pOutScanBuffer.GetPointerUnsafe());
        // バッファの先頭にスキャン結果全体に関する情報を入れるためのヘッダを配置
        ScanResultHeader* resultHeader = reinterpret_cast<ScanResultHeader*>(pScanBuf);
        resultHeader->bufferLength = pOutScanBuffer.GetSize();
        resultHeader->bssCount = 0;
        resultHeader->resultLength = sizeof(ScanResultHeader);

        // ポインタを進める
        pScanBuf += sizeof(ScanResultHeader);

        if( pBssInfo == NULL )
        {
            // 1個もBSSが見つからなかったということ
            WLAN_LOG_DEBUG("Can't find any BSS\n");
        }
        else
        {
            size_t capacity = pOutScanBuffer.GetSize();  // スキャンバッファの空き容量を記録する変数

            WLAN_LOG_DEBUG("Scan buff capacity is %d\n", capacity);

            capacity -= sizeof(ScanResultHeader);  // ヘッダ分を削っておく

            // スキャン結果を掘っていき、BSSがスキャンバッファに収まるようであれば、収めていく。
            // スキャンバッファが足りなくなったらBSSの格納を止める。収まりきらなかったBSSは上位層には渡らない。
            while(pBssInfo != NULL)
            {
                // BSSのサイズを取得
                IeParser* pParser = NULL;
                size_t bssSize = sizeof(nn::wlan::BssInfo); // IE以外の基本情報を格納する構造体のサイズ

                // IEのサイズを取得
                // VIEに限定せず全てのタグIEを含める
                uint32_t ieCount = 0;
                if( pBssInfo->IeData != NULL )
                {
                    pParser = new IeParser(pBssInfo->IeData, pBssInfo->IeLength);
                    NN_SDK_ASSERT_NOT_NULL(pParser);

                    uint8_t* rp;
                    uint8_t id;
                    uint8_t length;
                    rp = pParser->SearchIe(-1);
                    while(rp != NULL)
                    {
                        pParser->ReadIdAndLen(&id, &length, rp);
                        bssSize += sizeof(uint8_t); // IE ID サイズ
                        bssSize += sizeof(uint8_t); // IE length サイズ
                        bssSize += length;         // IE body の長さ
                        ieCount++;
                        rp = pParser->SearchIeContinue(-1);
                    }
                    delete pParser;
                    WLAN_LOG_DEBUG("BSS size is %d\n", bssSize);
                }

                // 上位層から与えられたScanBufferの空き容量に収まれば、BSS情報を格納する
                if( capacity >= bssSize )
                {
                    nn::wlan::BssInfo* pInfo = reinterpret_cast<nn::wlan::BssInfo*>(pScanBuf);
                    pInfo->beaconPeriod = pBssInfo->BeaconPeriod;
                    pInfo->bssSize = bssSize;
                    std::memcpy(&pInfo->bssid[0], &pBssInfo->Bssid[0], nn::wlan::MacAddress::MacAddressSize);
                    pInfo->ssidLength = pBssInfo->SsidInfo.length;
                    std::memcpy(&pInfo->ssid[0], &pBssInfo->SsidInfo.ssid[0], pBssInfo->SsidInfo.length);
                    pInfo->capability = pBssInfo->Capability;
                    if ((pBssInfo->NCapability) && (pBssInfo->ControlChannel))
                    {
                        pInfo->channel = pBssInfo->ControlChannel;
                    }
                    else
                    {
                        pInfo->channel = pBssInfo->Channel;
                    }
                    pInfo->ieCount = ieCount;
                    pInfo->rateCount = pBssInfo->RateCount;
                    std::memset(&pInfo->rates[0], 0, sizeof(pInfo->rates));  // 先に0埋めしておく
                    std::memcpy(&pInfo->rates[0], &pBssInfo->Rates[0], pBssInfo->RateCount);
                    pInfo->rssi = pBssInfo->Rssi;
                    if( pInfo->channel < WirelessChannel_36ch )
                    {
                        pInfo->linkLevel = static_cast<uint32_t>(SignalStrength(pBssInfo->Rssi).GetLinkLevelInfra2G());
                    }
                    else
                    {
                        pInfo->linkLevel = static_cast<uint32_t>(SignalStrength(pBssInfo->Rssi).GetLinkLevelInfra5G());
                    }

                    // ポインタを進める
                    pScanBuf += sizeof(nn::wlan::BssInfo);

                    // IEのコピー
                    if( pBssInfo->IeData != NULL )
                    {
                        pParser = new IeParser(pBssInfo->IeData, pBssInfo->IeLength);
                        NN_SDK_ASSERT_NOT_NULL(pParser);

                        uint8_t* rp;
                        uint8_t id;
                        uint8_t length;
                        rp = pParser->SearchIe(-1);
                        while(rp != NULL)
                        {
                            pParser->ReadIdAndLen(&id, &length, rp);

                            // IE ID のコピー
                            *reinterpret_cast<uint8_t*>(pScanBuf) = id;
                            // ポインタを進める
                            pScanBuf += sizeof(uint8_t);

                            // IE length のコピー
                            *reinterpret_cast<uint8_t*>(pScanBuf) = length;
                            // ポインタを進める
                            pScanBuf += sizeof(uint8_t);

                            // VIE 本体のコピー（OUI以降）
                            std::memcpy(pScanBuf, pParser->GetPayloadAddress(rp), length);
                            // ポインタを進める
                            pScanBuf += length;

                            rp = pParser->SearchIeContinue(-1);
                        }
                        delete pParser;
                    }

                    capacity -= bssSize;
                    resultHeader->bssCount += 1;
                    resultHeader->resultLength += bssSize;
                }
                else
                {
                    // Scan buffer サイズが足りない
                    WLAN_LOG_DEBUG("Scan buffer does not have an empty space to store a BSS info\n");
                    break;
                }

                pBssInfo = pBssInfo->NextItem;
            }
            WLAN_LOG_DEBUG("Scan buff capacity is %d left\n", capacity);
            WLAN_LOG_DEBUG("BSS info count : %d\n", resultHeader->bssCount);
        }

        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff); // コマンドバッファの削除

        NN_RESULT_SUCCESS;

    } //NOLINT(impl/function_size)

    nn::Result InfraApiImpl::StopScan() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta
        };

        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_CANCEL_SCAN;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);

        return result;
    }

    nn::Result InfraApiImpl::Connect(const nn::wlan::detail::SfdlSsid& ssid, nn::wlan::detail::SfdlMacAddress bssid, int16_t channel,
            const nn::wlan::detail::SfdlSecurity& security, bool autoKeepAlive, int32_t beaconLostTimeout) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        NN_UNUSED(autoKeepAlive);

        // パラメータチェック
        if( ssid.length == 0 )
        {
            NN_SDK_REQUIRES(false, "SSID must not be empty");
            return ResultInvalidArgument();
        }
        if( ssid.length > Ssid::SsidLengthMax )
        {
            NN_SDK_REQUIRES(false, "SSID size must be less than %d", Ssid::SsidLengthMax + 1);
            return ResultInvalidArgument();
        }
        if( std::memcmp(&bssid.data[0], MacAddress::CreateZeroMacAddress().GetMacAddressData(),
                MacAddress::MacAddressSize) == 0 )
        {
            NN_SDK_REQUIRES(false, "bssid must not be ZeroMacAddress");
            return ResultInvalidArgument();
        }
        if( !(channel == -1 || (channel >= WirelessChannel_1ch && channel <= WirelessChannel_165ch)) )
        {
            NN_SDK_REQUIRES(false, "channel is invalid");
            return ResultInvalidArgument();
        }
        if( security.privacyMode > static_cast<uint32_t>(SecurityMode_Wpa2Aes) )
        {
            NN_SDK_REQUIRES(false, "privacyMode is invalid");
            return ResultInvalidArgument();
        }
        if( security.privacyMode >= SecurityMode_WpaTkip && security.groupPrivacyMode == SecurityMode_StaticAes )
        {
            NN_SDK_REQUIRES(false, "groupPrivacyMode is invalid");
            return nn::wlan::ResultInvalidArgument();
        }
        if( security.privacyMode >= static_cast<uint32_t>(SecurityMode_Wep64Open) &&
                security.privacyMode <= static_cast<uint32_t>(SecurityMode_Wep128Shared) )
        {
            if(security.privacyMode == nn::wlan::SecurityMode_Wep64Open || security.privacyMode == nn::wlan::SecurityMode_Wep64Shared )
            {
                if( nn::util::Strnlen(security.key, 11) != 10 )
                {
                    NN_SDK_REQUIRES(false, "WEP64 key length must be 10\n");
                    return nn::wlan::ResultInvalidArgument();
                }
            }
            else if(security.privacyMode == nn::wlan::SecurityMode_Wep128Open || security.privacyMode == nn::wlan::SecurityMode_Wep128Shared )
            {
                if( nn::util::Strnlen(security.key, 27) != 26 )
                {
                    NN_SDK_REQUIRES(false, "WEP128 key length must be 26\n");
                    return nn::wlan::ResultInvalidArgument();
                }
            }
            if( security.keyIdx > 3 )
            {
                NN_SDK_REQUIRES(false, "keyIdx is invalid");
                return ResultInvalidArgument();
            }
        }
        if( !(beaconLostTimeout >= 1 && beaconLostTimeout <= 30) )
        {
            NN_SDK_REQUIRES(false, "beaconLostTimeout is invalid");
            return ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_InfraIdle,
        };
        nn::Result result = CheckState(states, 1);
        if( result.IsFailure() )
        {
            return result;
        }

#if !defined(USE_WPA_SUPPLICANT)
        if( security.privacyMode != nn::wlan::SecurityMode_Open )
        {
            WLAN_LOG_ERROR("Sorry!! You can not use privacyMode other than SecurityMode_Open mode.\n");
            return ResultInvalidArgument();
        }
#endif

        switch(security.privacyMode)
        {
        case SecurityMode_WpaTkip:
        case SecurityMode_WpaAes:
        case SecurityMode_Wpa2Tkip:
        case SecurityMode_Wpa2Aes:
        {
            int length;
            length = nn::util::Strnlen(security.key, 65);
            if( length < 8 || length > 64 )
            {
                NN_SDK_ASSERT(false, "security key length is out of range.\n");
                return ResultInvalidArgument();
            }
            if( length == 64 )
            {
                for(int i = 0; i < 64; i++)
                {
                    if( isxdigit(security.key[i]) == false )
                    {
                        NN_SDK_ASSERT(false, "security key (64 hexadecimal digits) accepts only [ 0~9 a~f A~F]\n");
                        return ResultInvalidArgument();
                    }
                }
            }
            break;
        }
        default:
            break;
        }

        WlanCommand* pcmdbuff;

        // I/FのUP要求
        // 既にUPされていたら、スルーされる
        result = RequestIfUpDown(1);
        if( result.IsFailure() )
        {
            return result;
        }

        // Set beacon lost timeout
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(int32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SET_BEACONLOST_TIMEOUT;
        int32_t* pTimeout = reinterpret_cast<int32_t*>(pcmdbuff->Args);
        *pTimeout = beaconLostTimeout;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        if( result.IsFailure() )
        {
            return result;
        }
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);

        // 接続試行
        WLAN_LOG_DEBUG("[%s] start to connect\n", __FUNCTION__);
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanConnectinoParameters));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_JOIN_NETWORK_STA;
        WlanConnectinoParameters* pConnectParam = reinterpret_cast<WlanConnectinoParameters*>(pcmdbuff->Args);

        // 接続パラメータのコピー
        pConnectParam->ssid.Set(ssid.data, ssid.length);
        pConnectParam->bssid.Set(bssid.data);
        pConnectParam->channel = channel;
        pConnectParam->security.privacyMode = static_cast<nn::wlan::SecurityMode>(security.privacyMode);
        pConnectParam->security.groupPrivacyMode = static_cast<nn::wlan::SecurityMode>(security.groupPrivacyMode);
        pConnectParam->security.keyIdx = static_cast<uint32_t>(security.keyIdx);
        memcpy(pConnectParam->security.key, security.key, sizeof(security.key));

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);

        return result;
    } //NOLINT(impl/function_size)

    nn::Result InfraApiImpl::CancelConnect() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_InfraIdle,
        };
        nn::Result result = CheckState(states, 1);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_CANCEL_JOIN_NETWORK;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::Disconnect() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta,
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_DISASSOCIATE;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::GetSystemEvent(nn::sf::Out<nn::sf::NativeHandle> pOutSystemEventReadableHandle, std::uint32_t type) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        if( type < detail::SystemEventType_ConnectionEvent || type > detail::SystemEventType_ConnectionComplete )
        {
            NN_SDK_ASSERT(false, "%s: type is invalid", __FUNCTION__);
            return ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Stop,
                WdrvMainState_Ready,
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta,
                WdrvMainState_Sleep
        };
        nn::Result result = CheckState(states, 5);
        if( result.IsFailure() )
        {
            return result;
        }

        pOutSystemEventReadableHandle.Set(
                nn::sf::NativeHandle(m_pStateMachine->GetSystemEventReadableHandle(static_cast<detail::SystemEventType>(type)),
                        false)
        );

        NN_RESULT_SUCCESS;
    }

    nn::Result InfraApiImpl::GetConnectionStatus(nn::sf::Out<nn::wlan::detail::SfdlConnectionStatus> pOutStatus) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Stop,
                WdrvMainState_Ready,
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta,
                WdrvMainState_Sleep
        };
        nn::Result result = CheckState(states, 5);
        if( result.IsFailure() )
        {
            return result;
        }

        // 出力用
        ConnectionStatus status;
        m_pStateMachine->GetConnectionStatus(&status);

        // コピー
        pOutStatus->state = static_cast<uint32_t>(status.state);
        pOutStatus->cause = static_cast<uint32_t>(status.cause);
        pOutStatus->channel = status.channel;
        std::memcpy(pOutStatus->bssid.data, status.bssid.GetMacAddressData(), MacAddress::MacAddressSize);
        pOutStatus->ssid.length = static_cast<uint8_t>(status.ssid.GetLength());
        std::memcpy(pOutStatus->ssid.data, status.ssid.GetSsidData(), status.ssid.GetLength());
        pOutStatus->aid = status.aid;
        pOutStatus->statusReasonCode = status.statusReasonCode;
        pOutStatus->capabilityInfo = status.capabilityInfo;
        pOutStatus->beaconInterval = status.beaconInterval;

        NN_RESULT_SUCCESS;
    }

    nn::Result InfraApiImpl::GetRssi(nn::sf::Out<std::int32_t> pOutRssi) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_InfraSta,
        };
        nn::Result result = CheckState(states, 1);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(int32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_RSSI;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        int32_t* pRssi = reinterpret_cast<int32_t*>(pcmdbuff->Args);
        NN_SDK_ASSERT_NOT_NULL(pRssi);
        pOutRssi.Set(*pRssi);
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::ChangeRxAntenna(std::int32_t rxAntenna) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        if( rxAntenna < RxAntennaPattern_0 || rxAntenna > RxAntennaPattern_Both )
        {
            NN_SDK_ASSERT(false, "%s: invalid argument", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta,
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(int32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SET_RX_CHAIN;
        int32_t* pRxAntenna = reinterpret_cast<int32_t*>(pcmdbuff->Args);
        *pRxAntenna = rxAntenna;

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::GetFwVersion(const nn::sf::OutBuffer& pOutBuffer) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        if( pOutBuffer.GetSize() < 256 )
        {
            return ResultInvalidArgument();
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanIoctlResult));
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_FW_VER;
        WlanIoctlResult* pOutResult = reinterpret_cast<WlanIoctlResult*>(pcmdbuff->Args);
        NN_SDK_ASSERT_NOT_NULL(pOutResult);
        pOutResult->pStr = new char[256];
        NN_SDK_ASSERT_NOT_NULL(pOutResult->pStr);
        pOutResult->size = 256;

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        auto result = pcmdbuff->nnResult;
        if( result.IsSuccess() )
        {
            std::memcpy(pOutBuffer.GetPointerUnsafe(), pOutResult->pStr, pOutResult->size);
        }

        delete[] pOutResult->pStr;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::RequestSleep() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        nn::Result result;

#if defined(ENABLE_DETECT_AGING_DEBUG)
        WlanCommand* pcmdbuff;
        // ステートチェック
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_STATE;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        WlanStates* state = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
        WdrvMainState main = state->mainState;
        WdrvSubState  sub  = state->subState;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        switch( main )
        {
        case WdrvMainState_OutOfService:
        case WdrvMainState_Ready:
            // 通常スリープに入ることになる
            break;
        case WdrvMainState_Sleep:
            // 既にスリープ状態なので成功を返しておく
            NN_RESULT_SUCCESS;
            break;
        case WdrvMainState_DetectIdle:
            // すれちがい予約がされているかチェック
            switch( m_pStateMachine->GetDetectSleepCondition() )
            {
            case DetectSleepCondition_Ok:
                // すれちがいスリープに入る
                WLAN_LOG_INFO("%s: Go to detect sleep.\n", __FUNCTION__);

                // interfaceのUPを要求する
                NN_ABORT_UNLESS_RESULT_SUCCESS(RequestIfUpDown(1));
                // 受信に関係する無線パラメータをセットする. SAモード
                NN_ABORT_UNLESS_RESULT_SUCCESS(SetDetectRecvConfig(ActionFrameRecvMode_DetectSleep));
                break;
            case DetectSleepCondition_NoData:
                // すれちがい予約されているが、送信データがセットされていない
                return ResultNotSetDetectData();
                break;
            case DetectSleepCondition_NotReserved:
                // 通常スリープに入る
                WLAN_LOG_INFO("%s: Go to normal sleep.\n", __FUNCTION__);
                break;
            default:
                // あり得ない
                NN_ABORT("%s: GetDetectSleepCondition() returned unexpected value\n", __FUNCTION__);
            }
            break;
        default:
            return ResultInvalidState();
        }

        // すれちがいスリープを要求。
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_REQUEST_SLEEP_FOR_ND;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        return result;
#else
        // WPA GTKのセット
        result = SetGtkInfo();
        if( ResultFailedToSetGtk().Includes(result) )
        {
            // WoWLスリープに入る条件が整っていたのにGTKのセットに失敗。
            // APから切断しておく。
            result = Disconnect();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            result = CloseMode();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        // GTKセット後のこのタイミングで切断が発生してしまうと、GTKを登録しっぱなしになってしまう。
        // StateMachineにフラグを持たせることで、登録しっぱなしが検知された場合にGTKクリアを行えるように対処した。

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

        pcmdbuff->id = WLAN_REQUEST_SLEEP;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        if( ResultSleepFailure().Includes(result) == true )
        {
            // WoWLスリープ遷移に失敗。APから切断し、無線機能をシャットダウンする
            WLAN_LOG_INFO("Failed to enter WoWL sleep mode. Start to enter normal sleep mode.\n");
            // GTK情報のクリア
            ClearGtkInfo();
            result = Disconnect();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            result = CloseMode();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            // Readyステートへの遷移が完了しているはずなので再びスリープ要求。
            result = RequestSleep();
        }

        return result;
#endif
    }

    nn::Result InfraApiImpl::RequestWakeUp() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        nn::Result result;
        WlanCommand* pcmdbuff;

#if defined(ENABLE_DETECT_AGING_DEBUG)
        // ステートチェック
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_STATE;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        WlanStates* state = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
        WdrvMainState main = state->mainState;
        WdrvSubState  sub  = state->subState;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        switch( main )
        {
        case WdrvMainState_OutOfService:
            WLAN_LOG_ERROR("Wifi driver is in abnormal state or wlan is disabled.\n");
            result = ResultWlanDeviceAbnormal();
            break;
        case WdrvMainState_Sleep:
            switch( sub )
            {
            case WdrvSubState_None:
                // スリープからの復帰処理可能
                break;
            default:
                return ResultInvalidState();
            }
            break;
        case WdrvMainState_Ready:
        case WdrvMainState_DetectIdle:
        case WdrvMainState_Detect:
            return ResultAlreadyAwake();
            break;
        default:
            return ResultInvalidState();
        }

        // 起床を要求
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_REQUEST_WAKEUP_FOR_ND;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        // result値で起床後の処理を分岐
        if( result.IsSuccess() )
        {
            // キャッシュされたAFを吸い上げるまでちょっと待っておく
            // TODO イベント等でちゃんと待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            m_pStateMachine->ClearAfCache();

            // スリープ前はIdle状態だったので、その状態に戻しておく
            // 受信に関係する無線パラメータをセットする.
            NN_ABORT_UNLESS_RESULT_SUCCESS(SetDetectRecvConfig(ActionFrameRecvMode_Normal));
            // interfaceのDOWNを要求する
            NN_ABORT_UNLESS_RESULT_SUCCESS(RequestIfUpDown(0));
        }
        else if( ResultWakeupNormalSleep().Includes(result) == true )
        {
            // 無線ドライバーの初期化
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
            NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_INITIALIZE;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            result = pcmdbuff->nnResult;
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
            if( result.IsFailure() )
            {
                if( ResultWlanDeviceAbnormal().Includes(result) == true )
                {
                    // 無線ドライバーの初期化に失敗している。
                    result = ResultWlanDeviceAbnormal();
                }
                return result;
            }

            // Sleep前のステートに戻しておく
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
            NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_GET_STATE_BEFORE_SLEEP;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            WlanStates* statebfs = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

            switch( statebfs->mainState )
            {
            case WdrvMainState_Ready:
                return ResultSuccess();
            case WdrvMainState_DetectIdle:
                NN_ABORT_UNLESS_RESULT_SUCCESS(OpenDetectModeWithChannel(0));
                break;
            default:
                NN_ABORT_UNLESS(false, "WlanMainStateBeforeSleep is wrong.");
            }
        }
        return result;
#else
        // ステートチェック
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_STATE;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        WlanStates* state = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
        WdrvMainState main = state->mainState;
        WdrvSubState  sub  = state->subState;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        switch( main )
        {
        case WdrvMainState_OutOfService:
            WLAN_LOG_ERROR("Wifi driver is in abnormal state or wlan is disabled.\n");
            return ResultSuccess();
            break;
        case WdrvMainState_Sleep:
            switch( sub )
            {
            case WdrvSubState_None:
                // スリープからの復帰処理可能
                break;
            default:
                return ResultInvalidState();
            }
            break;
        case WdrvMainState_Ready:
        case WdrvMainState_InfraIdle:
        case WdrvMainState_InfraSta:
            return ResultAlreadyAwake();
            break;
        default:
            return ResultInvalidState();
        }

        // 起床を要求
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_REQUEST_WAKEUP;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        // result値で起床後の処理を分岐
        if( result.IsSuccess() )
        {
            // GTK情報のクリア
            ClearGtkInfo();
            WLAN_LOG_INFO("Wake up from WoWL sleep with AP connection.\n");
        }
        else if( ResultWakeupWithLinkDown().Includes(result) == true )
        {
            // GTK情報のクリア
            ClearGtkInfo();
            // APから切断または切断した方が良い理由での起床
            WLAN_LOG_INFO("Wake up from WoWL sleep. But need to disconnect from AP or already disconnected.\n");
            result = Disconnect();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);  // WPAサプリカントが未接続状態での切断依頼も正常にこなしてくれるか要確認
            // TODO 切断が理由の起床であったことを上位層に伝える。CauseOfInfoに入れれると良いが。
        }
        else if( ResultWakeupNormalSleep().Includes(result) == true )
        {
            // 無線ドライバーの初期化
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
            NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_INITIALIZE;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            result = pcmdbuff->nnResult;
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
            if( result.IsFailure() )
            {
                if( ResultWlanDeviceAbnormal().Includes(result) == true )
                {
                    // 無線ドライバーの初期化に失敗している。
                    // ここからは無線機能は使えなくなるが、上位層のResultハンドリングに合わせるために、
                    // ResultSuccessを返しておく必要がある。
                    result = ResultSuccess();
                }
                return result;
            }

            // Sleep前のステートに戻しておく
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
            NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_GET_STATE_BEFORE_SLEEP;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            WlanStates* statebfs = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

            switch( statebfs->mainState )
            {
            case WdrvMainState_Ready:
                return ResultSuccess();
            case WdrvMainState_InfraIdle:
                if( statebfs->subState == WdrvSubState_None )
                {
                    // OpenMode直後の状態に戻す
                    result = OpenMode();
                }
                else
                {
                    NN_ABORT_UNLESS(false, "WlanSubStateBeforeSleep is wrong.");
                }
                break;
            default:
                NN_ABORT_UNLESS(false, "WlanMainStateBeforeSleep is wrong.");
            }
        }
        return result;
#endif
    } //NOLINT(impl/function_size)

    nn::Result InfraApiImpl::ConnectWithWps(std::uint32_t method, const nn::sf::InArray<char>& wpsPinArray, int32_t beaconLostTimeout) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

#if !defined(USE_WPA_SUPPLICANT)
        return nn::wlan::ResultNotImplemented();
#endif

        WlanCommand* pcmdbuff;

        // I/FのUP要求
        // 既にUPされていたら、スルーされる
        RequestIfUpDown(1);

        // Set beacon lost timeout
        if( beaconLostTimeout < 1 || beaconLostTimeout > 30 )
        {
            NN_SDK_ASSERT(false, "beaconLostTimeout is out of range @ %s\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(int32_t));
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SET_BEACONLOST_TIMEOUT;
        int32_t* pTimeout = reinterpret_cast<int32_t*>(pcmdbuff->Args);
        *pTimeout = beaconLostTimeout;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        if( pcmdbuff->nnResult.IsFailure() )
        {
            return pcmdbuff->nnResult;
        }
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);

        // 接続試行
        if( !(static_cast<WpsMethod>(method) == WpsMethod_Pbc || static_cast<WpsMethod>(method) == WpsMethod_Pin) )
        {
            NN_SDK_ASSERT(false, "Wps method is out of range @ %s\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( static_cast<WpsMethod>(method) == WpsMethod_Pin )
        {
            if( wpsPinArray.GetLength() < WpsPinLength + 1 )
            {
                NN_SDK_ASSERT(false, "WpsPin size is out of range @ %s\n", __FUNCTION__);
                return nn::wlan::ResultInvalidArgument();
            }
            if( nn::util::Strnlen(wpsPinArray.GetData(), WpsPinLength + 1) != WpsPinLength )
            {
                NN_SDK_ASSERT(false, "Wps pin length is not %d @ %s\n", WpsPinLength, __FUNCTION__);
                return nn::wlan::ResultInvalidArgument();
            }
        }
        WLAN_LOG_DEBUG("[%s] start to connect with WPS\n", __FUNCTION__);
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanWpsParameters));
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_JOIN_NETWORK_STA_WITH_WPS;
        WlanWpsParameters* pWpsParam = reinterpret_cast<WlanWpsParameters*>(pcmdbuff->Args);

        // 接続情報のコピー
        pWpsParam->method = static_cast<WpsMethod>(method);
        if( pWpsParam->method == WpsMethod_Pin )
        {
            std::memcpy(pWpsParam->pin, wpsPinArray.GetData(), WpsPinLength);
            pWpsParam->pin[WpsPinLength] = '\0';
            WLAN_LOG_INFO("WPS PIN : %s\n", pWpsParam->pin);
        }

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff(pcmdbuff);

        return result;
    }

    nn::Result InfraApiImpl::GetWakeupReason(nn::sf::Out<std::uint32_t> pOutReason) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta
        };
        nn::Result result = CheckState(states, 3);
        if( result.IsFailure() )
        {
            pOutReason.Set(static_cast<uint32_t>(WowlWakeReason_Nothing));
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(uint32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_WAKEUP_REASON;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        uint32_t* pReason = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
        NN_SDK_ASSERT_NOT_NULL(pReason);
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        // WowlWakeupReasonに変換
        WowlWakeReason wakeReason = m_pStateMachine->ConvertWakeupReason(*pReason);
        pOutReason.Set(static_cast<uint32_t>(wakeReason));

        return result;
    }

    nn::Result InfraApiImpl::SetTcpSessionInformation(nn::wlan::detail::SfdlMacAddress dstMac, nn::wlan::WlanIpv4Address srcIp,
            nn::wlan::WlanIpv4Address dstIp, std::uint16_t srcPort, std::uint16_t dstPort, std::uint32_t ackNum, std::uint16_t windowSize) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        WlanWowlSetupParams params = {
                MacAddress(dstMac.data),   // ownMac
                srcIp,                     // source ip
                dstIp,                     // destination ip
                srcPort,                   // source port
                dstPort,                   // destination port
                ackNum,                    // ack number
                windowSize                 // window size
        };

        auto result = m_pStateMachine->RegisterTcpSessionInfo(params);
        return result;
    }

    nn::Result InfraApiImpl::RemoveTcpSessionInformation() NN_NOEXCEPT
    {
        auto result = m_pStateMachine->RemoveTcpSessionInfo();
        return result;
    }

    nn::Result InfraApiImpl::GetWakeupReasonRaw(nn::sf::Out<std::uint32_t> pOutReason) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta
        };
        nn::Result result = CheckState(states, 3);
        if( result.IsFailure() )
        {
            pOutReason.Set(0);
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(uint32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_WAKEUP_REASON;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        uint32_t* pReason = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
        NN_SDK_ASSERT_NOT_NULL(pReason);
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        pOutReason.Set(*pReason);

        return result;
    }

    nn::Result InfraApiImpl::SetWakeupReasonRaw(uint32_t reason) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta
        };
        nn::Result result = CheckState(states, 3);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(uint32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SET_WAKEUP_REASON;
        uint32_t* pReason = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
        *pReason = reason;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::EnableWowlFeatures(std::uint32_t features) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_InfraIdle,
                WdrvMainState_InfraSta
        };
        nn::Result result = CheckState(states, 3);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(uint32_t));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_ENABLE_WOWL_FEATURES;
        uint32_t* pFeatures = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
        *pFeatures = features;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::GetWowlStats(nn::sf::Out<nn::wlan::WowlWakeCount> pOutCounts, nn::sf::Out<nn::wlan::WowlSleepStats> pOutStats) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        nn::Result result;

        WlanCommand* pcmdbuff;

        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WowlWakeCount));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_WOWL_WAKE_COUNT;
        WowlWakeCount* pCounts = reinterpret_cast<WowlWakeCount*>(pcmdbuff->Args);
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        if( result.IsSuccess() )
        {
            WowlWakeCount counts;
            std::memcpy(&counts, pCounts, sizeof(WowlWakeCount));
            pOutCounts.Set(counts);
        }
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        if( result.IsFailure() )
        {
            return result;
        }

        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WowlSleepStats));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_WOWL_SLEEP_STATS;
        WowlSleepStats* pStats = reinterpret_cast<WowlSleepStats*>(pcmdbuff->Args);
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        if( result.IsSuccess() )
        {
            WowlSleepStats stats;
            std::memcpy(&stats, pStats, sizeof(WowlSleepStats));
            pOutStats.Set(stats);
        }
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::ClearWowlStats(bool counts, bool stats) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        WlanCommand* pcmdbuff;

        if( counts == true )
        {
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
            NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_CLEAR_WOWL_WAKE_COUNT;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        }

        if( stats == true )
        {
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
            NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_CLEAR_WOWL_SLEEP_STATS;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result InfraApiImpl::SetGtkInfo() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_InfraSta
        };
        nn::Result result = CheckState(states, 1);
        if( result.IsFailure() )
        {
            return result;
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_SET_GTK_INFO;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        bool commandResult = pcmdbuff->Result;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        if( ResultInvalidState().Includes(result) == true )
        {
            return result;
        }

        if( result.IsSuccess() && commandResult == false )
        {
            // GTKセットに失敗。
            WLAN_LOG_ERROR("[InfraImpl]Failed to set WPA GTK for wowl.\n");
            result = ResultFailedToSetGtk();
        }

        return result;
    }

    nn::Result InfraApiImpl::ClearGtkInfo() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_CLEAR_GTK_INFO;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        bool commandResult = pcmdbuff->Result;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        if( ResultInvalidState().Includes(result) == true )
        {
            return result;
        }

        if( result.IsSuccess() && commandResult == false )
        {
            // APから切断しておいた方が良い？
        }

        return result;
    }

    nn::Result InfraApiImpl::InitializeWlanOnSkipBoot() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_INITIALIZE_ON_SKIPBOOT;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::EmulateDriverInitFail() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_EMULATE_DRIVER_INIT_FAIL;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result InfraApiImpl::GetAllowedChannels(const nn::sf::OutArray<std::int16_t>& outArray, nn::sf::Out<std::uint32_t> pOutCount) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        uint32_t count;
        m_pStateMachine->GetAllowedChannels(outArray.GetData(), outArray.GetLength(), &count);
        pOutCount.Set(count);

        NN_RESULT_SUCCESS;
    }

#if defined(ENABLE_DETECT_AGING_DEBUG)
    nn::Result InfraApiImpl::OpenDetectModeWithChannel(std::uint16_t channel) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        if( channel < WirelessChannel_1ch || channel > WirelessChannel_165ch )
        {
            NN_SDK_REQUIRES(false, "%s: channel is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_DetectIdle
        };
        NN_RESULT_DO(CheckState(states, 2));

        // Homeチャンネルの設定
        m_pStateMachine->SetDetectHomeChannel(static_cast<uint32_t>(channel));

        // 上記で設定したホームチャンネルでオープンされる
        NN_RESULT_DO(ChangeMode(WLAN_CHANGE_STATE_DETECT));
        NN_RESULT_SUCCESS;
    }

    nn::Result InfraApiImpl::SetDetectRecvConfig(ActionFrameRecvMode mode) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( mode < ActionFrameRecvMode_Normal || mode >= ActionFrameRecvMode_Num )
        {
            NN_SDK_REQUIRES(false, "%s: mode is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(ActionFrameRecvMode));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        ActionFrameRecvMode* pMode = static_cast<ActionFrameRecvMode*>(pcmdbuff->Args);
        NN_ABORT_UNLESS_NOT_NULL(pMode);
        *pMode = mode;
        pcmdbuff->id = WLAN_SET_DETECT_RECV_CONF;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }
#endif
}}} // end of namespace nn::wlan::server

