﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/mbuf/mbuf_MbufInit.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_MacAddress.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_StateDefinition.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/wlan/driver/wlan_DriverTypes.h>

#include "wlan_DetectApiImpl.h"
#include "../wlan_MsgBuffer.h"
#include "../wlan_SignalStrength.h"
#include "../driver/wlan_Driver.h"
#include "../wlan_MemoryInit.h"

namespace nn { namespace wlan { namespace server
{
    nn::Result DetectApiImpl::OpenMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        return OpenModeWithChannel(static_cast<uint16_t>(DetectHomeChannel));
    }

    nn::Result DetectApiImpl::OpenModeWithChannel(std::uint16_t channel) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // チャンネルチェック。0の場合は現在セットされているチャンネルを使うようにする。
        if( channel > WirelessChannel_165ch )
        {
            NN_SDK_REQUIRES(false, "%s: channel is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_DetectIdle
        };
        NN_RESULT_DO(CheckState(states, 2));

        // Homeチャンネルの設定
        m_pStateMachine->SetDetectHomeChannel(static_cast<uint32_t>(channel));

        // 上記で設定したホームチャンネルでオープンされる
        NN_RESULT_DO(ChangeMode(WLAN_CHANGE_STATE_DETECT));
        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::CloseMode() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Ready,
                WdrvMainState_DetectIdle
        };
        NN_RESULT_DO(CheckState(states, 2));

        NN_RESULT_DO(ChangeMode(WLAN_CHANGE_STATE_READY));
        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::CreateRxEntryForActionFrame(nn::sf::Out<std::uint32_t> pOutRxId,
                const nn::sf::InArray<std::uint16_t>& pSubtypeArray, std::uint32_t capacity) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( pSubtypeArray.GetLength() > ActionFrameTypeCountMax )
        {
            NN_SDK_REQUIRES(false, "The number of ActionFrameType must be less than %d\n", ActionFrameTypeCountMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }
        if( capacity > ActionFrameRxCapacityMax )
        {
            NN_SDK_REQUIRES(false, "Capacity of RxEntry must be less than %d\n", ActionFrameRxCapacityMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }
        for( int i = 0; i < pSubtypeArray.GetLength(); i++ )
        {
            if( pSubtypeArray.GetData()[i] < ActionFrameType_Beacon || pSubtypeArray.GetData()[i] >= ActionFrameType_End )
            {
                return nn::wlan::ResultInvalidArgument();
            }
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        uint32_t rxId; // 出力用変数
        result = m_pStateMachine->AddRxEntryForActionFrame(&rxId, capacity, pSubtypeArray.GetData(), pSubtypeArray.GetLength(), true);  // RingBufferの上書き可能にしておく。つまり満杯の場合、古いものから上書きされる。
        if( result.IsSuccess() )
        {
            pOutRxId.Set(rxId);
        }

        return result;
    }

    nn::Result DetectApiImpl::DeleteRxEntryForActionFrame(std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        result = m_pStateMachine->RemoveRxEntryForActionFrame(rxId);

        return result;
    }

    nn::Result DetectApiImpl::AddSubtypeToRxEntryForActionFrame(std::uint32_t rxId, std::uint32_t type) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        if( type < ActionFrameType_Beacon || type >= ActionFrameType_End )
        {
            NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        result = m_pStateMachine->AddSubtypeToRxEntryForActionFrame(rxId, static_cast<uint16_t>(type));

        return result;
    }

    nn::Result DetectApiImpl::DeleteSubtypeFromRxEntryForActionFrame(nn::sf::Out<std::uint32_t> pOutRxId, std::uint32_t type) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        if( type < ActionFrameType_Beacon || type >= ActionFrameType_End )
        {
            NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        uint32_t rxId; // 出力用変数
        result = m_pStateMachine->RemoveSubtypeFromRxEntryForActionFrame(&rxId, static_cast<uint16_t>(type));
        if( result.IsSuccess() )
        {
            pOutRxId.Set(rxId);
        }

        return result;
    }

    nn::Result DetectApiImpl::CancelGetActionFrame(std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        nn::Result result = CheckState(states, 2);
        if( result.IsFailure() )
        {
            return result;
        }

        result = m_pStateMachine->CancelRxActionFrame(rxId);
        return result;
    }

    nn::Result DetectApiImpl::StartCommunication() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle
        };
        NN_RESULT_DO(CheckState(states, 1));
        // interfaceのUPを要求する
        NN_RESULT_DO(RequestIfUpDown(1));
        // 受信に関係する無線パラメータをセットする
        NN_RESULT_DO(SetDetectRecvConfig(ActionFrameRecvMode_DetectWake));
        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::StopCommunication() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 1));
        // 受信に関係する無線パラメータをリセットする
        NN_RESULT_DO(SetDetectRecvConfig(ActionFrameRecvMode_Normal));
        // interfaceのDOWNを要求する
        NN_RESULT_DO(RequestIfUpDown(0));
        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::SetMacAddress(nn::wlan::detail::SfdlMacAddress macAddr) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle
        };
        NN_RESULT_DO(CheckState(states, 1));

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(MacAddress));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        MacAddress* pMac = static_cast<MacAddress*>(pcmdbuff->Args);
        NN_ABORT_UNLESS_NOT_NULL(pMac);
        pcmdbuff->id = WLAN_SET_MACADDRESS;
        pMac->Set(&macAddr.data[0]);
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result DetectApiImpl::PutActionFrameOneShot(nn::wlan::detail::SfdlMacAddress macAddr,
            std::uint8_t subtype, DetectHash hash, const nn::sf::InBuffer& pTxData, std::uint32_t dwellTime) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( std::memcmp(macAddr.data, MacAddress::CreateZeroMacAddress().GetMacAddressData(), MacAddress::MacAddressSize) == 0 )
        {
            NN_SDK_REQUIRES(false, "%s: macAddr must not be Zero mac address\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( subtype != ActionFrameType_Detect )
        {
            NN_SDK_REQUIRES(false, "%s: subtype must be ActionFrameType_Detect\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( pTxData.GetSize() > DetectActionFramePayloadMax )
        {
            NN_SDK_REQUIRES(false, "%s: pTxData.size is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( dwellTime > 300 )
        {
            NN_SDK_REQUIRES(false, "%s: dwellTime must be less than 301\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 1));

        nn::Result result;
        WlanCommand* pcmdbuff;
        bool isRetried = false;
        while( 1 )
        {
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanOneShotActionFrameInfo));
            NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
            WlanOneShotActionFrameInfo* pInfo = static_cast<WlanOneShotActionFrameInfo*>(pcmdbuff->Args);
            pInfo->pData = new char[(sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader) + pTxData.GetSize())];
            NN_ABORT_UNLESS_NOT_NULL(pInfo->pData);
            // Data格納
            {
                NintendoActionFrameHeader ninAfHeader = {
                        0x7f,
                        { 0x00, 0x22, 0xAA },
                        subtype,
                        0
                };
                DetectHeader dHeader = {
                        DhpMajor,
                        DhpMiner,
                        DhpCmdOneShot,
                        0,
                        hash
                };
                std::memcpy(&pInfo->pData[0], &ninAfHeader, sizeof(NintendoActionFrameHeader));
                std::memcpy(&pInfo->pData[sizeof(NintendoActionFrameHeader)], &dHeader, sizeof(DetectHeader));
                std::memcpy(&pInfo->pData[sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader)],
                        const_cast<char*>(pTxData.GetPointerUnsafe()), pTxData.GetSize());
            }
            pInfo->size = sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader) + pTxData.GetSize();
            std::memcpy(&pInfo->dstMac[0], &macAddr.data[0], MacAddress::MacAddressSize);
            std::memcpy(&pInfo->bssid[0], &MacAddress::CreateZeroMacAddress().GetMacAddressData()[0], MacAddress::MacAddressSize);
            pInfo->channel = m_pStateMachine->GetDetectHomeChannel();
            pInfo->dwellTime = dwellTime;
            pcmdbuff->id = WLAN_PUT_ONE_SHOT_ACTION_FRAME;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            result = pcmdbuff->nnResult;
            delete pInfo->pData;
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

            if( ResultNoMemory().Includes(result) == true && isRetried == false )
            {
                // 400us待って1度だけリトライ
                nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(400));
                isRetried = true;
            }
            else
            {
                break;
            }
        }

        return result;
    }

    nn::Result DetectApiImpl::StartPeriodicActionFrame(std::uint8_t subtype, DetectHash hash,
            const nn::sf::InBuffer& pTxData, std::uint32_t interval) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( subtype != ActionFrameType_Detect )
        {
            NN_SDK_REQUIRES(false, "%s: subtype must be ActionFrameType_Detect\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( pTxData.GetSize() > DetectActionFramePayloadMax )
        {
            NN_SDK_REQUIRES(false, "%s: pTxData.size is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( interval > PeriodicActionFrameIntervalMax )
        {
            NN_SDK_REQUIRES(false, "%s: interval is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 1));

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanPeriodicActionFrameInfo));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        WlanPeriodicActionFrameInfo* pInfo = static_cast<WlanPeriodicActionFrameInfo*>(pcmdbuff->Args);
        pInfo->pData = new char[(sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader) + pTxData.GetSize())];
        NN_ABORT_UNLESS_NOT_NULL(pInfo->pData);
        memset(&pInfo->pData[0], 0xCC, sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader) + pTxData.GetSize());
        // Data格納
        {
            NintendoActionFrameHeader ninAfHeader = {
                    0x7f,
                    { 0x00, 0x22, 0xAA },
                    subtype,
                    0
            };
            DetectHeader dHeader = {
                    DhpMajor,
                    DhpMiner,
                    DhpCmdPeriodic,
                    0,
                    hash
            };
            std::memcpy(&pInfo->pData[0], &ninAfHeader, sizeof(NintendoActionFrameHeader));
            std::memcpy(&pInfo->pData[sizeof(NintendoActionFrameHeader)], &dHeader, sizeof(DetectHeader));
            std::memcpy(&pInfo->pData[sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader)],
                    const_cast<char*>(pTxData.GetPointerUnsafe()), pTxData.GetSize());
        }
        pInfo->size = sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader) + pTxData.GetSize();
        pInfo->interval = interval;
        pcmdbuff->id = WLAN_START_PERIODIC_ACTION_FRAME;

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        delete pInfo->pData;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result DetectApiImpl::CancelPeriodicActionFrame() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 1));

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_CANCEL_PERIODIC_ACTION_FRAME;

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result DetectApiImpl::ReserveDetectSleep() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 2));

        m_pStateMachine->ReserveDetectSleep(true);
        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::CancelDetectSleep() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 2));

        m_pStateMachine->ReserveDetectSleep(false);
        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::SetActionFrameForSleep(std::uint8_t subtype, nn::wlan::DetectHash hash, const nn::sf::InBuffer& pTxData) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( subtype != ActionFrameType_Detect )
        {
            NN_SDK_REQUIRES(false, "%s: subtype must be ActionFrameType_Detect\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( pTxData.GetSize() > DetectActionFramePayloadMax )
        {
            NN_SDK_REQUIRES(false, "%s: pTxData.size is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 2));

        m_pStateMachine->SetActionFrameForDetectSleep(subtype, hash, pTxData.GetPointerUnsafe(), pTxData.GetSize());

        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::SetHashList(const nn::sf::InBuffer& pHashList) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        if( pHashList.GetSize() > DetectSleepFilterMax * sizeof(uint64_t) )
        {
            NN_SDK_REQUIRES(false, "%s: hash size is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 2));

        return m_pStateMachine->SetHashListForDetectSleep(pHashList.GetPointerUnsafe(), pHashList.GetSize());
    }

    nn::Result DetectApiImpl::SetPeriodicActionFrameCycle(const nn::wlan::DetectPeriodicAfCycle& pattern, uint8_t target) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // TODO パラメータチェック

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 2));

        if( target == DetectPeriodicAfCycleTarget_Hd || target == DetectPeriodicAfCycleTarget_Sa || target == DetectPeriodicAfCycleTarget_Both )
        {
            m_pStateMachine->SetPeriodicActionFrameCycle(pattern, static_cast<DetectPeriodicAfCycleTarget>(target));
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::RequestSleep() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        nn::Result result;
        WlanCommand* pcmdbuff;

        // ステートチェック
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_STATE;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        WlanStates* state = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
        WdrvMainState main = state->mainState;
        WdrvSubState  sub  = state->subState;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        switch( main )
        {
        case WdrvMainState_OutOfService:
        case WdrvMainState_Ready:
            // 通常スリープに入ることになる
            break;
        case WdrvMainState_Sleep:
            // 既にスリープ状態なので成功を返しておく
            NN_RESULT_SUCCESS;
            break;
        case WdrvMainState_DetectIdle:
            // すれちがい予約がされているかチェック
            switch( m_pStateMachine->GetDetectSleepCondition() )
            {
            case DetectSleepCondition_Ok:
                // すれちがいスリープに入る
                WLAN_LOG_INFO("%s: Go to detect sleep.\n", __FUNCTION__);

                // interfaceのUPを要求する
                NN_ABORT_UNLESS_RESULT_SUCCESS(RequestIfUpDown(1));
                // 受信に関係する無線パラメータをセットする. SAモード
                NN_ABORT_UNLESS_RESULT_SUCCESS(SetDetectRecvConfig(ActionFrameRecvMode_DetectSleep));
                break;
            case DetectSleepCondition_NoData:
                // すれちがい予約されているが、送信データがセットされていない
                return ResultNotSetDetectData();
                break;
            case DetectSleepCondition_NotReserved:
                // 通常スリープに入る
                WLAN_LOG_INFO("%s: Go to normal sleep.\n", __FUNCTION__);
                break;
            default:
                // あり得ない
                NN_ABORT("%s: GetDetectSleepCondition() returned unexpected value\n", __FUNCTION__);
            }
            break;
        default:
            return ResultInvalidState();
        }

        // すれちがいスリープを要求。
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_REQUEST_SLEEP_FOR_ND;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        return result;
    }

    nn::Result DetectApiImpl::RequestWakeUp() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        nn::Result result;
        WlanCommand* pcmdbuff;

        // ステートチェック
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_GET_STATE;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        WlanStates* state = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
        WdrvMainState main = state->mainState;
        WdrvSubState  sub  = state->subState;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        switch( main )
        {
        case WdrvMainState_OutOfService:
            WLAN_LOG_ERROR("Wifi driver is in abnormal state or wlan is disabled.\n");
            result = ResultWlanDeviceAbnormal();
            break;
        case WdrvMainState_Sleep:
            switch( sub )
            {
            case WdrvSubState_None:
                // スリープからの復帰処理可能
                break;
            default:
                return ResultInvalidState();
            }
            break;
        case WdrvMainState_Ready:
        case WdrvMainState_DetectIdle:
        case WdrvMainState_Detect:
            return ResultAlreadyAwake();
            break;
        default:
            return ResultInvalidState();
        }

        // 起床を要求
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_REQUEST_WAKEUP_FOR_ND;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        // result値で起床後の処理を分岐
        if( result.IsSuccess() )
        {
            // キャッシュされたAFを吸い上げるまでちょっと待っておく
            // TODO イベント等でちゃんと待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            m_pStateMachine->ClearAfCache();

            // スリープ前はIdle状態だったので、その状態に戻しておく
            // 受信に関係する無線パラメータをセットする.
            NN_ABORT_UNLESS_RESULT_SUCCESS(SetDetectRecvConfig(ActionFrameRecvMode_Normal));
            // interfaceのDOWNを要求する
            NN_ABORT_UNLESS_RESULT_SUCCESS(RequestIfUpDown(0));
        }
        else if( ResultWakeupNormalSleep().Includes(result) == true )
        {
            // 無線ドライバーの初期化
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
            NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_INITIALIZE;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            result = pcmdbuff->nnResult;
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
            if( result.IsFailure() )
            {
                if( ResultWlanDeviceAbnormal().Includes(result) == true )
                {
                    // 無線ドライバーの初期化に失敗している。
                    result = ResultWlanDeviceAbnormal();
                }
                return result;
            }

            // Sleep前のステートに戻しておく
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanStates));
            NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
            pcmdbuff->id = WLAN_GET_STATE_BEFORE_SLEEP;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            WlanStates* statebfs = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

            switch( statebfs->mainState )
            {
            case WdrvMainState_Ready:
                return ResultSuccess();
            case WdrvMainState_DetectIdle:
                NN_ABORT_UNLESS_RESULT_SUCCESS(OpenModeWithChannel(0));
                break;
            default:
                NN_ABORT_UNLESS(false, "WlanMainStateBeforeSleep is wrong.");
            }
        }
        return result;
    }

    nn::Result DetectApiImpl::PutActionFrameOneShotEx(nn::wlan::detail::SfdlMacAddress macAddr, nn::wlan::detail::SfdlMacAddress bssid,
            std::uint8_t subtype, const nn::wlan::DetectHeader& dhp, const nn::sf::InBuffer& pTxData, std::uint32_t dwellTime) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( std::memcmp(macAddr.data, MacAddress::CreateZeroMacAddress().GetMacAddressData(), MacAddress::MacAddressSize) == 0 )
        {
            NN_SDK_REQUIRES(false, "%s: macAddr must not be Zero mac address\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( subtype != ActionFrameType_Detect )
        {
            NN_SDK_REQUIRES(false, "%s: subtype must be ActionFrameType_Detect\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( pTxData.GetSize() > DetectActionFramePayloadMax )
        {
            NN_SDK_REQUIRES(false, "%s: pTxData.size is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( dwellTime > 300 )
        {
            NN_SDK_REQUIRES(false, "%s: dwellTime must be less than 301\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 1));

        nn::Result result;
        WlanCommand* pcmdbuff;
        bool isRetried = false;
        while( 1 )
        {
            pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanOneShotActionFrameInfo));
            NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
            WlanOneShotActionFrameInfo* pInfo = static_cast<WlanOneShotActionFrameInfo*>(pcmdbuff->Args);
            pInfo->pData = new char[(sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader) + pTxData.GetSize())];
            NN_ABORT_UNLESS_NOT_NULL(pInfo->pData);
            // Data格納
            {
                NintendoActionFrameHeader ninAfHeader = {
                        0x7f,
                        { 0x00, 0x22, 0xAA },
                        subtype,
                        0
                };
                std::memcpy(&pInfo->pData[0], &ninAfHeader, sizeof(NintendoActionFrameHeader));
                std::memcpy(&pInfo->pData[sizeof(NintendoActionFrameHeader)], &dhp, sizeof(DetectHeader));
                std::memcpy(&pInfo->pData[sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader)],
                        const_cast<char*>(pTxData.GetPointerUnsafe()), pTxData.GetSize());
            }
            pInfo->size = sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader) + pTxData.GetSize();
            std::memcpy(&pInfo->dstMac[0], &macAddr.data[0], MacAddress::MacAddressSize);
            std::memcpy(&pInfo->bssid[0], &bssid.data[0], MacAddress::MacAddressSize);
            pInfo->channel = m_pStateMachine->GetDetectHomeChannel();
            pInfo->dwellTime = dwellTime;
            pcmdbuff->id = WLAN_PUT_ONE_SHOT_ACTION_FRAME;
            m_pStateMachine->PostCommandMessage(pcmdbuff);
            result = pcmdbuff->nnResult;
            delete pInfo->pData;
            m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

            if( ResultNoMemory().Includes(result) == true && isRetried == false )
            {
                // 400us待って1度だけリトライ
                nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(400));
                isRetried = true;
            }
            else
            {
                break;
            }
        }

        return result;
    }

    nn::Result DetectApiImpl::SetDetectRecvConfig(ActionFrameRecvMode mode) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( mode < ActionFrameRecvMode_Normal || mode >= ActionFrameRecvMode_Num )
        {
            NN_SDK_REQUIRES(false, "%s: mode is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(ActionFrameRecvMode));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        ActionFrameRecvMode* pMode = static_cast<ActionFrameRecvMode*>(pcmdbuff->Args);
        NN_ABORT_UNLESS_NOT_NULL(pMode);
        *pMode = mode;
        pcmdbuff->id = WLAN_SET_DETECT_RECV_CONF;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }

    nn::Result DetectApiImpl::GetActionFrame(nn::sf::Out<nn::wlan::detail::SfdlMacAddress> pOutMacAddr,
            const nn::sf::OutBuffer& pOutRxData, nn::sf::Out<std::uint32_t> pOutSize, std::uint32_t rxId,
            nn::sf::Out<std::uint16_t> pOutChannel, nn::sf::Out<std::int16_t> pOutRssi, nn::sf::Out<std::int64_t> pOutTick) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 2));

        nn::mbuf::Mbuf* pOutMbuf;
        nn::Result result = m_pStateMachine->PullRxActionFrame(&pOutMbuf, rxId);
        size_t dataSize = 0;

        if( result.IsSuccess() )
        {
            NN_SDK_ASSERT_NOT_NULL(pOutMbuf);

            // データコピー
            // mbufから上位層に渡すべきデータの抽出
            uint8_t* pData = reinterpret_cast<uint8_t*>(nn::mbuf::MbufTod(pOutMbuf));

            // 送信元MACアドレスの抽出
            driver::GetPeerMacAddress(pOutMacAddr->data, nn::mbuf::MbufTod(pOutMbuf));

            // 出力先バッファの容量が足りているかチェック
            dataSize = driver::GetActionFrameDataSizeEx(nn::mbuf::MbufTod(pOutMbuf));
            WLAN_LOG_DEBUG("[%s] action frame size : %d\n", __FUNCTION__, dataSize);

            // 本体コピー
            if( pOutRxData.GetSize() >=  dataSize )
            {
                std::memcpy(pOutRxData.GetPointerUnsafe(), &pData[driver::offsetToDataEx], dataSize);
                WlanActionFrameRecvInfo* pInfo = reinterpret_cast<WlanActionFrameRecvInfo*>(pOutMbuf->_pGeneral);
                if( pInfo != NULL )
                {
                    pOutChannel.Set(pInfo->channel);
                    pOutRssi.Set(static_cast<int16_t>(pInfo->rssi));
                    pOutTick.Set(pInfo->sysTick);
                }
                else
                {
                    pOutChannel.Set(0);
                    pOutRssi.Set(-999);
                    pOutTick.Set(0);
                }
            }
            else
            {
                WLAN_LOG_DEBUG("Failed to copy rx data due to capacity shortage of out buffer\n");
                result = ResultBufferTooShort();
            }

            // コピー版なので、mbufはここで破棄
            if( pOutMbuf->_pGeneral != NULL )
            {
                nnwlanFreeNormal(pOutMbuf->_pGeneral);
            }
            nn::mbuf::MbufFreem(pOutMbuf);
        }

        pOutSize.Set(dataSize);
        return result;
    }

    nn::Result DetectApiImpl::GetAllowedChannels(const nn::sf::OutArray<std::int16_t>& outArray, nn::sf::Out<std::uint32_t> pOutCount) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        uint32_t count;
        m_pStateMachine->GetAllowedChannels(outArray.GetData(), outArray.GetLength(), &count);
        pOutCount.Set(count);

        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::GetTotalRecvCountInSleep(nn::sf::Out<std::uint64_t> pOutCount) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 2));

        uint64_t count;
        count = m_pStateMachine->GetDetectSaTotalRecvCnt();
        pOutCount.Set(count);

        NN_RESULT_SUCCESS;
    }

    nn::Result DetectApiImpl::ClearTotalRecvCountInSleep() NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        // ステートチェック
        WdrvMainState states[] = {
                WdrvMainState_DetectIdle,
                WdrvMainState_Detect
        };
        NN_RESULT_DO(CheckState(states, 2));

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(0);
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        pcmdbuff->id = WLAN_CLEAR_DETECT_SA_TOTAL_RECV_CNT;
        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        return result;
    }

}}} // end of namespace nn::wlan::server

