﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdio>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>

#include "wlan_Htcs.h"
#include "../wlan_MemoryInit.h"

namespace nn{ namespace wlan{ namespace dbg{

/* WlanHtcs Library Initialize --------------------------------------------- */
bool WlanHtcs::WlanHtcsInit() NN_NOEXCEPT
{
    return true;
}

/* WlanHtcs Main process --------------------------------------------------- */
void WlanHtcs::WlanHtcsMain() NN_NOEXCEPT
{
    ConsoleServerInit();

    while( AcceptWait() )
    {
        ConsoleMain();
    }

    CloseSocket();
}

/* Console close request --------------------------------------------------- */
void WlanHtcs::CloseConsole() NN_NOEXCEPT
{
    NN_SDK_LOG("Console close request\n");
    m_MainLoopAbortFlag = 1;
}

/* Send string buffer to host ---------------------------------------------- */
int WlanHtcs::SendStrings(const char* str, int length) NN_NOEXCEPT
{
    int ret;

    ret = nn::htcs::Send(m_PeerSocket, (void*)str, length, 0);

    return ret;
}

/* Initial socket ---------------------------------------------------------- */
void WlanHtcs::ConsoleServerInit() NN_NOEXCEPT
{
    // サービスを登録
    m_ServiceAddr.family = nn::htcs::HTCS_AF_HTCS;
    m_ServiceAddr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(m_ServiceAddr.portName.name, "WLANConsole");

    while( (m_ServiceSocket = nn::htcs::Socket()) < 0 )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }
    nn::htcs::Bind(m_ServiceSocket, &m_ServiceAddr);
}

/* Accept wait ------------------------------------------------------------- */
bool WlanHtcs::AcceptWait() NN_NOEXCEPT
{
    nn::htcs::Listen(m_ServiceSocket, 1);
    // 接続を待ち受ける
    NN_SDK_LOG("Waiting for connection from host.\n");
    m_PeerSocket = nn::htcs::Accept(m_ServiceSocket, nullptr);
    if (m_PeerSocket < 0)
    {
        return false;
    }

    Fcntl(m_PeerSocket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK);

    return true;
}

/* Parse CR/LF and replace terminate --------------------------------------- */
static int ReplaceCrLf(char* buff, int len) NN_NOEXCEPT
{
    int n;
    char c;

    /* CR/LF to terminate char */
    for( n = 0; n < len; n++ ){
        c = buff[n];
        if((c == '\0') || (c == '\n') || (c == '\r')){
            buff[n] = '\0';
            break;
        }
    }

    /* Length is not include CR/LF/NULL */
    return n;
}

/* Console Main ------------------------------------------------------------ */
void WlanHtcs::ConsoleMain() NN_NOEXCEPT
{
    int send_len, recv_len;
    int last_err;

    NN_SDK_LOG("%s\n", __FUNCTION__);

    m_MainLoopAbortFlag = 0;

    /* 改行コードの送信 */
    send_len = SendStrings(BootMsg, std::strlen(BootMsg));
    if( send_len < 0 )
    {
        NN_SDK_LOG("send error %d\n", nn::htcs::GetLastError());
        CloseConsole();
    }

    while(m_MainLoopAbortFlag == 0)
    {
        send_len = SendStrings(ConsoleMsg, std::strlen(ConsoleMsg));
        if (send_len < 0)
        {
            CloseConsole();
        }

        recv_len = 0;
        while( (recv_len <= 0) && (m_MainLoopAbortFlag == 0) )
        {
            recv_len = nn::htcs::Recv(m_PeerSocket,
                                      (void*)m_InputBuffer,
                                      (m_ConsoleBufferLength - 3),
                                      0);
            if( recv_len <= 0 )
            {
                if( recv_len < 0 )
                {
                    last_err = nn::htcs::GetLastError();
                    if( last_err != nn::htcs::HTCS_EWOULDBLOCK )
                    {
                        NN_SDK_LOG("Recv error %d\n", nn::htcs::GetLastError());
                        CloseConsole();
                    }
                }
                else
                {
                    NN_SDK_LOG("gracefully closed\n");
                    CloseConsole();
                }
            }
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
        }


        if( recv_len > 0 )
        {
            /* send CRLF */
            send_len = SendStrings(CrLf, std::strlen(CrLf));
            if( send_len > 0 )
            {

                recv_len = ReplaceCrLf(m_InputBuffer, recv_len);
                if( recv_len > 0 )
                {
                    InputCallBack(m_InputBuffer, recv_len);
                }
            }
            else
            {
                CloseConsole();
            }
        }
    }
}

/* Clse socket ------------------------------------------------------------- */
void WlanHtcs::CloseSocket() NN_NOEXCEPT
{
    // 切断
    TerminateCallback();

    nn::htcs::Close(m_PeerSocket);
    nn::htcs::Close(m_ServiceSocket);
    NN_SDK_LOG("Closed.\n");
}

}}} // namespace
