﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <new>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ISettingsServer.sfdl.h>
#include <nn/settings/settings_IFirmwareDebugSettingsServer.sfdl.h>
#include <nn/settings/settings_IFactorySettingsServer.sfdl.h>
#include <nn/settings/settings_ISystemSettingsServer.sfdl.h>
#include <nn/settings/settings_ServiceName.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include "settings_FirmwareDebugSettingsServer.h"
#include "settings_FactorySettingsServer.h"
#include "settings_SettingsServer.h"
#include "settings_Spl.h"
#include "settings_StaticObject.h"
#include "settings_SystemSettingsServer.h"

namespace {

//!< 拡張ヒープに割り当てるメモリブロックのバイトサイズ
const size_t HeapMemorySize = 256 << 10;

//!< セッション数の最大値
const int SessionCountMax = 60;

//!< ポート数の最大値
const int PortCountMax = 4;

//!< サーバマネージャのオプション宣言です。
struct ServerManagerOption
{
    static const size_t PointerTransferBufferSize = 256;
};

//!< サーバマネージャを扱うためのクラスです。
class ServerManager
    : public ::nn::sf::HipcSimpleAllInOneServerManager<
        SessionCountMax, PortCountMax, ServerManagerOption>
{
};

//!< 指定されたサイズのメモリ領域を確保します。
void* Allocate(size_t size) NN_NOEXCEPT;

//!< 指定されたメモリ領域を開放します。
void Deallocate(void* p, size_t size) NN_NOEXCEPT;

//!< fs の Result ハンドラを設定します。
void SetFsResultHandler() NN_NOEXCEPT;

//!< ServerManager に SettingsServer を登録します。
void RegisterSettingsServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;

//!< ServerManager に FactorySettingsServer を登録します。
void RegisterFactorySettingsServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;

//!< ServerManager に FirmwareDebugSettingsServer を登録します。
void RegisterFirmwareDebugSettingsServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;

//!< ServerManager に SystemSettingsServer を登録します。
void RegisterSystemSettingsServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;

} // namespace

void* operator new(size_t count)
{
    return Allocate(count);
}

void* operator new(size_t count, const ::std::nothrow_t& tag) NN_NOEXCEPT
{
    NN_UNUSED(tag);
    return Allocate(count);
}

void operator delete(void* ptr) NN_NOEXCEPT
{
    Deallocate(ptr, 0);
}

void* operator new[](size_t count)
{
    return Allocate(count);
}

void* operator new[](size_t count, const ::std::nothrow_t& tag) NN_NOEXCEPT
{
    NN_UNUSED(tag);
    return Allocate(count);
}

void operator delete[](void* ptr) NN_NOEXCEPT
{
    Deallocate(ptr, 0);
}

extern "C" void nninitStartup()
{
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    ::nn::os::ThreadType* const pCurrentThread = ::nn::os::GetCurrentThread();
    ::nn::os::ChangeThreadPriority(
        pCurrentThread, NN_SYSTEM_THREAD_PRIORITY(settings, Main));
    ::nn::os::SetThreadNamePointer(
        pCurrentThread, NN_SYSTEM_THREAD_NAME(settings, Main));

    ::nn::fs::InitializeWithMultiSessionForSystem();
    ::nn::fs::SetAllocator(Allocate, Deallocate);

    SetFsResultHandler();

    ServerManager& serverManager =
        ::nn::settings::detail::StaticObject<ServerManager>::Get();

    RegisterSettingsServerWithServerManager(&serverManager);

    RegisterFactorySettingsServerWithServerManager(&serverManager);

    if (::nn::settings::detail::IsSplDevelopment())
    {
        RegisterFirmwareDebugSettingsServerWithServerManager(&serverManager);
    }

    RegisterSystemSettingsServerWithServerManager(&serverManager);

    ::nn::os::ChangeThreadPriority(
        pCurrentThread, NN_SYSTEM_THREAD_PRIORITY(settings, IpcServer));
    ::nn::os::SetThreadNamePointer(
        pCurrentThread, NN_SYSTEM_THREAD_NAME(settings, IpcServer));

    serverManager.Start();

    serverManager.LoopAuto();
}

namespace {

//!< 拡張ヒープのハンドルを返します。
::nn::lmem::HeapHandle& GetHeapHandle() NN_NOEXCEPT
{
    static ::nn::lmem::HeapHandle s_HeapHandle = nullptr;

    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));

    if (!s_IsInitialized)
    {
        static ::nn::os::SdkMutexType s_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsInitialized)
        {
            static char s_HeapMemory[HeapMemorySize] = {};

            s_HeapHandle = ::nn::lmem::CreateExpHeap(
                s_HeapMemory,
                sizeof(s_HeapMemory),
                ::nn::lmem::CreationOption_NoOption);

            s_IsInitialized = true;
        }
    }

    return s_HeapHandle;
}

void* Allocate(size_t size) NN_NOEXCEPT
{
    return ::nn::lmem::AllocateFromExpHeap(GetHeapHandle(), size);
}

void Deallocate(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    ::nn::lmem::FreeToExpHeap(GetHeapHandle(), p);
}

void SetFsResultHandler() NN_NOEXCEPT
{
    ::nn::fs::SetDefaultFsContextResultHandler([](
        ::nn::Result result) NN_NOEXCEPT
    {
        if (::nn::fs::ResultMmcAccessFailed::Includes(result))
        {
            return ::nn::fs::AbortSpecifier::ReturnResult;
        }

        if (::nn::fs::ResultDataCorrupted::Includes(result))
        {
            return ::nn::fs::AbortSpecifier::ReturnResult;
        }

        if (::nn::fs::ResultFatFsUnexpected::Includes(result))
        {
            return ::nn::fs::AbortSpecifier::ReturnResult;
        }

        return ::nn::fs::AbortSpecifier::Default;
    });
}

void RegisterSettingsServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    ::nn::sf::SharedPointer<
        ::nn::settings::ISettingsServer> pSettingsServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::settings::detail::CreateSettingsServerProxy(
            &pSettingsServer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pSettingsServer, SessionCountMax,
            ::nn::settings::SettingsServiceName));
}

void RegisterFactorySettingsServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<
        ::nn::settings::IFactorySettingsServer> pFactorySettingsServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::settings::detail::CreateFactorySettingsServerProxy(
            &pFactorySettingsServer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pFactorySettingsServer, SessionCountMax,
            ::nn::settings::FactorySettingsServiceName));
}

void RegisterFirmwareDebugSettingsServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<
        ::nn::settings::IFirmwareDebugSettingsServer
        > pFirmwareDebugSettingsServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::settings::detail::CreateFirmwareDebugSettingsServerProxy(
            &pFirmwareDebugSettingsServer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pFirmwareDebugSettingsServer, SessionCountMax,
            ::nn::settings::FirmwareDebugSettingsServiceName));
}

void RegisterSystemSettingsServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<
        ::nn::settings::ISystemSettingsServer> pSystemSettingsServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::settings::detail::CreateSystemSettingsServerProxy(
            &pSystemSettingsServer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pSystemSettingsServer, SessionCountMax,
            ::nn::settings::SystemSettingsServiceName));
}

} // namespace
