﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/nn_SdkLog.h>
#include <nn/ro/detail/ro_DebugMonitorInterface.h>
#include <nn/ro/detail/ro_RoInterface.h>
#include <nn/ro/detail/ro_PortName.h>
#include <nn/spl/spl_Api.h>

#include "ro_DebugMonitor.h"
#include "ro_RoServer.h"
#include "ro_Random.h"

namespace
{
    enum PortIndex
    {
        PortIndex_DebugMonitor,
        PortIndex_RoManager,
        PortIndex_Count
    };

    static const int DebugMonitorSessionCount   = 2;
    static const int RoManagerSessionCount      = 32;
    static const int TotalSessionCount = DebugMonitorSessionCount + RoManagerSessionCount;

    struct ServerManagerOptions
    {
        static const size_t PointerTransferBufferSize = 1024;
    };

    class ServerManager : public nn::sf::HipcSimpleAllInOneServerManager<TotalSessionCount, PortIndex_Count, ServerManagerOptions>
    {
    private:
        virtual nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT;
    };


    NN_ALIGNAS(64) nn::Bit8 g_DefaultServerAllocatorBuffer[16 * 1024];

    nn::sf::ExpHeapAllocator  g_DefaultServerAllocator;
    ServerManager g_ServerManager;



    nn::Result ServerManager::OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT
    {
        switch(portIndex)
        {
        case PortIndex_DebugMonitor:
            {
                nn::sf::SharedPointer<nn::ro::detail::IDebugMonitorInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::ro::detail::IDebugMonitorInterface,
                    nn::ro::DebugMonitor
                >(&g_DefaultServerAllocator);

                return this->AcceptImpl(pPort, refInterface);
            }
            break;

        case PortIndex_RoManager:
            {
                nn::sf::SharedPointer<nn::ro::detail::IRoInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::ro::detail::IRoInterface,
                    nn::ro::RoServer
                >(&g_DefaultServerAllocator);

                return this->AcceptImpl(pPort, refInterface);
            }
            break;


        default:
            NN_ABORT();
        }
    }
}

extern "C" void nndiagStartup()
{
}

extern "C" void nninitStartup()
{
}

extern "C" void nnMain() NN_NOEXCEPT
{
    auto heapHandle = nn::lmem::CreateExpHeap(&g_DefaultServerAllocatorBuffer, sizeof(g_DefaultServerAllocatorBuffer), nn::lmem::CreationOption_NoOption);
    g_DefaultServerAllocator.Attach(heapHandle);

    nn::Result result;

    nn::spl::Initialize();
    nn::ro::RoServer::SetDevelopmentHardware(nn::spl::IsDevelopment());
    nn::ro::RoServer::SetDevelopmentFunctionEnabled(nn::spl::GetConfigBool(nn::spl::ConfigItem_IsDevelopmentFunctionEnabled));
    nn::spl::Finalize();
    nn::ro::random::Initialize();

    result = g_ServerManager.InitializePort(PortIndex_DebugMonitor,     DebugMonitorSessionCount,   nn::ro::detail::PortNameForDebugMonitor);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    result = g_ServerManager.InitializePort(PortIndex_RoManager,        RoManagerSessionCount,      nn::ro::detail::PortNameForRo);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    g_ServerManager.Start();

    g_ServerManager.LoopAuto();
}
