﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <atomic>
#include <mutex>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/sdmmc/sdmmc_Common.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/usb/usb_Device.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/bpc/bpc_BoardPowerControl.h>
#include <nn/psm/psm_Api.h>
#include <nn/psm/psm_System.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

namespace {

::nn::usb::DsClient                  dsClient;
::nn::usb::DsInterface               dsInterface;
::nn::usb::DsEndpoint                dsEndpoints[2];


::nn::usb::UsbStringDescriptor languageStringDescriptor            = {4,   ::nn::usb::UsbDescriptorType_String, {0x0409}};
::nn::usb::UsbStringDescriptor manufacturerStringDescriptor        = {18,  ::nn::usb::UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o'}};
::nn::usb::UsbStringDescriptor productStringDescriptor             = {40,  ::nn::usb::UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o', 'S', 'd', 'k', 'D', 'e', 'b', 'u', 'g', 'g', 'e', 'r'}};
::nn::usb::UsbStringDescriptor serialNumberStringDescriptor        = {26,  ::nn::usb::UsbDescriptorType_String, {'S', 'e', 'r', 'i', 'a', 'l', 'N', 'u', 'm', 'b', 'e', 'r'}};
::nn::usb::UsbStringDescriptor interfaceStringDescriptor           = {26,  ::nn::usb::UsbDescriptorType_String, {'r', 'e', 'c', 'o', 'v', 'e', 'r', 'y', ' ', 'u', 's', 'b'}};


::nn::usb::UsbDeviceDescriptor fullSpeedDeviceDescriptor =
{
    ::nn::usb::UsbDescriptorSize_Device,                     // bLength
    ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
    0x0110,                                                  // bcdUSB 1.1
    0x00,                                                    // bDeviceClass
    0x00,                                                    // bDeviceSubClass
    0x00,                                                    // bDeviceProtocol
    0x40,                                                    // bMaxPacketSize0
    0x057e,                                                  // idVendor
    0x3000,                                                  // idProduct
    0x0100,                                                  // bcdDevice
    0x01,                                                    // iManufacturer
    0x02,                                                    // iProduct
    0x03,                                                    // iSerialNumber
    0x01                                                     // bNumConfigurations
};


::nn::usb::UsbDeviceDescriptor highSpeedDeviceDescriptor =
{
    ::nn::usb::UsbDescriptorSize_Device,                     // bLength
    ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
    0x0200,                                                  // bcdUSB 2.0
    0x00,                                                    // bDeviceClass
    0x00,                                                    // bDeviceSubClass
    0x00,                                                    // bDeviceProtocol
    0x40,                                                    // bMaxPacketSize0
    0x057e,                                                  // idVendor
    0x3000,                                                  // idProduct
    0x0100,                                                  // bcdDevice
    0x01,                                                    // iManufacturer
    0x02,                                                    // iProduct
    0x03,                                                    // iSerialNumber
    0x01                                                     // bNumConfigurations
};


::nn::usb::UsbDeviceDescriptor superSpeedDeviceDescriptor =
{
    ::nn::usb::UsbDescriptorSize_Device,                     // bLength
    ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
    0x0300,                                                  // bcdUSB 3.0
    0x00,                                                    // bDeviceClass
    0x00,                                                    // bDeviceSubClass
    0x00,                                                    // bDeviceProtocol
    0x09,                                                    // bMaxPacketSize0, SS 512 or 2^9
    0x057e,                                                  // idVendor
    0x3000,                                                  // idProduct
    0x0100,                                                  // bcdDevice
    0x01,                                                    // iManufacturer
    0x02,                                                    // iProduct
    0x03,                                                    // iSerialNumber
    0x01                                                     // bNumConfigurations
};


uint8_t binaryObjectStore[] =
{
    0x05,                                                    // bLength
    ::nn::usb::UsbDescriptorType_Bos,                        // bDescriptorType
    0x16,0x00,                                               // Length of this descriptor and all sub descriptors
    0x02,                                                    // Number of device capability descriptors

    // USB 2.0 extension
    0x07,                                                    // bLength
    ::nn::usb::UsbDescriptorType_DeviceCapability,           // bDescriptorType
    0x02,                                                    // USB 2.0 extension capability type
    0x02,0x00,0x00,0x00,                                     // Supported device level features: LPM support

    // SuperSpeed device capability
    0x0A,                                                    // bLength
    ::nn::usb::UsbDescriptorType_DeviceCapability,           // bDescriptorType
    0x03,                                                    // SuperSpeed device capability type
    0x00,                                                    // Supported device level features
    0x0e,0x00,                                               // Speeds supported by the device : SS, HS
    0x03,                                                    // Functionality support
    0x00,                                                    // U1 Device Exit latency
    0x00,0x00                                                // U2 Device Exit latency
};


::nn::usb::UsbInterfaceDescriptor             usbInterfaceDescriptor = {
    ::nn::usb::UsbDescriptorSize_Interface  ,                // bLength
    ::nn::usb::UsbDescriptorType_Interface  ,                // bDescriptorType
    0,                                                       // bInterfaceNumber
    0,                                                       // bAlternateSetting
    2,                                                       // bNumEndpoints
    0xff,                                                    // bInterfaceClass
    0xff,                                                    // bInterfaceSubClass
    0xff,                                                    // bInterfaceProtocol
    4,                                                       // iInterface
};


::nn::usb::UsbEndpointDescriptor             usbEndpointDescriptorsFullSpeed[2] = {
    {
        ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
        ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
        0x81,                                                // bEndpointAddress
        ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
        64,                                                  // wMaxPacketSize
        0                                                    // bInterval
    },
    {
        ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
        ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
        0x01,                                                // bEndpointAddress
        ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
        64,                                                  // wMaxPacketSize
        0                                                    // bInterval
    },
};


::nn::usb::UsbEndpointDescriptor             usbEndpointDescriptorsHighSpeed[2] = {
    {
        ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
        ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
        0x81,                                                // bEndpointAddress
        ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
        512,                                                 // wMaxPacketSize
        0                                                    // bInterval
    },
    {
        ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
        ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
        0x01,                                                // bEndpointAddress
        ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
        512,                                                 // wMaxPacketSize
        0                                                    // bInterval
    },
};


::nn::usb::UsbEndpointDescriptor             usbEndpointDescriptorsSuperSpeed[2] = {
    {
        ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
        ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
        0x81,                                                // bEndpointAddress
        ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
        1024,                                                // wMaxPacketSize
        0                                                    // bInterval
    },
    {
        ::nn::usb::UsbDescriptorSize_Endpoint,               // bLength
        ::nn::usb::UsbDescriptorType_Endpoint,               // bDescriptorType
        0x01,                                                // bEndpointAddress
        ::nn::usb::UsbEndpointAttributeMask_XferTypeBulk,    // bmAttributes
        1024,                                                // wMaxPacketSize
        0                                                    // bInterval
    },
};


// Use this endpoint companion descriptor for both endpoints :)
::nn::usb::UsbEndpointCompanionDescriptor    usbSuperSpeedUsbEndpointCompanionDescriptor =
{
    0x06,                                                    // bLength
    ::nn::usb::UsbDescriptorType_EndpointCompanion,          // bDescriptorType
    15,                                                      // bMaxBurst
    0,                                                       // bmAttributes
    0                                                        // wBytesPerInterval
};


struct DsHeader
{

    uint32_t                   begin;
    uint32_t                   bytes;
    uint32_t                   end;

};

const int g_BufferSize = 4096;
NN_USB_DMA_ALIGN char g_SendBuffer[g_BufferSize];
NN_USB_DMA_ALIGN char g_ReceiveBuffer[g_BufferSize];

void MakeDsHeader(DsHeader* pHeader, int bytesSend) NN_NOEXCEPT
{
    pHeader->begin = 0x12345678;
    pHeader->bytes = bytesSend;
    pHeader->end = 0xabcdef;
}

::nn::lmem::HeapHandle& GetHeapHandle() NN_NOEXCEPT;

void* Allocate(size_t size) NN_NOEXCEPT;

void Deallocate(void* p, size_t size) NN_NOEXCEPT;

//!< BasicLockable な MutexType を表す構造体です。
struct LockableMutexType final
{
    ::nn::os::MutexType _mutex;

    void lock() NN_NOEXCEPT
    {
        ::nn::os::LockMutex(&_mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        ::nn::os::UnlockMutex(&_mutex);
    }
};

const size_t BctSize = 16 * 1024;
const size_t EksSize = 0x200;
const int64_t BctAddressBase = 0x00000000;
const int64_t BctAddressOffset = 0x00004000;
const int64_t EksAddressBase = 0x00180000;
const int64_t EksAddressOffset = 0x00000200;
const int64_t WriteParameterOffset = 0x10000;
const uint64_t StorageOffsetBase = 0x00000;
const uint64_t SystemPartitionOffset = 0x308000000;
const uint64_t BootLoaderOffset = 0x100000;
const uint64_t MbrGptOffset = 0x10C00;
const uint64_t MbrGptSize = 17 * 1024;
const uint64_t Package2OffsetBase = 0x4000;

struct WriteParameter
{
    uint32_t value[8];
};

// BLバージョンデータ
const size_t BlVersionDataBuildDateTimeStringSize = 15;
const uint64_t BlVersionBuildDateOffset = 0x10;
const char UsableBlVersionDateTimeString[] = "20161109210334"; // BLバージョンが1のBLで最も古い、2016-1109-2103 のビルド日時

} // namespace

void* operator new(size_t size)
{
    return Allocate(size);
}

void operator delete(void* p) NN_NOEXCEPT
{
    Deallocate(p, 0);
}

extern "C" void nninitStartup()
{
}

extern "C" void nndiagStartup()
{
}

int32_t CalculateEksIndex(int32_t bootLoaderVersion) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(0 <= bootLoaderVersion && bootLoaderVersion <= 32);

    if(bootLoaderVersion == 0)
    {
        return 0;
    }
    else
    {
        return bootLoaderVersion - 1;
    }
}

::nn::Result ReadBct(void* bct, size_t size, nn::fs::IStorage *pStorage, int64_t index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(bct);
    NN_SDK_REQUIRES_EQUAL(size, BctSize);

    NN_RESULT_DO(
        pStorage->Read(BctAddressBase + BctAddressOffset * index, bct, BctSize));

    NN_RESULT_SUCCESS;
}

::nn::Result ReadEks(void* eks, size_t size, nn::fs::IStorage *pStorage, int64_t index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(eks);
    NN_SDK_REQUIRES_EQUAL(size, EksSize);

    NN_RESULT_DO(
        pStorage->Read(EksAddressBase + EksAddressOffset * index, eks, EksSize));

    NN_RESULT_SUCCESS;
}

::nn::Result WriteBct(nn::fs::IStorage *pStorage, int64_t index, void *buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_EQUAL(size, BctSize);

    NN_RESULT_DO(
        pStorage->Write(BctAddressBase + BctAddressOffset * index, buffer, size));

    NN_RESULT_SUCCESS;
}

::nn::Result ReadBootLoaderVersion(int32_t* pOut, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_EQUAL(size, BctSize);

    *pOut = *reinterpret_cast<int32_t*>(reinterpret_cast<uint8_t*>(buffer) + 0x2330);

    NN_ABORT_UNLESS(0 <= *pOut && *pOut <= 32);

    NN_RESULT_SUCCESS;
}

::nn::Result UpdateSecureInfo(void *pOutBctBuffer, void *pSourceEksBuffer) NN_NOEXCEPT
{
    const size_t CopySize = 0xB0;
    const int64_t SourceEksAddressBegin = 0x0000;
    const int64_t DestinationEksAddressBegin = 0x0450;

    ::std::memcpy(reinterpret_cast<uint8_t*>(pOutBctBuffer) + DestinationEksAddressBegin,
                  reinterpret_cast<uint8_t*>(pSourceEksBuffer) + SourceEksAddressBegin,
                  CopySize);

    NN_RESULT_SUCCESS;
}

::nn::Result UpdateSecureInfo(int index, ::nn::Bit8* bctBuffer) NN_NOEXCEPT
{
    ::std::unique_ptr<::nn::fs::IStorage> pBootPartition1;
    static uint8_t eksBuffer[EksSize];

    NN_RESULT_DO(
        ::nn::fs::OpenBisPartition(&pBootPartition1, ::nn::fs::BisPartitionId::BootPartition1Root));

    int32_t bootLoaderVersion;
    NN_RESULT_DO(
        ReadBootLoaderVersion(&bootLoaderVersion, bctBuffer, BctSize));

    int32_t eksIndex = CalculateEksIndex(bootLoaderVersion);

    NN_RESULT_DO(
        ReadEks(eksBuffer, EksSize, pBootPartition1.get(), eksIndex));

    NN_RESULT_DO(
        UpdateSecureInfo(bctBuffer, eksBuffer));

    NN_SDK_LOG("Parameter: bctIndex=%lld, bootLoaderVersion=%d, eksIndex=%d\n",
               index, bootLoaderVersion, eksIndex);

    NN_RESULT_DO(
        WriteBct(pBootPartition1.get(), index, bctBuffer, BctSize));

    NN_RESULT_SUCCESS;
}

::nn::Result InvalidateBct(int index) NN_NOEXCEPT
{
    ::std::unique_ptr<::nn::fs::IStorage> pBootPartition1;
    NN_RESULT_DO(
        ::nn::fs::OpenBisPartition(&pBootPartition1, ::nn::fs::BisPartitionId::BootPartition1Root));

    // BCT をつぶす（512バイト０埋めで書き込む）
    const size_t FillSize = 512;
    std::unique_ptr<uint8_t[]> fillData(new uint8_t[FillSize]);
    std::memset(fillData.get(), 0, FillSize);
    NN_RESULT_DO(
        pBootPartition1->Write(BctAddressBase + BctAddressOffset * index, fillData.get(), FillSize));

    NN_RESULT_SUCCESS;
}

::nn::Result ReadSafeBootLoaderBuildDateTime(char* bootloaderBuildDateTime) NN_NOEXCEPT
{
    ::std::unique_ptr<::nn::fs::IStorage> pBootPartition1;
    NN_RESULT_DO(
        ::nn::fs::OpenBisPartition(&pBootPartition1, ::nn::fs::BisPartitionId::BootPartition2Root));

    // セクタサイズ単位で読む
    const size_t readSize = nn::sdmmc::SectorSize;
    std::unique_ptr<char[]> readBuffer(new char[readSize]);

    NN_RESULT_DO(
        pBootPartition1->Read(0, readBuffer.get(), readSize));

    nn::util::Strlcpy(bootloaderBuildDateTime, readBuffer.get() + BlVersionBuildDateOffset, BlVersionDataBuildDateTimeStringSize);

    NN_RESULT_SUCCESS;
}

::nn::Result ReadBootLoaderBuildDateTimeInReceivedImage(char* bootloaderBuildDateTime, ::nn::Bit8* blBuffer)
{
    NN_SDK_REQUIRES_NOT_NULL(blBuffer);

    nn::util::Strlcpy(bootloaderBuildDateTime, reinterpret_cast<char*>(blBuffer + BlVersionBuildDateOffset), BlVersionDataBuildDateTimeStringSize);

    NN_RESULT_SUCCESS;
}

::nn::Result UsbInitialize() NN_NOEXCEPT
{
    NN_SDK_LOG("[recovery] USB Initialize \n");

    // device
    NN_RESULT_DO(dsClient.Initialize(nn::usb::ComplexId_Tegra21x));
    NN_RESULT_DO(dsClient.ClearDeviceData());

    uint8_t stringIndex;

    // String index will be in order starting from 0 after clearing all the device data
    NN_RESULT_DO(dsClient.AddUsbStringDescriptor(&stringIndex, &languageStringDescriptor));
    NN_RESULT_DO(dsClient.AddUsbStringDescriptor(&stringIndex, &manufacturerStringDescriptor));
    NN_RESULT_DO(dsClient.AddUsbStringDescriptor(&stringIndex, &productStringDescriptor));
    NN_RESULT_DO(dsClient.AddUsbStringDescriptor(&stringIndex, &serialNumberStringDescriptor));
    NN_RESULT_DO(dsClient.AddUsbStringDescriptor(&stringIndex, &interfaceStringDescriptor));

    // Device descriptor for all speeds
    NN_RESULT_DO(dsClient.SetUsbDeviceDescriptor(&fullSpeedDeviceDescriptor,   ::nn::usb::UsbDeviceSpeed_Full));
    NN_RESULT_DO(dsClient.SetUsbDeviceDescriptor(&highSpeedDeviceDescriptor,   ::nn::usb::UsbDeviceSpeed_High));
    NN_RESULT_DO(dsClient.SetUsbDeviceDescriptor(&superSpeedDeviceDescriptor,  ::nn::usb::UsbDeviceSpeed_Super));
    NN_RESULT_DO(dsClient.SetBinaryObjectStore(binaryObjectStore, sizeof(binaryObjectStore)));

    // Interface, configuration data per interface for all speeds
    NN_RESULT_DO(dsInterface.Initialize(&dsClient, usbInterfaceDescriptor.bInterfaceNumber));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Full, &usbInterfaceDescriptor, sizeof(::nn::usb::UsbInterfaceDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Full, &usbEndpointDescriptorsFullSpeed[0], sizeof(::nn::usb::UsbEndpointDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Full, &usbEndpointDescriptorsFullSpeed[1], sizeof(::nn::usb::UsbEndpointDescriptor)));

    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_High, &usbInterfaceDescriptor, sizeof(::nn::usb::UsbInterfaceDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_High, &usbEndpointDescriptorsHighSpeed[0], sizeof(::nn::usb::UsbEndpointDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_High, &usbEndpointDescriptorsHighSpeed[1], sizeof(::nn::usb::UsbEndpointDescriptor)));

    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbInterfaceDescriptor, sizeof(::nn::usb::UsbInterfaceDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbEndpointDescriptorsSuperSpeed[0], sizeof(::nn::usb::UsbEndpointDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbSuperSpeedUsbEndpointCompanionDescriptor, sizeof(::nn::usb::UsbEndpointCompanionDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbEndpointDescriptorsSuperSpeed[1], sizeof(::nn::usb::UsbEndpointDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbSuperSpeedUsbEndpointCompanionDescriptor, sizeof(::nn::usb::UsbEndpointCompanionDescriptor)));

    // Endpoints
    NN_RESULT_DO(dsEndpoints[0].Initialize(&dsInterface, 0x81));
    NN_RESULT_DO(dsEndpoints[1].Initialize(&dsInterface, 0x01));

    // This will enale the device
    NN_RESULT_DO(dsInterface.Enable());
    NN_RESULT_DO(dsClient.EnableDevice());

    NN_SDK_LOG("[recovery] USB Initialize Success\n");

    NN_RESULT_SUCCESS;
}

::nn::Result SendConfigurationId1() NN_NOEXCEPT
{
    uint32_t bytesSend = sizeof(::nn::settings::factory::ConfigurationId1);
    uint32_t bytesTransferred = 0;

    MakeDsHeader(reinterpret_cast<DsHeader *>(g_SendBuffer), bytesSend);

    NN_RESULT_DO(dsEndpoints[0].PostBuffer(&bytesTransferred, g_SendBuffer, sizeof(DsHeader)));

    ::nn::settings::factory::GetConfigurationId1(
        reinterpret_cast<::nn::settings::factory::ConfigurationId1 *>(g_SendBuffer));

    NN_SDK_LOG("[recovery] Send: %s\n", g_SendBuffer);
    NN_RESULT_DO(
        dsEndpoints[0].PostBuffer(&bytesTransferred,
                                  g_SendBuffer,
                                  sizeof(::nn::settings::factory::ConfigurationId1)));

    NN_RESULT_SUCCESS;
}

::nn::Result SendSerialNumber() NN_NOEXCEPT
{
    uint32_t bytesSend = sizeof(::nn::settings::factory::SerialNumber);
    uint32_t bytesTransferred = 0;

    MakeDsHeader(reinterpret_cast<DsHeader *>(g_SendBuffer), bytesSend);

    NN_RESULT_DO(dsEndpoints[0].PostBuffer(&bytesTransferred, g_SendBuffer, sizeof(DsHeader)));

    ::nn::settings::factory::GetSerialNumber(
        reinterpret_cast<::nn::settings::factory::SerialNumber *>(g_SendBuffer));

    NN_SDK_LOG("[recovery] Send: %s\n", g_SendBuffer);
    NN_RESULT_DO(dsEndpoints[0].PostBuffer(&bytesTransferred,
                                           g_SendBuffer,
                                           sizeof(::nn::settings::factory::SerialNumber)));

    NN_RESULT_SUCCESS;
}

::nn::Result SendFirmwareVersion() NN_NOEXCEPT
{
    uint32_t bytesSend = sizeof(::nn::settings::system::FirmwareVersion::displayName);
    uint32_t bytesTransferred = 0;

    MakeDsHeader(reinterpret_cast<DsHeader *>(g_SendBuffer), bytesSend);

    NN_RESULT_DO(dsEndpoints[0].PostBuffer(&bytesTransferred, g_SendBuffer, sizeof(DsHeader)));

    ::nn::settings::system::FirmwareVersion firmwareVersion;
    ::nn::settings::system::GetFirmwareVersion(&firmwareVersion);

    memcpy(g_SendBuffer, firmwareVersion.displayName, sizeof(nn::settings::system::FirmwareVersion::displayName));

    NN_SDK_LOG("[recovery] Send: %s\n", g_SendBuffer);
    NN_RESULT_DO(dsEndpoints[0].PostBuffer(&bytesTransferred,
                                           g_SendBuffer,
                                           sizeof(::nn::settings::system::FirmwareVersion::displayName)));

    NN_RESULT_SUCCESS;
}

::nn::Result SendBatteryChargePercentage() NN_NOEXCEPT
{
    int64_t battery = ::nn::psm::GetBatteryChargePercentage();
    uint32_t bytesSend = sizeof(battery);
    uint32_t bytesTransferred = 0;

    MakeDsHeader(reinterpret_cast<DsHeader *>(g_SendBuffer), bytesSend);
    NN_RESULT_DO(dsEndpoints[0].PostBuffer(&bytesTransferred, g_SendBuffer, sizeof(DsHeader)));

    int sendSize = nn::util::SNPrintf(g_SendBuffer, g_BufferSize, "%lld", battery);

    NN_SDK_LOG("[recovery] Send: %s\n", g_SendBuffer);
    NN_RESULT_DO(dsEndpoints[0].PostBuffer(&bytesTransferred,
        g_SendBuffer,
        sendSize));

    NN_RESULT_SUCCESS;
}


::nn::Result WriteToStorage(::nn::fs::BisPartitionId id, uint64_t offset, void* buf, uint64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buf);

    ::std::unique_ptr<::nn::fs::IStorage> storage;
    NN_RESULT_DO(::nn::fs::OpenBisPartition(&storage, id));

    NN_RESULT_DO(storage.get()->Write(offset, reinterpret_cast<::nn::Bit8 *>(buf), size));

    NN_RESULT_SUCCESS;
}

::nn::Result ReceiveImage(::nn::Bit8* pInBuffer, uint64_t totalImageSize)
{
    uint32_t bytesRead = 0;

    NN_RESULT_DO(dsEndpoints[1].PostBuffer(&bytesRead,
                                            pInBuffer,
                                            totalImageSize));

    NN_RESULT_SUCCESS;
}

bool VerifyRecievedImage(uint64_t totalImageSize,
                                 uint64_t bctOffset,
                                 uint64_t writeBctSize,
                                 uint64_t blOffset,
                                 uint64_t blSize,
                                 uint64_t package2Offset,
                                 uint64_t package2Size,
                                 uint64_t systemOffset,
                                 uint64_t systemSize)
{
    if(totalImageSize < WriteParameterOffset + sizeof(WriteParameter))
    {
        NN_SDK_LOG("[recovery] Invalid Image pattern = 1\n");
        return false;
    }

    if(MbrGptSize != 0 && totalImageSize < MbrGptOffset + MbrGptSize)
    {
        NN_SDK_LOG("[recovery] Invalid Image pattern = 2\n");
        return false;
    }

    if(writeBctSize != 0 && totalImageSize < bctOffset +  writeBctSize)
    {
        NN_SDK_LOG("[recovery] Invalid Image pattern = 3\n");
        return false;
    }

    if(blSize != 0 && totalImageSize < blOffset + blSize)
    {
        NN_SDK_LOG("[recovery] Invalid Image pattern = 4\n");
        return false;
    }

    if(package2Size != 0 && totalImageSize < package2Offset + package2Size)
    {
        NN_SDK_LOG("[recovery] Invalid Image pattern = 5\n");
        return false;
    }

    if(systemSize != 0 && totalImageSize < systemOffset + systemSize)
    {
        NN_SDK_LOG("[recovery] Invalid Image pattern = 6\n");
        return false;
    }

    return true;
}

::nn::Result ReceiveAndWriteImage() NN_NOEXCEPT
{
    uint32_t bytesRead = 0;
    uint64_t totalImageSize  = 0;
    DsHeader header;

    //初めにイメージ全体のファイルサイズを受け取る
    NN_RESULT_DO(dsEndpoints[1].PostBuffer(&bytesRead,
                                           &g_ReceiveBuffer,
                                           sizeof(DsHeader)));

    header = *reinterpret_cast<DsHeader *>(g_ReceiveBuffer);
    totalImageSize = header.bytes;
    NN_SDK_LOG("[recovery] TotalImageSize = %d\n", totalImageSize);

    //次から具体的なイメージ受け取り, 書き出し
    ::std::unique_ptr<::nn::Bit8[]> imageStorage(new ::nn::Bit8[totalImageSize]);
    NN_RESULT_DO(ReceiveImage(imageStorage.get(), totalImageSize));

    const WriteParameter* writeParameter =
        reinterpret_cast<WriteParameter *>(imageStorage.get() + WriteParameterOffset);

    for (int i = 0; i < 8; i++)
    {
        NN_SDK_LOG("[recovery] writeParameter %d = %x\n", i, writeParameter->value[i]);
    }

    const uint64_t BctOffset = writeParameter->value[0];
    const uint64_t WriteBctSize = writeParameter->value[1];
    const uint64_t BlOffset = writeParameter->value[2];
    const uint64_t BlSize = writeParameter->value[3];
    const uint64_t Package2Offset = writeParameter->value[4];
    const uint64_t Package2Size = writeParameter->value[5];
    const uint64_t SystemOffset = writeParameter->value[6];
    const uint64_t SystemSize = writeParameter->value[7];

    NN_SDK_LOG("[recovery] checking Received Image...\n");
    if(!VerifyRecievedImage(totalImageSize,
                            BctOffset,
                            WriteBctSize,
                            BlOffset,
                            BlSize,
                            Package2Offset,
                            Package2Size,
                            SystemOffset,
                            SystemSize))
    {
        NN_RESULT_SUCCESS;
    }

    NN_SDK_LOG("[recovery] System Recovering ...\n");

    // anti rollback fuseで止まってしまうほど古いイメージは書き込まない
    char safeBootloaderBuildDateTime[BlVersionDataBuildDateTimeStringSize];
    NN_RESULT_DO(
        ReadSafeBootLoaderBuildDateTime(safeBootloaderBuildDateTime));
    NN_SDK_LOG("[recovery] SafeMode Bootloader Build Date Time = %s\n", safeBootloaderBuildDateTime);
    char bootloaderBuildDateTimeInImage[BlVersionDataBuildDateTimeStringSize];
    NN_RESULT_DO(
        ReadBootLoaderBuildDateTimeInReceivedImage(bootloaderBuildDateTimeInImage, imageStorage.get() + BlOffset));
    NN_SDK_LOG("[recovery] Bootloader Build Date Time in image = %s\n", bootloaderBuildDateTimeInImage);
    NN_ABORT_UNLESS(std::strncmp(UsableBlVersionDateTimeString, bootloaderBuildDateTimeInImage, BlVersionDataBuildDateTimeStringSize - 1) <= 0);

    //Write MBR + GPT
    NN_SDK_LOG("[recovery] Write MBR + GPT\n");
    NN_RESULT_DO(
        WriteToStorage(::nn::fs::BisPartitionId::UserDataRoot,
                       StorageOffsetBase,
                       imageStorage.get() + MbrGptOffset,
                       MbrGptSize));

    // Normal main BCT(index 0)を無効化
    NN_SDK_LOG("[recovery] Invalidate BCT (index 0)\n");
    NN_RESULT_DO(
        InvalidateBct(0));


    //Write BootLoader
    NN_SDK_LOG("[recovery] Write BootLoader\n");
    NN_RESULT_DO(
        WriteToStorage(::nn::fs::BisPartitionId::BootPartition1Root,
            StorageOffsetBase + BootLoaderOffset,
            imageStorage.get() + BlOffset,
            BlSize));

    //Write Package2
    NN_SDK_LOG("[recovery] Write Package2\n");
    NN_RESULT_DO(
        WriteToStorage(::nn::fs::BisPartitionId::BootConfigAndPackage2Part1,
                       Package2OffsetBase,
                       imageStorage.get() + Package2Offset,
                       Package2Size));

    //Write System Partition
    NN_SDK_LOG("[recovery] Write System Partition\n");
    NN_RESULT_DO(
        WriteToStorage(::nn::fs::BisPartitionId::UserDataRoot,
                       StorageOffsetBase + SystemPartitionOffset,
                       imageStorage.get() + SystemOffset,
                       SystemSize));

    // Normal Sub BCT(index 2)をRecovery BL用にする
    //Eks をメモリ上で Bct にコピーして書き込む
    NN_SDK_LOG("[recovery] Write Recovery BCT to index 2\n");
    NN_RESULT_DO(
        UpdateSecureInfo(2, imageStorage.get() + BctOffset));

    // Safe Sub BCT(index 3)をRecovery BL用にする
    NN_SDK_LOG("[recovery] Write Recovery BCT to index 3\n");
    NN_RESULT_DO(
        UpdateSecureInfo(3, imageStorage.get() + BctOffset));

    // BLバージョンが増える更新の場合 Safe BCT(index 1)を無効化
    if(std::strncmp(safeBootloaderBuildDateTime, bootloaderBuildDateTimeInImage, BlVersionDataBuildDateTimeStringSize - 1) < 0)
    {
        NN_SDK_LOG("[recovery] Invalidate BCT (index 1)\n");
        NN_RESULT_DO(
            InvalidateBct(1));
    }

    // Normal Main BCT(index 0)をRecovery BL用にする
    NN_SDK_LOG("[recovery] Write Recovery BCT to index 0\n");
    NN_RESULT_DO(
        UpdateSecureInfo(0, imageStorage.get() + BctOffset));


    NN_SDK_LOG("[recovery] System Recovering Success\n");

    //再起動
    NN_SDK_LOG("[recovery] Rebooting...\n");
    ::nn::os::SleepThread(::nn::TimeSpan::FromSeconds(5));

    ::nn::bpc::InitializeBoardPowerControl();
    ::nn::bpc::RebootSystem();

    NN_RESULT_SUCCESS;
}

::nn::Result ReadCommand() NN_NOEXCEPT
{
    uint32_t bytesRead = 0;

    NN_RESULT_DO(
        dsEndpoints[1].PostBuffer(&bytesRead, g_ReceiveBuffer, g_BufferSize));

    memset(g_ReceiveBuffer, 0 ,sizeof(g_ReceiveBuffer));

    NN_RESULT_DO(
        dsEndpoints[1].PostBuffer(&bytesRead, g_ReceiveBuffer, g_BufferSize));

    // Prevent possible string buffer read overflow in strcmp calls below.
    if(bytesRead < g_BufferSize)
    {
        NN_SDK_LOG("[recovery] %d Bytes read - Received : %s\n", bytesRead, g_ReceiveBuffer);
    }
    else
    {
        NN_SDK_LOG("[recovery] 4096 Bytes read - Unknown Command\n");
        NN_RESULT_SUCCESS;
    }

    if (::std::strcmp(g_ReceiveBuffer, "0") == 0)
    {
        NN_SDK_LOG("[recovery] Send ConfigurationId1\n");
        NN_RESULT_DO(SendConfigurationId1());
    }
    else if (::std::strcmp(g_ReceiveBuffer, "1") == 0)
    {
        NN_SDK_LOG("[recovery] Send SerialNumber\n");
        NN_RESULT_DO(SendSerialNumber());
    }
    else if (::std::strcmp(g_ReceiveBuffer, "2") == 0)
    {
        NN_SDK_LOG("[recovery] Receive Recovery Image\n");
        NN_RESULT_DO(ReceiveAndWriteImage());
    }
    else if (::std::strcmp(g_ReceiveBuffer, "3") == 0)
    {
        NN_SDK_LOG("[recovery] Send FirmwareVersion\n");
        NN_RESULT_DO(SendFirmwareVersion());
    }
    else if (::std::strcmp(g_ReceiveBuffer, "4") == 0)
    {
        NN_SDK_LOG("[recovery] Send Battery\n");
        NN_RESULT_DO(SendBatteryChargePercentage());
    }
    else
    {
        NN_SDK_LOG("[recovery] Unknown Command\n");
    }

    NN_RESULT_SUCCESS;
}

extern "C" void nnMain()
{
    ::nn::fs::SetAllocator(Allocate, Deallocate);
    NN_ABORT_UNLESS_RESULT_SUCCESS(UsbInitialize());

    ::nn::psm::Initialize();
    while (NN_STATIC_CONDITION(true))
    {
        ::nn::usb::UsbState usbState;
        dsClient.GetState(&usbState);
        if (usbState == ::nn::usb::UsbState::UsbState_Configured)
        {
            ::nn::Result result = ReadCommand();
            if (result.IsFailure())
            {
                NN_SDK_LOG("[recovery] An error ocurred. InnerValue = %x\n", result.GetInnerValueForDebug());
            }
        }

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(5));
    }

    ::nn::psm::Finalize();
}

namespace {

::nn::lmem::HeapHandle& GetHeapHandle() NN_NOEXCEPT
{
    static ::std::atomic<bool> s_IsInitialised(false);
    static LockableMutexType s_Mutex =
    {
        NN_OS_MUTEX_INITIALIZER(false)
    };
    static ::nn::lmem::HeapHandle s_HeapHandle;
    static char s_HeapMemory[512 * 1024 * 1024];
    if (!s_IsInitialised)
    {
        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);
        if (!s_IsInitialised)
        {
            s_HeapHandle = ::nn::lmem::CreateExpHeap(
                s_HeapMemory,
                sizeof(s_HeapMemory),
                ::nn::lmem::CreationOption_NoOption);
            s_IsInitialised = true;
        }
    }
    return s_HeapHandle;
}

void* Allocate(size_t size) NN_NOEXCEPT
{
    return ::nn::lmem::AllocateFromExpHeap(GetHeapHandle(), size, ::nn::usb::HwLimitDmaBufferAlignmentSize);
}

void Deallocate(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    ::nn::lmem::FreeToExpHeap(GetHeapHandle(), p);
}

} // namespace
