﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // memset
#include <new>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>

#include "profiler_Workarea.h"
#include "profiler_Defines.h"
#include "profiler_Memory.h"
#include "profiler_Logging.h"
#include "profiler_SamplingThread.h"


namespace nn { namespace profiler {

namespace /*anonymous*/
{
    const size_t WorkAreaCount = SampleBufferIndex_MAX;

    struct Globals
    {
        WorkArea workAreas[WorkAreaCount];
        nn::os::MutexType mutexes[WorkAreaCount];
    };
    Globals* globals;
} // anonymous



void InitializeWorkAreas()
{
    globals = Memory::GetInstance()->Allocate<Globals>();
    new (globals) Globals;
    for (size_t i = 0; i < WorkAreaCount; ++i)
    {
        nn::os::InitializeMutex(&globals->mutexes[i], false, 0);
    }
}



void FinalizeWorkAreas()
{
    for (size_t i = 0; i < WorkAreaCount; ++i)
    {
        nn::os::FinalizeMutex(&globals->mutexes[i]);
    }
    Memory::GetInstance()->Free(globals);
    globals = nullptr;
}



WorkArea* GetWorkAreaForCore(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    return &globals->workAreas[core];
}



void InitializeCoreWorkArea(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea* ws = &globals->workAreas[core];
    memset(ws, 0, sizeof(WorkArea));
    ws->core_number = static_cast<uint8_t>(core);
}



void FinalizeCoreWorkArea(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea* ws = &globals->workAreas[core];
    SampleBuffers::GetInstance()->CloseCurrent(ws->curPtr, core);
    SampleBuffers::GetInstance()->FinalizeBuffer(core);
}



bool GetMoreSampleBuffer(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea* ws = &globals->workAreas[core];
    bool result = true;

    nn::os::LockMutex(&globals->mutexes[core]);
    if (ws->curPtr >= ws->sentinel)
    {
        INFO_LOG("Buffer full, attempting to start a new buffer\n");
        SampleBuffers::GetInstance()->CloseCurrent(ws->curPtr, core);
        result = ObtainAndSetupSampleBuffer(core);
    }
    nn::os::UnlockMutex(&globals->mutexes[core]);

    return result;
}



bool ObtainAndSetupSampleBuffer(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea *ws = &globals->workAreas[core];

    void* memory = SampleBuffers::GetInstance()->Allocate(core);
    if (memory != nullptr)
    {
        ws->startPtr = reinterpret_cast<uint8_t*>(memory);
        ws->curPtr = ws->startPtr;
        ws->sentinel = ws->startPtr + SampleMemoryBlockSize - SentinelSize;
        return true;
    }
    else
    {
        return false;
    }
}



bool CheckAndExpandBuffersIfNeeded(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea* ws = &globals->workAreas[core];

    if (!(ws->record_cores & (1 << core)))
    {
        DEBUG_LOG("Core %d no longer active, stopping profile\n", core);
        SampleBuffers::GetInstance()->CloseCurrent(ws->curPtr, core);
        StopProfilingSamplingThreads();
        return false;
    }

    if (ws->curPtr >= ws->sentinel)
    {
        bool moreMemory = GetMoreSampleBuffer(core);
        if (!moreMemory)
        {
            DEBUG_LOG("Core %d is full, stop recording\n", core);
            StopProfilingSamplingThreads();
            return false;
        }
        DEBUG_LOG("Got new buffer for core %d\n", core);
    }

    return true;
}


} // profiler
} // nn
