﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <new>
#include <cstdio>

#include <nn/nn_Common.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma GCC diagnostic ignored "-Wsign-conversion"
#pragma GCC diagnostic ignored "-Wmissing-field-initializers"
#include <nn/os.h>
#include <nn/util/util_StringUtil.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Synchronization.h>
#include <nn/osdbg/osdbg_ThreadApi.h>
#include <nn/pm/pm_DebugMonitorApi.h>
#include <nn/pm/pm_Result.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_Result.h>
NN_PRAGMA_POP_WARNINGS

#include <nn/profiler/profiler_Result.h>

#include "profiler_CodeRewriting.h"
#include "profiler_Comms.h"
#include "profiler_CommsIpc.h"
#include "profiler_CommMessages.h"
#include "profiler_Defines.h"
#include "profiler_ExceptionHandling.h"
#include "profiler_ForceSend.h"
#include "profiler_LibModule.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_Messages.h"
#include "profiler_SamplingThread.h"
#include "profiler_ResultPrivate.h"
#include "profiler_TargetApplication.h"
#include "profiler_ThreadPriorities.h"
#include "profiler_Timing.h"
#include "profiler_Workarea.h"


//#define NN_PROFILER_RECORD_FROM_ONE_CORE


namespace nn { namespace profiler {

TargetApplication* TargetApplication::s_CurrentApplication = nullptr;


namespace /*anonymous*/
{
    //const int64_t DefaultWaitTimeNanoSeconds = MicrosecondsToNanoseconds(500);
    const int64_t DefaultWaitTimeNanoSeconds = nn::svc::WAIT_INFINITE;

    const unsigned ContinueFlagBase = (
        nn::svc::ContinueFlag_ContinueAll |
        nn::svc::ContinueFlag_EnableExceptionEvent);

    const unsigned ThreadContextFlagDefault = (
        nn::svc::ContextFlag_Control |
        nn::svc::ContextFlag_General);

    const unsigned ThreadContextFlagAll = (
        nn::svc::ContextFlag_Control |
        nn::svc::ContextFlag_Fpu |
        nn::svc::ContextFlag_FpuControl |
        nn::svc::ContextFlag_General);

    enum EnableExceptionHandledFlag
    {
        EnableExceptionHandledFlag_False = 0,
        EnableExceptionHandledFlag_True  = 1,
    };

    volatile bool sKill = false;
    volatile int64_t sWaitTime = DefaultWaitTimeNanoSeconds; // in nanoseconds
    volatile bool sTimeoutBreaks = false;
    nn::os::EventType sLaunchEvent;
    nn::os::ThreadId sPrimaryThreadId;

    nn::svc::Handle sTargetProcessHandle = nn::svc::INVALID_HANDLE_VALUE;

    nn::os::ThreadType sTargetWatchThread;
    void *sTargetWatchStack;

    nn::svc::DebugInfoCreateProcess sProcessInfo;
    ThreadList* sThreadInfo = nullptr;

    volatile bool sShouldAttachOnLaunch = false;
    nn::ncm::ProgramId sTargetAttachProgramId = nn::ncm::ProgramId::GetInvalidId();

    enum TimerBlockId
    {
        TimerBlockId_Sleep = 0,
        TimerBlockId_NotSleep,
        TimerBlockId_TakeSample,
        TimerBlockId_WaitSamples,
        TimerBlockId_GetThreadContext,
        TimerBlockId_BreakRequest,
        TimerBlockId_ContinueRequest,
        TimerBlockId_GetDebugEvent,

        TimerBlockId_Count
    };

#if defined(NN_PROFILER_TIMING_ENABLE)
    nn::profiler::Timer sTimers[TimerBlockId_Count];
#endif


    inline nn::Result WaitSynchronization(int64_t ns)
    {
        nn::Result result;
        VERBOSE_LOG("Wait Sychronization wait time == %ld\n", sWaitTime);
        int waitHandleTrigger = 0;
        result = nn::svc::WaitSynchronization(&waitHandleTrigger, &sTargetProcessHandle, 1, ns);
        return result;
    }


    inline nn::Result GetDebugEvent(nn::svc::DebugEventInfo* info)
    {
        nn::Result result;
        result = nn::svc::GetDebugEvent(info, sTargetProcessHandle);
        return result;
    }


    inline nn::Result ContinueDebugEvent(EnableExceptionHandledFlag exceptionHandled)
    {
        nn::Result result;
        nn::Bit32 flags = ContinueFlagBase;
        if (exceptionHandled == EnableExceptionHandledFlag_True)
        {
            flags |= nn::svc::ContinueFlag_ExceptionHandled;
        }
        result = nn::svc::ContinueDebugEvent(sTargetProcessHandle, flags, nullptr, 0);
        return result;
    }


    inline nn::Result QueryDebugProcessMemory(
        nn::svc::MemoryInfo* pMemInfo,
        nn::svc::PageInfo* pPageInfo,
        uintptr_t address)
    {
        nn::Result result;
        result = nn::svc::QueryDebugProcessMemory(pMemInfo, pPageInfo, sTargetProcessHandle, address);
        return result;
    }


    inline nn::Result GetInfo(
        nn::Bit64* pOut,
        nn::svc::InfoType infoType,
        nn::Bit64 param)
    {
        nn::Result result;
        result = nn::svc::GetInfo(pOut, infoType, sTargetProcessHandle, param);
        return result;
    }


    void UpdateThreadInfo()
    {
        sThreadInfo->ReadLock();
        DEBUG_LOG("Thread List\n");
        DEBUG_LOG("---\n");
        for (int i = 0; i < sThreadInfo->Count(); ++i)
        {
            ThreadListItem *t = sThreadInfo->GetByIndex(i);
            UpdateSingleThreadInfo(t);

            DEBUG_LOG("ID: %lld", t->id);
            DEBUG_LOG(" ThreadType: %p\n", t->threadType);
            DEBUG_LOG(" StackBase: %p\n", t->stackBase);
            DEBUG_LOG(" StackSize: %ld\n", t->stackSize);
            DEBUG_LOG(" Name: %s\n", t->threadName);
            DEBUG_LOG(" Priority: %d\n", t->priority);
            DEBUG_LOG(" CoreMask: %d\n", t->coreMask);
            DEBUG_LOG(" IdealCore: %d\n", t->idealCore);
        }
        DEBUG_LOG("---\n");
        sThreadInfo->ReadUnlock();
    }

    typedef EnableExceptionHandledFlag (*DebugEventHandler)(nn::svc::DebugEventInfo* eventInfo);

    EnableExceptionHandledFlag DebugEventHandler_CreateProcess(nn::svc::DebugEventInfo* eventInfo);
    EnableExceptionHandledFlag DebugEventHandler_CreateThread(nn::svc::DebugEventInfo* eventInfo);
    EnableExceptionHandledFlag DebugEventHandler_ExitThread(nn::svc::DebugEventInfo* eventInfo);
    EnableExceptionHandledFlag DebugEventHandler_ExitProcess(nn::svc::DebugEventInfo* eventInfo);
    EnableExceptionHandledFlag DebugEventHandler_Exception(nn::svc::DebugEventInfo* eventInfo);
    EnableExceptionHandledFlag DebugEventHandler_Default(nn::svc::DebugEventInfo* eventInfo) NN_IS_UNUSED_MEMBER;

    DebugEventHandler gDebugEventHandlers[] =
    {
        DebugEventHandler_CreateProcess,
        DebugEventHandler_CreateThread,
        DebugEventHandler_ExitProcess,
        DebugEventHandler_ExitThread,
        DebugEventHandler_Exception,
    };
    NN_STATIC_ASSERT(nn::svc::DebugEvent_CreateProcess == 0);
    NN_STATIC_ASSERT(nn::svc::DebugEvent_CreateThread == 1);
    NN_STATIC_ASSERT(nn::svc::DebugEvent_ExitProcess == 2);
    NN_STATIC_ASSERT(nn::svc::DebugEvent_ExitThread == 3);
    NN_STATIC_ASSERT(nn::svc::DebugEvent_Exception == 4);


    EnableExceptionHandledFlag HandleDebugEvent(
        nn::svc::DebugEventInfo* eventInfo)
    {
        VERBOSE_LOG("Debug Event: %d\n", eventInfo->event);
        NN_SDK_ASSERT(eventInfo->event < (sizeof(gDebugEventHandlers) / sizeof(DebugEventHandler)));
        return gDebugEventHandlers[eventInfo->event](eventInfo);
    }



    EnableExceptionHandledFlag DebugEventHandler_Default(
        nn::svc::DebugEventInfo* eventInfo)
    {
        NN_UNUSED(eventInfo);
        return EnableExceptionHandledFlag_True;
    }



    EnableExceptionHandledFlag DebugEventHandler_CreateProcess(
        nn::svc::DebugEventInfo* eventInfo)
    {
        memcpy(&sProcessInfo, &eventInfo->info.createProcess, sizeof(sProcessInfo));

        char progName[13] = { 0 };
        strncpy(progName, eventInfo->info.createProcess.programName, 12);

        INFO_LOG("TargetWatch : DebugEvent_CreateProcess\n");
        INFO_LOG(" Process ID = %lu\n", eventInfo->info.createProcess.processId);
        INFO_LOG(" Program ID = %016lx\n", eventInfo->info.createProcess.programId);
        INFO_LOG(" Program Name = %s\n", progName);
        INFO_LOG(" Flags: 0x%08x\n", eventInfo->info.createProcess.flag);
        return EnableExceptionHandledFlag_True;
    }



    EnableExceptionHandledFlag DebugEventHandler_CreateThread(
        nn::svc::DebugEventInfo* eventInfo)
    {
        nn::Result result;

        INFO_LOG("-- Thread Created --\n");
        INFO_LOG("TargetWatch : DebugEvent_CreateThread\n");
        INFO_LOG(" New Thread ID = %d\n", eventInfo->info.createThread.id);
        INFO_LOG(
            " Base Address = %p\n",
            eventInfo->info.createThread.localRegionBaseAddress);
        INFO_LOG(
            " Entry Address = %p\n",
            eventInfo->info.createThread.entryAddress);

        nn::os::ThreadId threadId = eventInfo->info.createThread.id;

        ThreadListItem* item = sThreadInfo->Insert(threadId);
        NN_STATIC_ASSERT(sizeof(*item->GetDebugInfo()) == sizeof(eventInfo->info.createThread));
        memcpy(item->GetDebugInfo(), &eventInfo->info.createThread, sizeof(*item->GetDebugInfo()));

        nn::osdbg::ThreadInfo threadInfo;
        memset(&threadInfo, 0, sizeof(threadInfo));
        result = nn::osdbg::InitializeThreadInfo(
            &threadInfo,
            sTargetProcessHandle,
            &sProcessInfo,
            item->GetDebugInfo());

        if (result.IsSuccess())
        {
            item->stackBase = threadInfo._stack + threadInfo._stackSize;
            item->stackSize = threadInfo._stackSize;
            item->threadType = threadInfo._threadType;
        }
        else
        {
            WARNING_LOG("Failed to get thread info during thread creation for thread %lld\n", threadId);
            DumpResultInformation(LOG_AS_WARNING, result);
        }

        return EnableExceptionHandledFlag_True;
    }



    EnableExceptionHandledFlag DebugEventHandler_ExitThread(
        nn::svc::DebugEventInfo* eventInfo)
    {
        nn::os::ThreadId threadId = eventInfo->threadId;
        INFO_LOG("-- Thread Exited: %lu --\n", threadId);
        INFO_LOG(
            "Thread Removal Reason: %d\n",
            eventInfo->info.exitThread.exitReason);

        sThreadInfo->Remove(threadId);
        return EnableExceptionHandledFlag_True;
    }



    EnableExceptionHandledFlag DebugEventHandler_ExitProcess(
        nn::svc::DebugEventInfo* eventInfo)
    {
        NN_UNUSED(eventInfo);

        INFO_LOG("-- EXIT PROCESS --\n");
        INFO_LOG(" Reason: %d\n", eventInfo->info.exitProcess.exitReason);

        TargetApplication::GetCurrent()->Close();
        return EnableExceptionHandledFlag_True;
    }



    int requestSampleCoreMask = 0;
    EnableExceptionHandledFlag DebugEventHandler_Exception(
        nn::svc::DebugEventInfo* eventInfo)
    {
        nn::Result result;

        if (eventInfo->info.exception.exceptionCode == nn::svc::DebugException_DebuggerBreak)
        {
            requestSampleCoreMask = 0;
            NN_PROFILER_TIMING_BEGIN(&sTimers[TimerBlockId_TakeSample]);
            for (unsigned core = 0; core < SupportedCoreCount; ++core)
            {
                WorkArea *ws = GetWorkAreaForCore(SampleBufferIndex(core));
                if (ws->record_cores & (1 << core))
                {
                    nn::os::ThreadId threadID =
                        eventInfo->info.exception.detail.debuggerBreak.currentThreadId[core];

                    VERBOSE_LOG("Attempting to take a sample from thread %lu on core %d\n", threadID, core);

                    if (threadID != DebugEventOtherProcessThreadId &&
                        threadID != DebugEventSystemSleepThreadId)
                    {
                        nn::svc::ThreadContext context;
                        NN_PROFILER_TIMING_BEGIN(&sTimers[TimerBlockId_GetThreadContext]);
                        result = TargetApplication::GetCurrent()->GetThreadContext(&context, threadID, ThreadContextFlagDefault);
                        NN_PROFILER_TIMING_END(&sTimers[TimerBlockId_GetThreadContext]);
                        if (result.IsFailure())
                        {
                            DumpResultInformation(LOG_AS_ERROR, result);
                            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                        }
                        ws->context.fp = context.fp;
                        ws->context.lr = context.lr;
                        ws->context.sp = context.sp;
                        ws->context.pc = context.pc;
                    }

                    requestSampleCoreMask |= (1 << core);

#if defined(NN_PROFILER_RECORD_FROM_ONE_CORE)
                    // TODO: Still needs implementation for reading from performance counters
                    SamplingThread_RecordImmediate(ws, core, threadID);
#else
                    SamplingTriggerSample(core, threadID);
#endif

                    // TODO: Sample by perf counter
                    //if (ws->usefulSettings.sample_using_perf_counters)
                    //{
                    //    ws->usefulSettings.record_cores &= static_cast<uint8_t>(~(1 << core));
                    //}
                }
            }

            DUMP_CURRENT_LINE();
            NN_PROFILER_TIMING_BEGIN(&sTimers[TimerBlockId_WaitSamples]);
            if (requestSampleCoreMask != 0)
            {
                // The very first DebuggerBreak is done to reset the timeout.
                // There will be nothing to sample, so don't try to wait.
                WaitSamples();
            }
            NN_PROFILER_TIMING_END(&sTimers[TimerBlockId_WaitSamples]);
            DUMP_CURRENT_LINE();
            NN_PROFILER_TIMING_END(&sTimers[TimerBlockId_TakeSample]);
        }
        else if (eventInfo->info.exception.exceptionCode == nn::svc::DebugException_AttachBreak ||
                 eventInfo->info.exception.exceptionCode == nn::svc::DebugException_BreakPoint)
        {
            INFO_LOG("Ignoring DebugInfoException %d\n", eventInfo->info.exception.exceptionCode);
        }
        else
        {
            auto threadInfo = sThreadInfo->GetByKey(eventInfo->threadId);
            if (threadInfo != nullptr)
            {
                nn::svc::ThreadContext context;
                result = TargetApplication::GetCurrent()->GetThreadContext(
                    &context,
                    eventInfo->threadId,
                    ThreadContextFlagAll);

                if (result.IsFailure())
                {
                    ERROR_LOG("GetThreadContext FAILED in main loop for thread %lu.\n", eventInfo->threadId);
                    DumpResultInformation(LOG_AS_ERROR, result);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    return EnableExceptionHandledFlag_False;
                }

                result = HandleException(eventInfo, &context);

                if (nn::profiler::ResultContinueFromException::Includes(result))
                {
                    INFO_LOG("ResultContinueFromException! Attempting to continue.\n");
                    return EnableExceptionHandledFlag_True;
                }
                else
                {
                    NN_SDK_ASSERT(nn::profiler::ResultKillApplication::Includes(result));

                    auto app = TargetApplication::GetCurrent();
                    app->Close();
                }

                return EnableExceptionHandledFlag_False;
            }
        }

        return EnableExceptionHandledFlag_True;
    }



    void TargetWatch(void *)
    {
        DumpThreadInformation();

        sPrimaryThreadId = nn::os::GetThreadId(nn::os::GetCurrentThread());

        while (NN_STATIC_CONDITION(true))
        {
            INFO_LOG("Waiting for attach.\n");

            nn::os::WaitEvent(&sLaunchEvent);

            if (sShouldAttachOnLaunch)
            {
                sShouldAttachOnLaunch = false;
                nn::Result result = TargetApplication::AttachOnLaunch(sTargetAttachProgramId);
                if (result.IsFailure())
                {
                    ERROR_LOG("Could not attach on launch!\n");
                    DumpResultInformation(LOG_AS_ERROR, result);

                    nn::os::ClearEvent(&sLaunchEvent);

                    if (nn::profiler::ResultDebuggerAttached::Includes(result))
                    {
                        SendNotificationToPC(NotificationWarningDebuggerAttached, result);
                    }
                    else /*if (nn::profiler::ResultCouldNotAttach::Includes(result))*/
                    {
                        SendNotificationToPC(NotificationErrorCouldNotAttach, result);
                    }

                    continue;
                }

                // Just clear the event as the AttachOnLaunch will result in it being set.
                nn::os::ClearEvent(&sLaunchEvent);
                SendReadyToProfile(SampleBuffers::GetInstance()->GetSize());
            }

            while (sTargetProcessHandle.IsValid())
            {
                nn::Result result;

                NN_PROFILER_TIMING_END(&sTimers[TimerBlockId_NotSleep]);
                NN_PROFILER_TIMING_BEGIN(&sTimers[TimerBlockId_Sleep]);
                result = WaitSynchronization(sWaitTime);
                NN_PROFILER_TIMING_END(&sTimers[TimerBlockId_Sleep]);
                NN_PROFILER_TIMING_BEGIN(&sTimers[TimerBlockId_NotSleep]);

                if (sKill == true)
                {
                    TargetApplication::GetCurrent()->Close();
                    sKill = false;
                }

                if (nn::svc::ResultTimeout::Includes(result))
                {
                    if (sTimeoutBreaks)
                    {
                        NN_PROFILER_TIMING_BEGIN(&sTimers[TimerBlockId_BreakRequest]);
                        TargetApplication::GetCurrent()->RequestBreak();
                        NN_PROFILER_TIMING_END(&sTimers[TimerBlockId_BreakRequest]);
                    }
                    else { continue; }
                }
                else if (result.IsFailure())
                {
                    ERROR_LOG("Error waiting for synchronization!\n");
                    DumpResultInformation(LOG_AS_ERROR, result);
                    continue;
                }

                EnableExceptionHandledFlag exceptionHandled = EnableExceptionHandledFlag_True;

                while (sTargetProcessHandle.IsValid() && exceptionHandled == EnableExceptionHandledFlag_True)
                {
                    nn::svc::DebugEventInfo debug_event_info;
                    NN_PROFILER_TIMING_BEGIN(&sTimers[TimerBlockId_GetDebugEvent]);
                    result = GetDebugEvent(&debug_event_info);
                    NN_PROFILER_TIMING_END(&sTimers[TimerBlockId_GetDebugEvent]);
                    if (nn::svc::ResultNoEvent::Includes(result))
                    {
                        break;
                    }
                    else if (result.IsFailure())
                    {
                        ERROR_LOG("Error getting Debug Event Info\n");
                        DumpResultInformation(LOG_AS_ERROR, result);
                        continue;
                    }

                    exceptionHandled = HandleDebugEvent(&debug_event_info);
                }

                //if (debug_event_info.flags & nn::svc::DebugEventFlag_Stopped)
                {
                    NN_PROFILER_TIMING_BEGIN(&sTimers[TimerBlockId_ContinueRequest]);
                    result = ContinueDebugEvent(exceptionHandled);
                    if (result.IsFailure() && !nn::svc::ResultBusy::Includes(result))
                    {
                        ERROR_LOG("ContinueDebugEvent FAILED.\n");
                        DumpResultInformation(LOG_AS_ERROR, result);
                    }
                    NN_PROFILER_TIMING_END(&sTimers[TimerBlockId_ContinueRequest]);
                }
            }

            DEBUG_LOG("Exiting loop due to invalid process handle!\n");

            // Do some cleanup for anything that may contain information about the attached process.
            nn::os::SignalEvent(GetStopEvent());
            TargetApplication::GetCurrent()->Close();
        }
    } //NOLINT(impl/function_size)
} // anonymous



void UpdateSingleThreadInfo(ThreadListItem* threadList)
{
    nn::Bit64 param1;
    nn::Bit32 param2;
    nn::Bit64 tid = threadList->id;
    char threadName[nn::os::ThreadNameLengthMax];
    nn::Result result = nn::ResultSuccess();

    // Core Affinity
    nn::svc::GetDebugThreadParam(
        &param1,
        &param2,
        sTargetProcessHandle,
        tid,
        nn::svc::DebugThreadParam_AffinityMask);
    threadList->coreMask = static_cast<int>(param1);

    // Ideal Core
    nn::svc::GetDebugThreadParam(
        &param1,
        &param2,
        sTargetProcessHandle,
        tid,
        nn::svc::DebugThreadParam_IdealProcessor);
    threadList->idealCore = param2;

    nn::osdbg::ThreadInfo threadInfo;
    memset(&threadInfo, 0, sizeof(threadInfo));
    result = nn::osdbg::InitializeThreadInfo(
        &threadInfo,
        sTargetProcessHandle,
        &sProcessInfo,
        &threadList->threadInfo);
    if (result.IsSuccess())
    {
        // Priority
        threadList->priority = threadInfo._currentPriority;
        threadList->stackBase = threadInfo._stack + threadInfo._stackSize;
        threadList->stackSize = threadInfo._stackSize;
        threadList->threadType = threadInfo._threadType;

        // Thread Name
        result = TargetApplication::GetCurrent()->ReadMemory(
            reinterpret_cast<uintptr_t>(threadName),
            threadInfo._namePointer,
            nn::os::ThreadNameLengthMax);
        if (result.IsSuccess())
        {
            strncpy(threadList->threadName, threadName, nn::os::ThreadNameLengthMax);
            threadList->threadName[nn::os::ThreadNameLengthMax - 1] = '\0';
        }
        else
        {
            WARNING_LOG("Failed to read thread name from debug process for thread %lld.\n", tid);
            DumpResultInformation(LOG_AS_WARNING, result);
            // No need to overwrite value. threadName was memset to 0 during construction.
            // If this wasn't a new object, we would prefer to continue using previously obtained value.
        }
    }
    else
    {
        WARNING_LOG("Failed to get thread info from osdbg for thread %lld.\n", tid);
        DumpResultInformation(LOG_AS_WARNING, result);
    }
}



void DumpTargetApplicationInfo()
{
    FORCE_LOG(" Target Application\n");
    FORCE_LOG("  Request Sample Core Mask: 0x%x\n", requestSampleCoreMask);
}



/*****************************************************************************
    Static Class Definitions
 *****************************************************************************/
void TargetApplication::Initialize()
{
    if (s_CurrentApplication != nullptr) { return; }

    s_CurrentApplication = Memory::GetInstance()->Allocate<TargetApplication>();
    memset(s_CurrentApplication, 0, sizeof(TargetApplication));
    new (s_CurrentApplication) TargetApplication();

#if defined(NN_PROFILER_SUPPORT_SYSTEM_PROCESS)
    // For system process developers, we can use pm since they can use unsigned desc
    // Note, pm's API may be changed later.
    nn::Result result = nn::pm::InitializeForDebugMonitor();
#else
    nn::Result result = nn::ns::InitializeForDevelop();
#endif
    if (result.IsFailure())
    {
        ERROR_LOG("Failed to initialize PM library\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    nn::os::InitializeEvent(&sLaunchEvent, false, nn::os::EventClearMode_AutoClear);

    size_t targetWatchStackSize = 4096 * 2;
    int priority = ThreadPriority_SvcWatch;
    sTargetWatchStack = Memory::GetInstance()->Allocate(targetWatchStackSize, nn::os::GuardedStackAlignment);
    nn::os::CreateThread(
        &sTargetWatchThread,
        TargetWatch,
        nullptr,
        sTargetWatchStack,
        targetWatchStackSize,
        priority,
        ProfilerPrimaryCore);
    nn::os::SetThreadName(&sTargetWatchThread, "[profiler] Target Watch");
    nn::os::StartThread(&sTargetWatchThread);
}



void TargetApplication::Finalize()
{
    nn::os::SignalEvent(&sLaunchEvent);

    s_CurrentApplication->RequestBreak();

    nn::os::WaitThread(&sTargetWatchThread);
    nn::os::DestroyThread(&sTargetWatchThread);
    Memory::GetInstance()->Free(sTargetWatchStack);
    sTargetWatchStack = nullptr;

    if (s_CurrentApplication != nullptr)
    {
        s_CurrentApplication->~TargetApplication();
        Memory::GetInstance()->Free(s_CurrentApplication);
        s_CurrentApplication = nullptr;
    }

    nn::os::FinalizeEvent(&sLaunchEvent);
}


nn::Result TargetApplication::Launch()
{
    return ScheduleAttachOnLaunch(nn::ncm::ProgramId::GetInvalidId());
}



nn::Result TargetApplication::ScheduleAttachOnLaunch(nn::ncm::ProgramId programId)
{
    sShouldAttachOnLaunch = true;
    sTargetAttachProgramId = programId;
    nn::os::SignalEvent(&sLaunchEvent);
    return nn::ResultSuccess();
}



nn::Result TargetApplication::AttachOnLaunch(nn::ncm::ProgramId programId)
{
    nn::Result result;
    nn::os::ProcessId processId;
    NN_UNUSED(programId);

    if (sTargetProcessHandle.IsValid())
    {
        WARNING_LOG("Target process handle is already valid\n");
        return nn::profiler::ResultAlreadyAttached();
    }

    s_CurrentApplication->m_isAttached = false;

#if defined(NN_PROFILER_SUPPORT_SYSTEM_PROCESS)
    nn::os::NativeHandle handle;
    if (programId == nn::ncm::ProgramId::GetInvalidId())
    {
        result = nn::pm::HookToCreateApplicationProcess(&handle);
    }
    else
    {
        result = nn::pm::HookToCreateProcess(&handle, programId);
    }
    if (result.IsFailure())
    {
        ERROR_LOG("Could not create hook for program id: 0x%016llx.\n", programId.value);
        DumpResultInformation(LOG_AS_ERROR, result);
        return result;
    }

    nn::os::SystemEventType event;
    nn::os::AttachReadableHandleToSystemEvent(&event, handle, true, nn::os::EventClearMode_AutoClear);

    while (NN_STATIC_CONDITION(true))
    {
        nn::TimeSpan waitTime = nn::TimeSpan::FromMilliSeconds(1);
        bool timeout = !nn::os::TimedWaitSystemEvent(&event, waitTime);

        if (timeout)
        {
            if (!IsPCConnected())
            {
                // TODO: Figure out how to stop ProgramManager from causing issues if we want to allow
                //       cancelling of AttachOnLaunch.
                //ERROR_LOG("PC has disconnected!\n");
                //return nn::profiler::ResultCouldNotAttach();
            }
            continue;
        }

        if (programId == nn::ncm::ProgramId::GetInvalidId())
        {
            result = nn::pm::GetApplicationProcessId(&processId);
            break;
        }
        else
        {
            result = nn::pm::GetProcessId(&processId, programId);
            if (result.IsSuccess()) { break; }

            // An unexpected error occurred. Break out of the connection loop.
            if (!nn::pm::ResultProcessNotFound::Includes(result)) { break; }
        }
    }
#else
    while (NN_STATIC_CONDITION(true))
    {
        if (!IsPCConnected())
        {
            // TODO: Figure out how to stop ProgramManager from causing issues if we want to allow
            //       cancelling of AttachOnLaunch.
            //ERROR_LOG("PC has disconnected!\n");
            //return nn::profiler::ResultCouldNotAttach();
        }

        result = nn::ns::GetRunningApplicationProcessIdForDevelop(&processId);
        if (result.IsSuccess())
        {
            break;
        }
        else if (result <= nn::ns::ResultApplicationNotRunning())
        {
        }
        else
        {
            return nn::profiler::ResultCouldNotAttach();
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }
#endif

    if (result.IsFailure())
    {
        ERROR_LOG("Could not get process id for program id: 0x%016llx.\n", programId.value);
        DumpResultInformation(LOG_AS_ERROR, result);
        return nn::profiler::ResultCouldNotAttach();
    }

    result = AttachImpl(processId.value);
    if (result.IsFailure())
    {
        ERROR_LOG("Error attaching to process %lld for program id: 0x%016llx.\n", processId.value, programId.value);
        DumpResultInformation(LOG_AS_ERROR, result);
        return result;
    }

#if defined(NN_PROFILER_SUPPORT_SYSTEM_PROCESS)
    result = nn::pm::StartProcess(processId);
    if (result.IsFailure())
    {
        ERROR_LOG("Could not StartProcess %lld\n.", processId.value);
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
#endif

    return result;
}



nn::Result TargetApplication::Attach(nn::ncm::ProgramId programId)
{
    nn::Result result;
    nn::os::ProcessId processId;

    if (sTargetProcessHandle.IsValid())
    {
        WARNING_LOG("Target process handle is already valid\n");
        return nn::profiler::ResultAlreadyAttached();
    }

    sTargetAttachProgramId = programId;

    s_CurrentApplication->m_isAttached = false;

#if defined(NN_PROFILER_SUPPORT_SYSTEM_PROCESS)
    result = nn::pm::GetProcessId(&processId, programId);
#else
    result = nn::ns::GetRunningApplicationProcessIdForDevelop(&processId);
#endif
    if (result.IsFailure())
    {
        ERROR_LOG("Could not get ProcessId for Program: 0x%016llx\n", programId.value);
        DumpResultInformation(LOG_AS_ERROR, result);
        return result;
    }

    sTargetProcessHandle = nn::svc::INVALID_HANDLE_VALUE;

    return AttachImpl(processId.value);

}



nn::Result TargetApplication::RegisterLibrary(uint64_t processId)
{
    nn::Bit64 pid = static_cast<uint64_t>(processId);
    if (pid == nn::os::ProcessId::GetInvalidId().value)
    {
        return nn::profiler::ResultInvalidArgument();
    }

    if (s_CurrentApplication->IsAttached() &&
        s_CurrentApplication->m_processId != pid)
    {
        return nn::profiler::ResultPidMismatch();
    }

    ResetInstrumentation();

    s_CurrentApplication->m_libraryPid = pid;
    if (s_CurrentApplication->GetProfilingMode() == ProfilingMode_Disabled)
    {
        s_CurrentApplication->SetProfilingMode(ProfilingMode_InProcess);
    }

    s_CurrentApplication->AvailableProfilingModesChanged();

    return nn::ResultSuccess();
}



nn::Result TargetApplication::UnregisterLibrary(uint64_t processId)
{
    nn::Bit64 pid = static_cast<uint64_t>(processId);
    if (pid == nn::os::ProcessId::GetInvalidId().value)
    {
        return nn::profiler::ResultInvalidArgument();
    }

    if (s_CurrentApplication->IsAttached() &&
        s_CurrentApplication->m_processId != pid)
    {
        return nn::profiler::ResultPidMismatch();
    }

    s_CurrentApplication->m_libraryPid = nn::os::ProcessId::GetInvalidId().value;

    ResetInstrumentation();

    nn::Result result = s_CurrentApplication->SetProfilingMode(ProfilingMode_OutOfProcess);
    if (nn::profiler::ResultInvalidProfilerStatus::Includes(result))
    {
        s_CurrentApplication->SetProfilingMode(ProfilingMode_Disabled);
    }

    s_CurrentApplication->AvailableProfilingModesChanged();

    return nn::ResultSuccess();
}



nn::Result TargetApplication::Attach(nn::Bit64 processId)
{
    sTargetAttachProgramId = nn::ncm::ProgramId::GetInvalidId();
    return AttachImpl(processId);
}



nn::Result TargetApplication::AttachImpl(nn::Bit64 processId)
{
    nn::Result result;
    if (sTargetProcessHandle.IsValid())
    {
        WARNING_LOG("Target process handle is already valid\n");
        return nn::profiler::ResultAlreadyAttached();
    }

    s_CurrentApplication->m_isAttached = false;

    result = nn::svc::DebugActiveProcess(&sTargetProcessHandle, processId);
    if (nn::svc::ResultBusy::Includes(result))
    {
        sTargetProcessHandle = nn::svc::INVALID_HANDLE_VALUE;
        return nn::profiler::ResultDebuggerAttached();
    }
    else if (result.IsFailure())
    {
        sTargetProcessHandle = nn::svc::INVALID_HANDLE_VALUE;
        DumpResultInformation(LOG_AS_ERROR, result);
        return nn::profiler::ResultCouldNotAttach();
    }

    EnableExceptionHandledFlag exceptionHandled = EnableExceptionHandledFlag_True;

    while (!s_CurrentApplication->m_isAttached && exceptionHandled == EnableExceptionHandledFlag_True)
    {
        DEBUG_LOG("Waiting for process to start\n");
        result = WaitSynchronization(10 * 1000 * 1000);
        if (result.IsFailure())
        {
            ERROR_LOG("Error in wait synchronization\n");
            DumpResultInformation(LOG_AS_ERROR, result);
            return nn::profiler::ResultCouldNotAttach();
            //return result;
        }

        while (exceptionHandled == EnableExceptionHandledFlag_True)
        {
            nn::svc::DebugEventInfo eventInfo;
            result = GetDebugEvent(&eventInfo);
            if (nn::svc::ResultNoEvent::Includes(result))
            {
                break;
            }
            else if (result.IsFailure())
            {
                ERROR_LOG("Error in getting debug event\n");
                DumpResultInformation(LOG_AS_ERROR, result);
                return nn::profiler::ResultCouldNotAttach();
                //return result;
            }

            exceptionHandled = HandleDebugEvent(&eventInfo);
            if (eventInfo.event == nn::svc::DebugEvent_CreateProcess)
            {
                if (sTargetAttachProgramId != nn::ncm::ProgramId::GetInvalidId() &&
                    sTargetAttachProgramId.value != sProcessInfo.programId)
                {
                    s_CurrentApplication->m_isAttached = false;
                    ERROR_LOG("Attached to the wrong program: Wanted (0x%016llx), Got (0x%016llx)\n",
                        sTargetAttachProgramId.value,
                        sProcessInfo.programId);
                    ContinueDebugEvent(nn::profiler::EnableExceptionHandledFlag_True);
                    nn::svc::CloseHandle(sTargetProcessHandle);
                    sTargetProcessHandle = nn::svc::INVALID_HANDLE_VALUE;
                    return nn::profiler::ResultCouldNotAttach();
                }
                else
                {
                    s_CurrentApplication->m_isAttached = true;
                }
            }
        }
    }

    nn::Bit64 coreMask;
    result = GetInfo(&coreMask, nn::svc::InfoType_CoreMask, 0);
    if (result.IsSuccess())
    {
        s_CurrentApplication->SetCoreMask(static_cast<uint32_t>(coreMask));
    }
    else
    {
        WARNING_LOG("Unable to obtain target application's core mask.\n");
        DumpResultInformation(LOG_AS_WARNING, result);
    }

    {
        result = ContinueDebugEvent(exceptionHandled);

        if (result.IsFailure())
        {
            ERROR_LOG("ContinueDebugEvent FAILED.\n");
            DumpResultInformation(LOG_AS_ERROR, result);
        }
    }

    DEBUG_LOG("Attached to application.\n");
    s_CurrentApplication->m_processId = processId;

    if (s_CurrentApplication->GetProfilingMode() == ProfilingMode_Disabled)
    {
        s_CurrentApplication->SetProfilingMode(ProfilingMode_OutOfProcess);
    }

    s_CurrentApplication->AvailableProfilingModesChanged();

    nn::os::SignalEvent(&sLaunchEvent);

    DumpMemoryInfo();

    return nn::ResultSuccess();
}



TargetApplication* TargetApplication::GetCurrent()
{
    return s_CurrentApplication;
}



bool TargetApplication::FindModuleName(
    uintptr_t baseaddr,
    uintptr_t endaddr,
    char* pOutName)
{
    NN_SDK_ASSERT(baseaddr <= endaddr);

    const int RWRegionAlignment = 0x1000;

    const char* name = nullptr;

    uintptr_t tempBuffer = reinterpret_cast<uintptr_t>(Memory::GetInstance()->Allocate(RWRegionAlignment * 2));

    // Look for the module name at the start of the RO section
    if (name == nullptr)
    {
        TargetApplication::GetCurrent()->ReadMemory(tempBuffer, baseaddr, MaximumFilePathLength);
        uint64_t tempValue = *reinterpret_cast<uint64_t*>(tempBuffer);
        uint64_t value = (tempValue << 32) | (tempValue >> 32);
        if (value <= MaximumFilePathLength)
        {
            name = reinterpret_cast<const char*>(tempBuffer + 8);
        }
    }

    // Look for the module name before the .note.gnu.build-id section
    if (name == nullptr)
    {
        const uint32_t buildIdHeader = GnuBuildId::GnuId;
        const size_t buildIdHeaderU32Count = 4;

        size_t readSize = RWRegionAlignment + sizeof(GnuBuildId) + MaximumFilePathLength + 8;
        size_t regionSize = endaddr - baseaddr;
        readSize = std::min(readSize, regionSize);
        TargetApplication::GetCurrent()->ReadMemory(tempBuffer, endaddr - readSize, readSize);
        uintptr_t tempEnd = tempBuffer + readSize;

        uint32_t* searchAddr = reinterpret_cast<uint32_t*>(tempEnd - 4);
        uint32_t* endSearchAddr = reinterpret_cast<uint32_t*>(tempEnd - RWRegionAlignment - sizeof(GnuBuildId));
        if (endSearchAddr < reinterpret_cast<uint32_t*>(tempBuffer))
        {
            endSearchAddr = reinterpret_cast<uint32_t*>(tempBuffer);
        }

        while (searchAddr >= endSearchAddr && *searchAddr != buildIdHeader)
        {
            --searchAddr;
        }
        searchAddr -= buildIdHeaderU32Count;

        if (searchAddr < endSearchAddr)
        {
            Memory::GetInstance()->Free(reinterpret_cast<void*>(tempBuffer));
            return false;
        }

        uint8_t* moduleIdFinder = reinterpret_cast<uint8_t*>(searchAddr) + 3;
        uint8_t* endIdSearchAddr = reinterpret_cast<uint8_t*>(searchAddr) - MaximumFilePathLength - 8;
        if (endIdSearchAddr < reinterpret_cast<uint8_t*>(tempBuffer))
        {
            endIdSearchAddr = reinterpret_cast<uint8_t*>(tempBuffer);
        }

        while (moduleIdFinder >= endIdSearchAddr && *moduleIdFinder == 0)
        {
            --moduleIdFinder;
        }

        if (moduleIdFinder < endIdSearchAddr)
        {
            Memory::GetInstance()->Free(reinterpret_cast<void*>(tempBuffer));
            return false;
        }

        uint8_t* endModuleName = moduleIdFinder;

        while (moduleIdFinder >= endIdSearchAddr && *moduleIdFinder != 0)
        {
            --moduleIdFinder;
        }

        if (moduleIdFinder - 7 < endIdSearchAddr)
        {
            Memory::GetInstance()->Free(reinterpret_cast<void*>(tempBuffer));
            return false;
        }

        uint8_t* startModuleName = moduleIdFinder + 1;
        uint64_t* moduleIdReader = reinterpret_cast<uint64_t*>(moduleIdFinder - 7);
        uint64_t calculatedNameLength = static_cast<uint64_t>(endModuleName - startModuleName + 1);

        uint64_t tempValue = *moduleIdReader;
        uint64_t storedNameLength = (tempValue << 32) | (tempValue >> 32);
        if (storedNameLength <= MaximumFilePathLength && storedNameLength == calculatedNameLength)
        {
            name = reinterpret_cast<const char*>(moduleIdReader + 1);
        }
    }

    if (name != nullptr)
    {
        strncpy(pOutName, name, MaximumFilePathLength);
        pOutName[MaximumFilePathLength - 1] = '\0';
    }

    Memory::GetInstance()->Free(reinterpret_cast<void*>(tempBuffer));

    return name != nullptr;
}


bool TargetApplication::FindModuleBuildId(uintptr_t baseaddr, uintptr_t endaddr, void* gnuBuildId)
{
    GnuBuildId* buildId = static_cast<GnuBuildId*>(gnuBuildId);

    NN_UNUSED(baseaddr);
    NN_SDK_ASSERT(baseaddr <= endaddr);

    const int RWRegionAlignment = 0x1000;

    uintptr_t tempBuffer = reinterpret_cast<uintptr_t>(Memory::GetInstance()->Allocate(RWRegionAlignment * 2));

    size_t readSize = RWRegionAlignment + sizeof(GnuBuildId);
    size_t regionSize = endaddr - baseaddr;
    readSize = std::min(readSize, regionSize);
    TargetApplication::GetCurrent()->ReadMemory(tempBuffer, endaddr - readSize, readSize);
    uintptr_t tempEnd = tempBuffer + readSize;

    uint32_t* searchAddr = reinterpret_cast<uint32_t*>(tempEnd - 4);
    uint32_t* endSearchAddr = reinterpret_cast<uint32_t*>(tempEnd - readSize);
    if (endSearchAddr < reinterpret_cast<uint32_t*>(tempBuffer))
    {
        endSearchAddr = reinterpret_cast<uint32_t*>(tempBuffer);
    }

    while (searchAddr >= endSearchAddr && *searchAddr != GnuBuildId::GnuId)
    {
        --searchAddr;
    }
    searchAddr -= 3;

    if (searchAddr >= endSearchAddr)
    {
        buildId->Fill(searchAddr);
    }

    Memory::GetInstance()->Free(reinterpret_cast<void*>(tempBuffer));

    return (searchAddr >= endSearchAddr);
}



/*****************************************************************************
    Class Definitions
 *****************************************************************************/
TargetApplication::TargetApplication() :
    m_sdkVersion(0),
    m_minCodeAddress(UINT64_MAX),
    m_maxCodeAddress(0),
    m_minStaticCodeAddress(UINT64_MAX),
    m_maxStaticCodeAddress(0),
    m_codeRegionCount(0),
    m_staticCodeRegionCount(0),
    m_profilingMode(ProfilingMode_Disabled),
    m_isAttached(false)
{
    m_processId = nn::os::ProcessId::GetInvalidId().value;
    m_libraryPid = nn::os::ProcessId::GetInvalidId().value;

    sThreadInfo = &m_threadList;

    // Default core mask to standard user applications
    SetCoreMask((1u << 0) | (1u << 1) | (1u << 2) | (0u << 3));
}



TargetApplication::~TargetApplication()
{
}



nn::Result TargetApplication::Kill()
{
    DEBUG_LOG("Attempting to kill target application\n");
    if (sTargetProcessHandle.IsValid())
    {
        sKill = true;
        RequestBreak();
    }
    else
    {
        WARNING_LOG("Attempted to kill an invalid handle\n");
    }
    return nn::ResultSuccess();
}



nn::Result TargetApplication::Close()
{
    nn::Result result;

    DEBUG_LOG("Attempting to close target application\n");

    if (sTargetProcessHandle.IsValid())
    {
        DEBUG_LOG("Attempting to close target handle\n");
        result = nn::svc::CloseHandle(sTargetProcessHandle);
        if (nn::svc::ResultInvalidHandle::Includes(result))
        {
            sTargetProcessHandle = nn::svc::INVALID_HANDLE_VALUE;
        }
        else if (result.IsFailure())
        {
            ERROR_LOG("Error closing handle\n");
            DumpResultInformation(LOG_AS_ERROR, result);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    else
    {
        WARNING_LOG("Target process handle is invalid and cannot be closed\n");
    }

    m_processId = nn::os::ProcessId::GetInvalidId().value;
    m_isAttached = false;

    result = SetProfilingMode(ProfilingMode_InProcess);
    if (nn::profiler::ResultInvalidProfilerStatus::Includes(result))
    {
        SetProfilingMode(ProfilingMode_Disabled);
    }

    AvailableProfilingModesChanged();

    sThreadInfo->Clear();
    sTargetProcessHandle = nn::svc::INVALID_HANDLE_VALUE;

    return nn::ResultSuccess();
}



bool TargetApplication::IsAttached() const
{
    return this->m_isAttached;
}



bool TargetApplication::IsLibraryInitialized() const
{
    return (this->m_libraryPid != nn::os::ProcessId::GetInvalidId().value);
}



bool TargetApplication::IsAttachedToLibrary() const
{
    return (IsAttached() && (this->m_processId == this->m_libraryPid));
}



uint32_t TargetApplication::GetCoreMask()
{
    return m_CoreMask;
}



void TargetApplication::SetCoreMask(uint32_t mask)
{
    const uint32_t supportedMask = 0x7; //((1 << SupportedCoreCount) - 1);
    //NN_SDK_ASSERT((mask & ~supportedMask) == 0);
    mask &= supportedMask;
    m_CoreCount = nn::util::popcount(mask);
    m_CoreMask = mask;
}



int TargetApplication::GetCoreCount()
{
    return m_CoreCount;
}



void TargetApplication::GetApplicationName(const char*& pName, size_t& pLength) const
{
    if (m_staticCodeRegionCount > 0)
    {
        int count = m_staticCodeRegionCount;
        for (auto& x : gPastModules)
        {
            if (count <= 0) { break; }
            --count;

            int nameLength = nn::util::Strnlen(x.Name, MaximumFilePathLength);
            if (nameLength > 4)
            {
                int compareValue = nn::util::Strnicmp(
                    &x.Name[nameLength - 4],
                    ".nss",
                    4);
                if (compareValue == 0)
                {
                    pName = x.Name;
                    pLength = static_cast<size_t>(nameLength);
                    return;
                }
            }
        }
    }

    // Return the program name. This will likely be wrong.
    // The default is "Application"; which is not really helpful but better than nothing.
    pName = sProcessInfo.programName;

    int len = nn::util::Strnlen(pName, 12);
    NN_SDK_ASSERT(len >= 0);
    pLength = static_cast<size_t>(len);
}



nn::Result TargetApplication::GetProcessMemoryInfo(
    int *pOutCount,
    nn::svc::MemoryInfo *pOutMemoryInfo,
    int arrayCount)
{
    NN_SDK_ASSERT(pOutCount != nullptr);
    NN_SDK_ASSERT(pOutMemoryInfo != nullptr);

    uintptr_t addr = 0;
    int currentInfo = 0;

    while (currentInfo < arrayCount)
    {
        nn::svc::MemoryInfo memInfo;
        nn::svc::PageInfo pageInfo;
        nn::Result result = QueryDebugProcessMemory(&memInfo, &pageInfo, addr);
        if (result.IsFailure())
        {
            ERROR_LOG("Failed to query address: %p\n", addr);
            DumpResultInformation(LOG_AS_ERROR, result);
            return result;
        }

        if (memInfo.permission != nn::svc::MemoryPermission_None)
        {
            memcpy(&pOutMemoryInfo[currentInfo], &memInfo, sizeof(memInfo));
            ++currentInfo;
        }
        else if (memInfo.state == nn::svc::MemoryState_Inaccessible ||
            static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size) <= addr)
        {
            break;
        }

        addr = static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size);
    }

    if (pOutCount != nullptr) { *pOutCount = currentInfo; }

    return nn::ResultSuccess();
}



nn::Result TargetApplication::QueryMemory(
    nn::svc::MemoryInfo* pMemInfo,
    nn::svc::PageInfo* pPageInfo,
    uintptr_t address)
{
    return QueryDebugProcessMemory(pMemInfo, pPageInfo, address);
}



nn::Result TargetApplication::IsValidCodeAddress(uintptr_t address)
{
    nn::Result result;
    nn::svc::MemoryInfo memInfo;
    nn::svc::PageInfo pageInfo;

    result = QueryMemory(&memInfo, &pageInfo, address);
    if (result.IsFailure())
    {
        return result;
    }

    bool validCodeRegion = ((memInfo.permission == nn::svc::MemoryPermission_ReadExecute) &&
        (memInfo.state == nn::svc::MemoryState_Code || memInfo.state == nn::svc::MemoryState_AliasCode));
    if (!validCodeRegion)
    {
        ERROR_LOG("Address %p is not a valid code region\n");
        return nn::profiler::ResultInvalidArgument(); // todo: more specific?
    }

    return nn::ResultSuccess();
}



void TargetApplication::FindCodeRegions()
{
    const int MemoryInfoCount = 1024;
    int count = 0;
    nn::svc::MemoryInfo *memoryInfo = reinterpret_cast<nn::svc::MemoryInfo*>(
        Memory::GetInstance()->Allocate(
            MemoryInfoCount * sizeof(nn::svc::MemoryInfo),
            alignof(nn::svc::MemoryInfo)));
    nn::Result result = GetProcessMemoryInfo(&count, memoryInfo, MemoryInfoCount);
    if (result.IsFailure())
    {
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    ClearModuleLists();
    BuildStaticModules(memoryInfo, count);
    FindStaticCodeRegions();
    BuildActiveModules(memoryInfo, count);
    FindDynamicCodeRegions();

    Memory::GetInstance()->Free(memoryInfo);
}



void TargetApplication::FindStaticCodeRegions()
{
    m_staticCodeRegionCount = 0;
    m_minStaticCodeAddress = UINT64_MAX;
    m_maxStaticCodeAddress = 0;

    for (auto& x : gPastModules)
    {
        m_minStaticCodeAddress = std::min(m_minStaticCodeAddress, x.Address);
        m_maxStaticCodeAddress = std::max(m_maxStaticCodeAddress, x.Address + x.Size);
    }

    m_staticCodeRegionCount = gPastModules.size();
    m_codeRegionCount = m_staticCodeRegionCount;
}



void TargetApplication::FindDynamicCodeRegions()
{
    m_minCodeAddress = m_minStaticCodeAddress;
    m_maxCodeAddress = m_maxStaticCodeAddress;

    for (auto&x : gActiveModules)
    {
        m_minCodeAddress = std::min(m_minCodeAddress, x.Address);
        m_maxCodeAddress = std::max(m_maxCodeAddress, x.Address + x.Size);
    }

    m_codeRegionCount = m_staticCodeRegionCount + gActiveModules.size();
}



uintptr_t TargetApplication::GetMinCodeAddress() const
{
    NN_STATIC_ASSERT(sizeof(uintptr_t) <= sizeof(m_minCodeAddress));
    return static_cast<uintptr_t>(m_minCodeAddress);
}



uintptr_t TargetApplication::GetMaxCodeAddress() const
{
    NN_STATIC_ASSERT(sizeof(uintptr_t) <= sizeof(m_maxCodeAddress));
    return static_cast<uintptr_t>(m_maxCodeAddress);
}



int TargetApplication::GetCodeRegionCount() const
{
    return m_codeRegionCount;
}



bool TargetApplication::Is64Bit() const
{
    return (sProcessInfo.flag & nn::svc::CreateProcessParameterFlag_64Bit);
}



size_t TargetApplication::GetPointerSize() const
{
    return Is64Bit() ? 8 : 4;
}



nn::Result TargetApplication::ReadMemory(uintptr_t buffer, uintptr_t address, size_t size) const
{
    VERBOSE_LOG("Target application read memory\n");
    nn::Result result;
    result = nn::svc::ReadDebugProcessMemory(buffer, sTargetProcessHandle, address, size);
    if (result.IsFailure())
    {
        ERROR_LOG("ReadDebugMemory failed for address %p, size %d\n", address, static_cast<unsigned int>(size));
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    return result;
}



nn::Result TargetApplication::WriteMemory(uintptr_t buffer, uintptr_t address, size_t size)
{
    DEBUG_LOG("Target application write memory\n");
    nn::Result result;
    result = nn::svc::WriteDebugProcessMemory(sTargetProcessHandle, buffer, address, size);
    if (result.IsFailure())
    {
        ERROR_LOG("WriteDebugMemory failed for address %p, size %d\n", address, static_cast<unsigned int>(size));
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    return result;
}



nn::Result TargetApplication::GetThreadContext(
    nn::svc::ThreadContext* context,
    nn::os::ThreadId threadId,
    uint32_t flags)
{
    VERBOSE_LOG("Target application get thread context: %lu\n", threadId);
    nn::Result result;
    result = nn::svc::GetDebugThreadContext(context, sTargetProcessHandle, threadId, flags);
    if (result.IsFailure())
    {
        ERROR_LOG("GetDebugThreadContext failed for thread %d\n", threadId);
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    if (!Is64Bit())
    {
        // We need to fill in the specialized fields as the SVC call does not do this for us
        context->fp = context->r[11];
        context->sp = context->r[13];
        context->lr = context->r[14];
    }
    return result;
}



nn::Result TargetApplication::RequestBreak()
{
    VERBOSE_LOG("Target application break requested\n");
    nn::Result result;
    result = nn::svc::BreakDebugProcess(sTargetProcessHandle);
    if (result.IsFailure())
    {
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    return result;
}


namespace /*anonymous*/
{
    template<typename T> struct ThreadStackPointersImpl
    {
        T stack;
        T stackSize;
        T argument;
        T threadFunction;
        T currentFiber;
    };

    union ThreadStackPointers
    {
        ThreadStackPointersImpl<uint32_t> bit32;
        ThreadStackPointersImpl<uint64_t> bit64;
    };

    #define STACK_OFFSET_IN_THREADTYPE(N) (8 * (N) + 1 + 1 + 1 + 1 + 4 + (N))
    NN_STATIC_ASSERT(STACK_OFFSET_IN_THREADTYPE(sizeof(uintptr_t)) == offsetof(nn::os::ThreadType, _stack));

    #define STACK_OFFSET_IN_FIBERTYPE(N) (4 * (N) + 1 + 1 + ((N) - 2) + 3 * (N))
    NN_STATIC_ASSERT(STACK_OFFSET_IN_FIBERTYPE(sizeof(uintptr_t)) == offsetof(nn::os::FiberType, _stack));

    inline void SetStackDetails(nn::profiler::ThreadListItem* info, ThreadStackPointers sp, bool is64bit)
    {
        if (is64bit)
        {
            info->stackBase = sp.bit64.stack + sp.bit64.stackSize;
            info->stackSize = sp.bit64.stackSize;
        }
        else
        {
            info->stackBase = sp.bit32.stack + sp.bit32.stackSize;
            info->stackSize = sp.bit32.stackSize;
        }
    }

    inline bool IsAddressOnStack(nn::profiler::ThreadListItem* info, uintptr_t sp)
    {
        return ((info->stackBase - info->stackSize <= sp) && (sp < info->stackBase));
    }

    inline bool IsInFiber(ThreadStackPointers sp, bool is64bit)
    {
        if (is64bit) { return sp.bit64.currentFiber != NULL; }
        else { return sp.bit32.currentFiber != NULL; }
    }
} // namespace <anonymous>


bool TargetApplication::GetStackStartFromThreadId(
    nn::os::ThreadId thread,
    uintptr_t *stackBase,
    uintptr_t sp) const
{
    auto info = sThreadInfo->GetByKey(thread);
    if (info != nullptr)
    {
        if (IsAddressOnStack(info, sp))
        {
            *stackBase = info->stackBase;
            return true;
        }

        uintptr_t addr = reinterpret_cast<uintptr_t>(info->threadType);
        addr += STACK_OFFSET_IN_THREADTYPE(GetPointerSize());

        ThreadStackPointers values;
        uintptr_t buffer = reinterpret_cast<uintptr_t>(&values);
        nn::Result result = ReadMemory(buffer, addr, sizeof(values));
        if (result.IsSuccess())
        {
            if (IsInFiber(values, Is64Bit()))
            {
                size_t readSize;
                if (Is64Bit())
                {
                    buffer = reinterpret_cast<uintptr_t>(&values.bit64.stack);
                    addr = values.bit64.currentFiber + STACK_OFFSET_IN_FIBERTYPE(GetPointerSize());
                    readSize = sizeof(uint64_t) * 2;
                }
                else
                {
                    buffer = reinterpret_cast<uintptr_t>(&values.bit32.stack);
                    addr = values.bit32.currentFiber + STACK_OFFSET_IN_FIBERTYPE(GetPointerSize());
                    readSize = sizeof(uint32_t) * 2;
                }
                result = ReadMemory(buffer, addr, readSize);
                if (result.IsFailure())
                {
                    DumpResultInformation(LOG_AS_WARNING, result);
                    return false;
                }
            }

            if (stackBase != nullptr)
            {
                SetStackDetails(info, values, Is64Bit());
                *stackBase = info->stackBase;
                return IsAddressOnStack(info, sp);
            }
        }
        else
        {
            DumpResultInformation(LOG_AS_WARNING, result);
        }
    }
    return false;
}



ThreadList* TargetApplication::GetThreadList()
{
    return &m_threadList;
}



void TargetApplication::UpdateThreadInfo()
{
    nn::profiler::UpdateThreadInfo();
}



uint32_t TargetApplication::GetSdkVersion() const
{
    return this->m_sdkVersion;
}



void TargetApplication::SetSdkVersion(uint32_t version)
{
    this->m_sdkVersion = version;
}



ProfilingMode TargetApplication::GetProfilingMode() const
{
    return this->m_profilingMode;
}



nn::Result TargetApplication::SetProfilingMode(ProfilingMode mode)
{
    if ((mode == ProfilingMode_OutOfProcess && !IsOutOfProcessAvailable()) ||
        (mode == ProfilingMode_InProcess && !IsInProcessAvailable()))
    {
        return nn::profiler::ResultInvalidProfilerStatus();
    }

    switch (mode)
    {
    case ProfilingMode_Disabled:
    case ProfilingMode_InProcess:
    case ProfilingMode_OutOfProcess:
        this->m_profilingMode = mode;
        return nn::ResultSuccess();

    default:
        return nn::profiler::ResultInvalidArgument();
    }
}



ProfilingMode TargetApplication::GetAllAvailableProfilingModes() const
{
    ProfilingMode modes = ProfilingMode_Disabled;
    if (IsOutOfProcessAvailable()) { modes |= ProfilingMode_OutOfProcess; }
    if (IsInProcessAvailable()) { modes |= ProfilingMode_InProcess; }
    return modes;
}



NN_NOTAILCALL void TargetApplication::AvailableProfilingModesChanged()
{
    auto modes = GetAllAvailableProfilingModes();
    INFO_LOG("TA Sending Available Profiling Modes: %d\n", modes);
    FORCE_SEND(CommSendMessage(
        ProfilerCommMessage_AvailableProfilingModes,
        &modes,
        sizeof(modes),
        0,
        false));
}



void TargetApplication::StartProfiling(int64_t waitTime)
{
    this->UpdateThreadInfo();

    if (waitTime != nn::svc::WAIT_INFINITE)
    {
        sWaitTime = waitTime;
        //sTimeoutBreaks = true;
        //RequestBreak();
    }
    else
    {
        sWaitTime = DefaultWaitTimeNanoSeconds;
        sTimeoutBreaks = false;
    }

#if defined(NN_PROFILER_TIMING_ENABLE)
    for (int i = 0; i < TimerBlockId_Count; ++i)
    {
        NN_PROFILER_TIMING_CLEAR(&sTimers[i]);
    }
#endif
}



void TargetApplication::StopProfiling()
{
    sTimeoutBreaks = false;
    sWaitTime = DefaultWaitTimeNanoSeconds;
}



void TargetApplication::ChangeWatchThreadAffinity(int core)
{
    NN_SDK_ASSERT(core >= 0 && core < static_cast<int>(SupportedCoreCount));
    nn::os::SetThreadCoreMask(&sTargetWatchThread, core, (1 << core));
}



void TargetApplication::DumpTimers() const
{
#if defined(NN_PROFILER_TIMING_ENABLE)
    for (int i = 0; i < TimerBlockId_Count; ++i)
    {
        nn::TimeSpan ts = nn::os::ConvertToTimeSpan(nn::os::Tick((int64_t)sTimers[i].GetAverageTime()));
        NN_UNUSED(ts);
        FORCE_LOG("TargetApplication Timer Block %d: %lluns -- %lld\n", i, ts.GetNanoSeconds(), sTimers[i].GetCount());
    }
#endif
}



nn::svc::Handle TargetApplication::GetDebugHandle()
{
    return sTargetProcessHandle;
}



int64_t TargetApplication::GetWaitTime()
{
    return sWaitTime;
}



bool TargetApplication::IsOutOfProcessAvailable() const
{
    return (IsAttached() && SampleBuffers::GetInstance()->IsInitialized());
}



bool TargetApplication::IsInProcessAvailable() const
{
    return (IsLibraryInitialized());
}



} // profiler
} // nn
