﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_ThreadTypes.h>

#include "profiler_PccomTaskQueue.h"

namespace nn { namespace profiler { namespace pccom
{
    nn::Result Initialize();


    enum PccomState
    {
        PccomState_Waiting,
        PccomState_Connected,
        PccomState_Disconnected,
    };


    typedef void (*RecvCompleteCallback)(
        void *destination,
        size_t size,
        uint32_t message);

    typedef void (*RecvCallback)(
        void **destination,
        size_t size,
        uint32_t message,
        RecvCompleteCallback *callback);

    typedef void(*StateChangeCallback)(
        PccomState state);



    struct MessageHeader
    {
        uint64_t size;
        uint32_t message;
        uint32_t partsRemaining;
    };



    class TwoWayCommunication
    {
    public:
        TwoWayCommunication();

        nn::Result Initialize(RecvCallback recv_callback, StateChangeCallback state_callback);

        void CreateTemporaryHeap();
        void DestroyTemporaryHeap();

        nn::Result Send(
            uint32_t message,
            const void *data,
            size_t data_size,
            SendCallback callback,
            bool copy = true);

        nn::Result SendImmediate(
            uint32_t message,
            const void *data,
            size_t data_size,
            SendCallback callback);

        nn::Result StartMultipartSend(
            uint32_t message,
            size_t total_transfer_size,
            uint32_t total_pieces,
            SendCallback callback);

        nn::Result StartMultipartSendImmediate(
            uint32_t message,
            size_t total_transfer_size,
            uint32_t total_pieces,
            SendCallback callback);

        nn::Result SendMultipart(
            const void *data,
            size_t data_size,
            uint32_t remaining,
            bool copy = true);

        nn::Result SendMultipartImmediate(
            const void *data,
            size_t data_size,
            uint32_t remaining);

        bool IsPCConnected() const;

    protected:
        nn::Result SendImmediate(const TaskItem * const item);

    private:
        nn::os::ThreadType m_sendThread;
        nn::os::ThreadType m_recvThread; // overloaded to also wait for connection
        int m_socket;
        volatile int m_serviceSocket;
        RecvCallback m_recvCallback;
        StateChangeCallback m_stateChangeCallback;
        volatile bool m_socketIsClosed;
        TaskQueue m_queue;

        uint32_t m_multipartPartsRemaining;
        uint32_t m_multipartMessage;
        SendCallback m_multipartTaskCallback;

    private:
        void WaitForConnection();
        void SendThreadProc();
        void RecvThreadProc();
        void FinalizeSendProc();
        bool ReadOnePiece(void *destination, size_t size);
        void ClearSendQueue();

        friend void TwoWayWaitForConnection(void *that);
        friend void TwoWaySendThreadProc(void *that);
    };

    void TwoWayWaitForConnection(void *that);
    void TwoWaySendThreadProc(void *that);

} // pccom
} // profiler
} // nn
