﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_ThreadCommon.h>

namespace nn { namespace profiler {

/*---------------------------------------------------------------------------*
  Constants
 *---------------------------------------------------------------------------*/

// The profiler currently uses about 200KB of its extra memory.
const size_t ProfilerExtraMemory = (1 * 1024 * 1024) - nn::os::GuardedStackAlignment;
const size_t SampleMemoryBlockSize = (1 * 1024 * 1024);
const size_t ReadStackStorageSize = (96 * 1024);

enum SampleBufferIndex : uint32_t
{
    SampleBufferIndex_Core0,
    SampleBufferIndex_Core1,
    SampleBufferIndex_Core2,
    SampleBufferIndex_Core3,
    SampleBufferIndex_Instrumentation,
    SampleBufferIndex_MAX,
};


/*---------------------------------------------------------------------------*
  Type Declarations
 *---------------------------------------------------------------------------*/
typedef struct SampleBufferListItem
{
    void* memory;
    size_t size;
} SampleBufferListItem;



typedef struct SampleBufferCoreControl
{
    uint32_t listCount NN_ALIGNAS(32);
    size_t totalSize;

    SampleBufferListItem *list;
} SampleBufferCoreControl;



class Memory
{
    NN_DISALLOW_COPY(Memory);
    NN_DISALLOW_MOVE(Memory);

public:
    static Memory* GetInstance();

    bool Initialize(void* buffer, size_t size);
    void Finalize();
    bool IsInitialized() const;

    bool AddSecondHeap(void* buffer, size_t size);
    bool IsSecondHeapInitialized() const;

    void* Allocate(size_t size);
    void* Allocate(size_t size, int alignment);
    void Free(void* ptr);

    template <typename T>
    inline T* Allocate()
    {
        return reinterpret_cast<T*>(Allocate(sizeof(T), alignof(T)));
    }

    template <typename T>
    inline T* Allocate(int alignment)
    {
        int a = alignment;
        if (a > 0 && a < alignof(T)) { a = alignof(T); }
        else if (a < 0 && -a < alignof(T)) { a = -alignof(T); }
        return reinterpret_cast<T*>(Allocate(sizeof(T), a));
    }

    size_t GetFreeSpace() const;
    size_t GetTotalSize() const;

protected:

private:
    static Memory* s_instance;
    bool m_isInitialized;

private:
    Memory();
};



class SampleBuffers
{
    NN_DISALLOW_COPY(SampleBuffers);
    NN_DISALLOW_MOVE(SampleBuffers);

public:
    static SampleBuffers * GetInstance();

    bool Initialize(void* buffer, size_t size);
    void Finalize();
    bool IsInitialized() const;

    size_t GetSize() const;
    void* GetStartAddress();

    void* Allocate(SampleBufferIndex index);
    void Reset();
    const SampleBufferCoreControl* GetBuffer(SampleBufferIndex index);

    bool HasData() const;

    void CloseCurrent(void* currentPtr, SampleBufferIndex index);
    void FinalizeBuffer(SampleBufferIndex index);

protected:

private:
    static SampleBuffers* s_instance;
    bool m_isInitialized;

private:
    SampleBuffers();
};

} // profiler
} // nn
