﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <atomic>
#include <new>
#include <algorithm>

#include <nn/nn_Common.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma GCC diagnostic ignored "-Wsign-conversion"
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Result.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/diag.h>
NN_PRAGMA_POP_WARNINGS

#include "profiler_Defines.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"



namespace nn { namespace profiler {

#define NN_PROFILER_DEBUG_MEMORY_SYSTEM (0 && !NN_SDK_BUILD_RELEASE)


/*---------------------------------------------------------------------------*
  Constants
 *---------------------------------------------------------------------------*/
#if NN_PROFILER_DEBUG_MEMORY_SYSTEM
// This can be changed to show additional details when allocating
// and deallocating memory. For shipping builds, generally leave
// this set to false.
#define PROFILER_DEBUG_MEMORY_ALLOCATIONS           false
#define PROFILER_DEBUG_MEMORY_FILLTYPE              nn::lmem::CreationOption_DebugFill
#define PROFILER_DEBUG_MEMORY_TRACKMINFREESIZE      1      // NOLINT(preprocessor/const)
#else
// DO NOT CHANGE THIS OFF OF FALSE
#define PROFILER_DEBUG_MEMORY_ALLOCATIONS           false
#define PROFILER_DEBUG_MEMORY_FILLTYPE              0      // NOLINT(preprocessor/const)
#define PROFILER_DEBUG_MEMORY_TRACKMINFREESIZE      0      // NOLINT(preprocessor/const)
#endif


/*---------------------------------------------------------------------------*
  Type Declarations
 *---------------------------------------------------------------------------*/
typedef struct ProfilerMemory
{
    void* memory;
    size_t size;

    nn::lmem::HeapHandle heapHandle;
    nn::lmem::HeapHandle secondHeapHandle;

#if PROFILER_DEBUG_MEMORY_TRACKMINFREESIZE
    size_t minimumFreeSize;
#endif

} ProfilerMemory;



typedef struct SampleBufferControl
{
    void* buffer;
    size_t size;

    std::atomic<uint32_t> currentBlock;
    uint32_t totalBlocks;
} SampleBufferControl;



/*---------------------------------------------------------------------------*
Global Variable Declarations
*---------------------------------------------------------------------------*/
Memory* Memory::s_instance = nullptr;
SampleBuffers* SampleBuffers::s_instance = nullptr;

static NN_ALIGNAS(32) char sMemoryInstance[sizeof(Memory)];
static NN_ALIGNAS(32) char sSampleBuffersInstance[sizeof(SampleBuffers)];



/*---------------------------------------------------------------------------*
  Local Variable Declarations
 *---------------------------------------------------------------------------*/
static ProfilerMemory s_profilerMemory;



/*---------------------------------------------------------------------------*
  Local Function Declarations
 *---------------------------------------------------------------------------*/
NN_ALIGNAS(32) static SampleBufferControl s_sampleBufferControl = { };
static SampleBufferCoreControl s_coreControl[SampleBufferIndex_MAX];

NN_STATIC_ASSERT(SampleBufferIndex_MAX == SupportedCoreCount + 1);



/*---------------------------------------------------------------------------*
  Global API Functions -- class Memory
 *---------------------------------------------------------------------------*/
Memory::Memory() :
    m_isInitialized(false)
{
    // Nothing to do
}



Memory* Memory::GetInstance()
{
    if (Memory::s_instance == nullptr) { Memory::s_instance = new (sMemoryInstance) Memory(); }
    return Memory::s_instance;
}



bool Memory::Initialize(void* buffer, size_t size)
{
    if (buffer == nullptr || size == 0)
    {
        ERROR_LOG("Unable to initialize memory system: %p [%ld]\n", buffer, size);
        return false;
    }

    s_profilerMemory.secondHeapHandle = nullptr;
    s_profilerMemory.memory = buffer;
    s_profilerMemory.size = size;
    s_profilerMemory.heapHandle = nn::lmem::CreateExpHeap(
        buffer,
        size,
        PROFILER_DEBUG_MEMORY_FILLTYPE | nn::lmem::CreationOption_ThreadSafe);
    if (s_profilerMemory.heapHandle == nullptr)
    {
        ERROR_LOG("Unable to create heap\n");
        return false;
    }

#if PROFILER_DEBUG_MEMORY_TRACKMINFREESIZE
    s_profilerMemory.minimumFreeSize = GetFreeSpace();
#endif

    m_isInitialized = true;

    return true;
}



void Memory::Finalize()
{
    m_isInitialized = false;

#if (PROFILER_DEBUG_MEMORY_ALLOCATIONS == true)
    nn::lmem::DumpExpHeap(s_profilerMemory.heapHandle);
#endif

    if (IsSecondHeapInitialized())
    {
        nn::lmem::DestroyExpHeap(s_profilerMemory.secondHeapHandle);
        s_profilerMemory.secondHeapHandle = nullptr;
    }

    nn::lmem::DestroyExpHeap(s_profilerMemory.heapHandle);
}



bool Memory::IsInitialized() const
{
    return m_isInitialized;
}



bool Memory::AddSecondHeap(void* buffer, size_t size)
{
    s_profilerMemory.secondHeapHandle = nn::lmem::CreateExpHeap(
        buffer,
        size,
        PROFILER_DEBUG_MEMORY_FILLTYPE | nn::lmem::CreationOption_ThreadSafe);
    return true;
}


bool Memory::IsSecondHeapInitialized() const
{
    return s_profilerMemory.secondHeapHandle != nullptr;
}



void* Memory::Allocate(size_t size)
{
    return Allocate(size, nn::lmem::DefaultAlignment);
}



void* Memory::Allocate(size_t size, int alignment)
{
    void* memory = nn::lmem::AllocateFromExpHeap(s_profilerMemory.heapHandle, size, alignment);

    if (memory == nullptr && IsSecondHeapInitialized())
    {
        memory = nn::lmem::AllocateFromExpHeap(s_profilerMemory.secondHeapHandle, size, alignment);
    }

#if (PROFILER_DEBUG_MEMORY_ALLOCATIONS == true)
    DEBUG_LOG("Allocated %d bytes at address %p\n", size, memory);
#endif

#if PROFILER_DEBUG_MEMORY_TRACKMINFREESIZE
    size_t freeSpace = GetFreeSpace();
    if (freeSpace < s_profilerMemory.minimumFreeSize)
    {
        s_profilerMemory.minimumFreeSize = freeSpace;
        FORCE_LOG("Free Space decreased to: %lld\n", freeSpace);

        if (size > 20 * 1024)
        {
            const size_t addrCount = 20;
            uintptr_t addr[addrCount];
            int count = nn::diag::GetBacktrace(addr, addrCount);
            FORCE_LOG("Large allocation from\n");
            for (int i = 0; i < count; ++i)
            {
                FORCE_LOG(" %p\n", addr[i]);
            }
        }
    }
#endif

    return memory;
}



void Memory::Free(void* ptr)
{
    if (ptr == nullptr) { return; }

#if (PROFILER_DEBUG_MEMORY_ALLOCATIONS == true)
    DEBUG_LOG("Attempting to free address %p\n", ptr);
#endif

    if (nn::lmem::HasAddress(s_profilerMemory.heapHandle, ptr))
    {
        nn::lmem::FreeToExpHeap(s_profilerMemory.heapHandle, ptr);
    }
    else if (IsSecondHeapInitialized() && nn::lmem::HasAddress(s_profilerMemory.secondHeapHandle, ptr))
    {
        nn::lmem::FreeToExpHeap(s_profilerMemory.secondHeapHandle, ptr);
    }
    else
    {
        NN_ABORT("Could not find the heap this value should be freed to.\n");
    }
}



size_t Memory::GetFreeSpace() const
{
    size_t freeSpace = nn::lmem::GetExpHeapTotalFreeSize(s_profilerMemory.heapHandle);
    if (IsSecondHeapInitialized())
    {
        freeSpace += nn::lmem::GetExpHeapTotalFreeSize(s_profilerMemory.secondHeapHandle);
    }
    return freeSpace;
}



size_t Memory::GetTotalSize() const
{
    size_t totalSize = nn::lmem::GetTotalSize(s_profilerMemory.heapHandle);
    if (IsSecondHeapInitialized())
    {
        totalSize += nn::lmem::GetTotalSize(s_profilerMemory.secondHeapHandle);
    }
    return totalSize;
}



/*---------------------------------------------------------------------------*
Global API Functions -- class SampleBuffers
*---------------------------------------------------------------------------*/
SampleBuffers::SampleBuffers() :
    m_isInitialized(false)
{
    // Do nothing
}



SampleBuffers* SampleBuffers::GetInstance()
{
    if (s_instance == nullptr) { s_instance = new (sSampleBuffersInstance) SampleBuffers(); }
    return s_instance;
}



bool SampleBuffers::Initialize(void* buffer, size_t size)
{
    if (buffer == nullptr || size == 0)
    {
        ERROR_LOG("Unable to initialize sample buffer: %p [%ld]\n", buffer, size);
        return false;
    }

    s_sampleBufferControl.buffer = buffer;
    s_sampleBufferControl.size = size;
    s_sampleBufferControl.totalBlocks =
        static_cast<uint32_t>(size / SampleMemoryBlockSize);

    for (uint32_t i = 0; i < SampleBufferIndex_MAX; ++i)
    {
        s_coreControl[i].list = reinterpret_cast<SampleBufferListItem*>(
            Memory::GetInstance()->Allocate(sizeof(SampleBufferListItem) * s_sampleBufferControl.totalBlocks));
        NN_ABORT_UNLESS_NOT_NULL(s_coreControl[i].list);
    }

    this->Reset();

    m_isInitialized = true;

    return true;
}



void SampleBuffers::Finalize()
{
    m_isInitialized = false;
    auto mem = Memory::GetInstance();
    for (uint32_t core = 0; core < SampleBufferIndex_MAX; ++core)
    {
        mem->Free(s_coreControl[core].list);
        s_coreControl[core].list = nullptr;
    }
    s_sampleBufferControl.buffer = nullptr;
}



bool SampleBuffers::IsInitialized() const
{
    return m_isInitialized;
}



size_t SampleBuffers::GetSize() const
{
    return s_sampleBufferControl.size;
}



void* SampleBuffers::GetStartAddress()
{
    return s_sampleBufferControl.buffer;
}



void* SampleBuffers::Allocate(SampleBufferIndex index)
{
    void* buffer = nullptr;
    uint32_t block;

    block = s_sampleBufferControl.currentBlock.fetch_add(1, std::memory_order_seq_cst);

    DEBUG_LOG("Request to allocate memory for index %d\n", index);
    if (block < s_sampleBufferControl.totalBlocks)
    {
        buffer = reinterpret_cast<void*>(
            reinterpret_cast<uintptr_t>(s_sampleBufferControl.buffer) +
            (block * SampleMemoryBlockSize));

        SampleBufferCoreControl *coreControl = &(s_coreControl[index]);
        SampleBufferListItem *listItem = &(coreControl->list[coreControl->listCount]);
        memset(listItem, 0, sizeof(*listItem));
        coreControl->listCount += 1;
        listItem->memory = buffer;

        DEBUG_LOG(" Allocated memory for index %d at address %p\n", index, buffer);
    }

    return buffer;
}



void SampleBuffers::Reset()
{
    for (uint32_t i = 0; i < SampleBufferIndex_MAX; ++i)
    {
        s_coreControl[i].listCount = 0;
        s_coreControl[i].totalSize = 0;
    }

    s_sampleBufferControl.currentBlock = 0;
}



const SampleBufferCoreControl* SampleBuffers::GetBuffer(SampleBufferIndex index)
{
    NN_SDK_ASSERT_LESS(index, SampleBufferIndex_MAX);
    return &(s_coreControl[index]);
}



bool SampleBuffers::HasData() const
{
    size_t totalSize = 0;
    for (uint32_t i = 0; i <= SampleBufferIndex_Core3; ++i)
    {
        totalSize += s_coreControl[i].totalSize;
    }
    return (totalSize != 0);
}



void SampleBuffers::CloseCurrent(void* currentPtr, SampleBufferIndex index)
{
    NN_SDK_ASSERT_LESS(index, SampleBufferIndex_MAX);
    SampleBufferCoreControl* coreControl = &(s_coreControl[index]);
    int i = static_cast<int>(coreControl->listCount) - 1;

    DEBUG_LOG("Profiler_SampleBuffer_CloseCurrent called on index %d\n", index);

    if (i >= 0)
    {
        SampleBufferListItem *listItem = &(coreControl->list[i]);
        uint32_t sizeOfCurrent = static_cast<uint32_t>(
            reinterpret_cast<uintptr_t>(currentPtr) -
            reinterpret_cast<uintptr_t>(listItem->memory));
        listItem->size = sizeOfCurrent;
    }
}



void SampleBuffers::FinalizeBuffer(SampleBufferIndex index)
{
    NN_SDK_ASSERT_LESS(index, SampleBufferIndex_MAX);
    SampleBufferCoreControl* coreControl = &(s_coreControl[index]);
    int i = static_cast<int>(coreControl->listCount) - 1;

    DEBUG_LOG("Profiler_SampleBuffer_Finalize called on index %d\n", index);

    coreControl->totalSize = 0;
    while (i >= 0)
    {
        SampleBufferListItem* listItem = &(coreControl->list[i]);
        coreControl->totalSize += listItem->size;
        --i;
    }
}



/*---------------------------------------------------------------------------*
  Local Functions
 *---------------------------------------------------------------------------*/

} // profiler
} // nn
