﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/erpt.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/psc.h>

#include <nn/apm/apm.h>
#include <nn/apm/apm_IManager.sfdl.h>
#include <nn/apm/apm_IManagerPrivileged.sfdl.h>
#include <nn/apm/apm_ISystemManager.sfdl.h>
#include <nn/apm/apm_ManagerImpl.h>
#include <nn/apm/apm_PrivilegedShimInternal.h>
#include <nn/apm/apm_ServiceName.h>
#include <nn/apm/apm_ShimInternal.h>
#include <nn/apm/apm_System.h>
#include <nn/apm/apm_SystemManagerImpl.h>
#include <nn/apm/apm_SystemServiceName.h>
#include <nn/apm/apm_SystemShimInternal.h>
#include <nn/apm/detail/apm_Log.h>
#include <nn/apm/server/apm.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardUserSystemClock.h>

#include "ppc_ApmServer.h"

namespace nn  { namespace ppc {

namespace {

#define NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(result) \
    do \
    { \
        if ( (static_cast<nn::Result>(result)).IsFailure() ) \
        { \
            NN_DETAIL_APM_WARN("%s failed.\n", NN_MACRO_STRINGIZE(result)); \
        } \
    } while ( NN_STATIC_CONDITION(false) )

const int ApmSessionCountMax = 4;
const int ApmSessionPrivilegedCountMax = 4;
const int ApmSessionForSystemCountMax = 6; // 4 + 2(DevMenuCommandSystem/DevMenu) TODO: 4 の内訳の確認(OMM, AM の他 2 つ)
const int ApmAllSessionsCountMax = ApmSessionCountMax + ApmSessionPrivilegedCountMax + ApmSessionForSystemCountMax;

const int NumberOfManagers = 3;

class MyServerManager : public nn::sf::HipcSimpleAllInOneServerManager<ApmAllSessionsCountMax, NumberOfManagers>
{};
MyServerManager* g_pMyServerManager;
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;

nn::sf::UnmanagedServiceObject<nn::apm::IManager, nn::apm::ManagerImpl> g_Manager;
nn::sf::UnmanagedServiceObject<nn::apm::IManagerPrivileged, nn::apm::ManagerPrivilegedImpl> g_ManagerPrivileged;
nn::sf::UnmanagedServiceObject<nn::apm::ISystemManager, nn::apm::SystemManagerImpl> g_SystemManager;

const size_t ServerStackSize = 0x4000;
NN_ALIGNAS(nn::os::ThreadStackAlignment)
uint8_t g_ServerStack[ServerStackSize];
nn::os::ThreadType g_ServerThread;

const size_t HandlerStackSize = 0x4000;
NN_ALIGNAS(nn::os::ThreadStackAlignment)
uint8_t g_HandlerStack[HandlerStackSize];
nn::os::ThreadType g_HandlerThread;

#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
const int SocthermEdpInterruptName = 32 + 51; // for soctherm
nn::os::InterruptEventType g_InterruptEvent; // for soctherm
nn::os::MultiWaitHolderType g_InterruptEventHolder;
#endif

// ハンドラスレッド終了用イベント
nn::os::EventType g_ExitEvent;
nn::os::MultiWaitHolderType g_ExitEventHolder;

// ハンドラスレッド用多重待ちヘッダ
nn::os::MultiWaitType g_HandlerMultiWait;

// システム電源状態の遷移通知を受け取るオブジェクト。
nn::psc::PmModule g_PmModule;

nn::os::SystemEventType* g_pStateChangedEvent;
nn::os::SystemEventType* g_pThrottlingDetectedEvent;
nn::os::SystemEventType* g_pThrottlingFinishedEvent;

nn::os::MultiWaitHolderType g_StateChangedEventHolder;
nn::os::MultiWaitHolderType g_ThrottlingDetectedEventHolder;
nn::os::MultiWaitHolderType g_ThrottlingFinishedEventHolder;
nn::os::MultiWaitHolderType g_PmEventHolder;

bool g_DfcInitialized = false;

void UndockHandler() NN_NOEXCEPT
{
    nn::apm::server::HandleSocthermInterrupts();

#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
    nn::os::ClearInterruptEvent(&g_InterruptEvent);
#endif
}

const uintptr_t UndockingTag         = 0x80000002;
const uintptr_t StateChangedTag      = 0x80000004;
const uintptr_t PmEventTag           = 0x80000006;
const uintptr_t ExitTag              = 0x80000007;
const uintptr_t ThrottlingFinishedTag = 0x80000008;
const uintptr_t ThrottlingDetectedTag = 0x80000009;

void InitializeServerEventsAndHolders() NN_NOEXCEPT
{
    // Windows 環境では InitializeInterruptEvent が合致しない。
    // いずれにせよ HARDWARE_NX 以外では割り込みを取ることはできないので実装分岐。
#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
    // Enable detecting soctherm interrupt
    nn::os::InitializeInterruptEvent(&g_InterruptEvent, SocthermEdpInterruptName, nn::os::EventClearMode_ManualClear);
#endif

    nn::os::InitializeEvent(&g_ExitEvent, false, nn::os::EventClearMode_ManualClear);

    nn::os::InitializeMultiWait(&g_HandlerMultiWait);

    g_pStateChangedEvent = nn::apm::server::GetEventForUpdate(nn::apm::server::EventTarget_PsmStateChanged);
    g_pThrottlingDetectedEvent = nn::apm::server::GetEventForUpdate(nn::apm::server::EventTarget_ThrottlingDetected);
    g_pThrottlingFinishedEvent = nn::apm::server::GetEventForUpdate(nn::apm::server::EventTarget_ThrottlingFinished);

    // PSC へのモジュール登録。
    const nn::psc::PmModuleId pmDependencies[] =
    {
        nn::psc::PmModuleId_Bpc,
        nn::psc::PmModuleId_Fgm,
        nn::psc::PmModuleId_Psm,
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.Initialize(nn::psc::PmModuleId_Apm,
        pmDependencies,
        sizeof(pmDependencies) / sizeof(pmDependencies[0]),
        nn::os::EventClearMode_ManualClear));

#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
    nn::os::InitializeMultiWaitHolder(&g_InterruptEventHolder, &g_InterruptEvent);
    nn::os::SetMultiWaitHolderUserData(&g_InterruptEventHolder, UndockingTag);
    nn::os::LinkMultiWaitHolder(&g_HandlerMultiWait, &g_InterruptEventHolder);
#endif

    nn::os::InitializeMultiWaitHolder(&g_ExitEventHolder, &g_ExitEvent);
    nn::os::SetMultiWaitHolderUserData(&g_ExitEventHolder, ExitTag);
    nn::os::LinkMultiWaitHolder(&g_HandlerMultiWait, &g_ExitEventHolder);

    nn::os::InitializeMultiWaitHolder(&g_StateChangedEventHolder, g_pStateChangedEvent);
    nn::os::SetMultiWaitHolderUserData(&g_StateChangedEventHolder, StateChangedTag);
    g_pMyServerManager->AddUserWaitHolder(&g_StateChangedEventHolder);

    nn::os::InitializeMultiWaitHolder(&g_ThrottlingDetectedEventHolder, g_pThrottlingDetectedEvent);
    nn::os::SetMultiWaitHolderUserData(&g_ThrottlingDetectedEventHolder, ThrottlingDetectedTag);
    g_pMyServerManager->AddUserWaitHolder(&g_ThrottlingDetectedEventHolder);

    nn::os::InitializeMultiWaitHolder(&g_ThrottlingFinishedEventHolder, g_pThrottlingFinishedEvent);
    nn::os::SetMultiWaitHolderUserData(&g_ThrottlingFinishedEventHolder, ThrottlingFinishedTag);
    g_pMyServerManager->AddUserWaitHolder(&g_ThrottlingFinishedEventHolder);

    nn::os::InitializeMultiWaitHolder(&g_PmEventHolder, g_PmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(&g_PmEventHolder, PmEventTag); // PmModuleId とローカル定義のタグが衝突しない保証がないので PmModuleId は使わない。
    g_pMyServerManager->AddUserWaitHolder(&g_PmEventHolder);
}

void FinalizeServerEventsAndHolders() NN_NOEXCEPT
{
    nn::os::FinalizeMultiWaitHolder(&g_StateChangedEventHolder);
    nn::os::FinalizeMultiWaitHolder(&g_ThrottlingDetectedEventHolder);
    nn::os::FinalizeMultiWaitHolder(&g_ThrottlingFinishedEventHolder);
    nn::os::FinalizeMultiWaitHolder(&g_PmEventHolder);

#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
    nn::os::UnlinkMultiWaitHolder(&g_InterruptEventHolder);
#endif

    nn::os::UnlinkMultiWaitHolder(&g_ExitEventHolder);

#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
    nn::os::FinalizeMultiWaitHolder(&g_InterruptEventHolder);
#endif
    nn::os::FinalizeMultiWaitHolder(&g_ExitEventHolder);

    nn::os::FinalizeMultiWait(&g_HandlerMultiWait);

    nn::os::FinalizeEvent(&g_ExitEvent);
#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
    nn::os::FinalizeInterruptEvent(&g_InterruptEvent);
#endif

    g_PmModule.Finalize();
}

void InitializeServer(bool initializeDfc) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);

    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;
    g_pMyServerManager->RegisterObjectForPort(g_Manager.GetShared(), ApmSessionCountMax, nn::apm::ApmServiceName);
    g_pMyServerManager->RegisterObjectForPort(g_ManagerPrivileged.GetShared(), ApmSessionPrivilegedCountMax, nn::apm::ApmServiceNamePrivileged);
    g_pMyServerManager->RegisterObjectForPort(g_SystemManager.GetShared(), ApmSessionForSystemCountMax, nn::apm::ApmServiceNameForSystem);

    if ( initializeDfc )
    {
        nn::apm::InitializeWith(g_Manager.GetShared());
        nn::apm::InitializePrivilegedWith(g_ManagerPrivileged.GetShared());
        nn::apm::InitializeForSystemWith(g_SystemManager.GetShared());

        g_DfcInitialized = true;
    }

    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    InitializeServerEventsAndHolders();

    g_pMyServerManager->Start();
}

void HandlePmEvent() NN_NOEXCEPT
{
    nn::psc::PmState state;
    nn::psc::PmFlagSet flags;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&state, &flags));

    switch ( state )
    {
    case nn::psc::PmState_FullAwake:
        nn::apm::server::SetBasePerformanceConfiguration(nn::apm::server::BasePerformanceConfiguration_Application);
        break;

    case nn::psc::PmState_MinimumAwake:
        nn::apm::server::SetExternalAccessEnabled(true);
        nn::apm::server::SetBasePerformanceConfiguration(nn::apm::server::BasePerformanceConfiguration_Background);
        break;

    case nn::psc::PmState_SleepReady:
    case nn::psc::PmState_ShutdownReady:
        nn::apm::server::SetExternalAccessEnabled(false);
        break;

    default:
        break;
    }

    g_PmModule.Acknowledge(state, ResultSuccess());
}

void SubmitThrottlingInfo(bool throttled, int64_t throttlingDuration) NN_NOEXCEPT
{
    nn::time::PosixTime timeStamp;
    auto result = nn::time::StandardUserSystemClock::GetCurrentTime(&timeStamp);
    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(result);
    if ( result.IsFailure() )
    {
        NN_DETAIL_APM_WARN("[server] GetCurrentTime failed.\n");
        timeStamp.value = static_cast<int64_t>(0);
    }

    nn::erpt::Context context(nn::erpt::ThrottlingInfo);
    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::Throttled, throttled));
    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ThrottlingDuration, throttlingDuration));
    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ThrottlingTimestamp, timeStamp.value));
    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
}

void ServerFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    NN_SDK_ASSERT(g_pMyServerManager);

    SubmitThrottlingInfo(false, static_cast<int64_t>(0));

    // FirmwareDebugSettings の初期状態取り込みのために 1 回 Update。
    nn::apm::server::Update();

    while ( auto p = g_pMyServerManager->Wait() )
    {
        uintptr_t userData =  nn::os::GetMultiWaitHolderUserData(p);
        switch ( userData )
        {
        case MyServerManager::InvokeTag:
        case MyServerManager::AcceptTag:
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_pMyServerManager->ProcessAuto(p));
            break;

        case StateChangedTag:
            NN_DETAIL_APM_TRACE("[server] Power supply state changed.\n");

            nn::os::ClearSystemEvent(g_pStateChangedEvent);
            nn::apm::server::Update();
            g_pMyServerManager->AddUserWaitHolder(p);
            break;

        case ThrottlingDetectedTag:
            NN_DETAIL_APM_TRACE("[server] Throttling detected.\n");

            nn::os::ClearSystemEvent(g_pThrottlingDetectedEvent);
            if ( !(nn::os::TryWaitSystemEvent(g_pThrottlingFinishedEvent)) )
            {
                SubmitThrottlingInfo(true, static_cast<int64_t>(0));
            }
            g_pMyServerManager->AddUserWaitHolder(p);
            break;

        case ThrottlingFinishedTag:
            NN_DETAIL_APM_TRACE("[server] Throttling finished.\n");

            nn::os::ClearSystemEvent(g_pThrottlingFinishedEvent);
            nn::apm::ThrottlingState throttlingState;
            nn::apm::server::GetLastThrottlingState(&throttlingState);
            SubmitThrottlingInfo(false, throttlingState.durationNanoSeconds);
            nn::apm::server::Update();
            g_pMyServerManager->AddUserWaitHolder(p);
            break;

        case PmEventTag:
            NN_DETAIL_APM_TRACE("[server] PmEvent signaled.\n");

            g_PmModule.GetEventPointer()->Clear();
            HandlePmEvent();
            g_pMyServerManager->AddUserWaitHolder(p);
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void HandlerFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    NN_SDK_ASSERT(g_pMyServerManager);

    while ( auto p = nn::os::WaitAny(&g_HandlerMultiWait) )
    {
        uintptr_t userData =  nn::os::GetMultiWaitHolderUserData(p);

        switch ( userData )
        {
        case UndockingTag:
            UndockHandler();
            NN_DETAIL_APM_TRACE("[handler] Undocked.\n");

            break;

        case ExitTag:
            NN_DETAIL_APM_TRACE("[handler] Exit soctherm handler.\n");

            nn::os::ClearEvent(&g_ExitEvent);

            return;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void FinalizeServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.Finalize());

    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;

    FinalizeServerEventsAndHolders();

    if ( g_DfcInitialized )
    {
        nn::apm::FinalizeForSystem();
        nn::apm::FinalizePrivileged();
        nn::apm::Finalize();

        g_DfcInitialized = false;
    }

    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
}

} // namespace

void InitializeApmServer(bool initializeDfc) NN_NOEXCEPT
{
    nn::apm::server::Initialize();

    InitializeServer(initializeDfc);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::CreateThread(
            &g_ServerThread,
            ServerFunction,
            nullptr,
            g_ServerStack,
            sizeof(g_ServerStack),
            NN_SYSTEM_THREAD_PRIORITY(apm, IpcServer)));

    nn::os::SetThreadNamePointer(&g_ServerThread, NN_SYSTEM_THREAD_NAME(apm, IpcServer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::CreateThread(
            &g_HandlerThread,
            HandlerFunction,
            nullptr,
            g_HandlerStack,
            sizeof(g_HandlerStack),
            NN_SYSTEM_THREAD_PRIORITY(apm, SocthermHandler)));

    nn::os::SetThreadNamePointer(&g_HandlerThread, NN_SYSTEM_THREAD_NAME(apm, SocthermHandler));
}

void StartApmServer() NN_NOEXCEPT
{
    ::nn::os::StartThread(&g_ServerThread);
    ::nn::os::StartThread(&g_HandlerThread);

    NN_DETAIL_APM_INFO("Start APM Server.\n");
}

void RequestStopApmServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    g_pMyServerManager->RequestStop();
    nn::os::SignalEvent(&g_ExitEvent);
}

void FinalizeApmServer() NN_NOEXCEPT
{
    nn::os::WaitThread(&g_ServerThread);
    nn::os::DestroyThread(&g_ServerThread);

    nn::os::WaitThread(&g_HandlerThread);
    nn::os::DestroyThread(&g_HandlerThread);

    FinalizeServer();

    nn::apm::server::Finalize();
}

// テスト用関数
void GetPmModuleForTest(nn::psc::PmModule** pPmModulePointer) NN_NOEXCEPT
{
    *pPmModulePointer = &g_PmModule;
}

}} // namespace nn::ppc
