﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/pctl_IpcServer.h>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/bgtc/bgtc_Api.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceFactory.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/watcher/pctl_IntermittentOperation.h>
#include <nn/pm/pm_BootModeApi.h>
#include <nn/psc.h>
#include <nn/sf/sf_HipcServer.h>

namespace nn { namespace pctl { namespace detail { namespace service {

namespace
{
    struct PctlServiceHipcSimpleAllInOneServerManagerOption
    {
        static const size_t PointerTransferBufferSize = 1024 - 1;

        static const bool CanDeferInvokeRequest = false;

        static const int SubDomainCountMax = ipc::ProcessCountMax * 2; // サブドメインは少数しか張らない
        static const int ObjectInSubDomainCountMax = 4 * SubDomainCountMax;
    };

    class PctlServiceServerManager :
        public nn::sf::HipcSimpleAllInOneServerManager<
            64,
            ipc::ServiceIndex_Count,
            PctlServiceHipcSimpleAllInOneServerManagerOption
        >
    {
    };

    struct ServicePortDefinition
    {
        ipc::ServiceIndex index;
        int capability;
        size_t sessionCount;
        const char* portName;
    };

    PctlServiceServerManager* g_pServerManager = nullptr;
    std::aligned_storage<sizeof(PctlServiceServerManager)>::type g_ServerManagerStorage;

    static const ServicePortDefinition PortDefinitions[] = {
        { ipc::ServiceIndex_General, ipc::ServiceCapabilityForGeneral, ipc::SessionCountMaxForGeneral, ipc::ServiceNameForGeneral },
        { ipc::ServiceIndex_System, ipc::ServiceCapabilityForSystem, ipc::SessionCountMaxForSystem, ipc::ServiceNameForSystem },
        { ipc::ServiceIndex_Authentication, ipc::ServiceCapabilityForAuthentication, ipc::SessionCountMaxForAuthentication, ipc::ServiceNameForAuthentication },
        { ipc::ServiceIndex_Recovery, ipc::ServiceCapabilityForRecovery, ipc::SessionCountMaxForRecovery, ipc::ServiceNameForRecovery }
    };
    NN_STATIC_ASSERT(static_cast<int>(std::extent<decltype(PortDefinitions)>::value) == static_cast<int>(ipc::ServiceIndex_Count));

    static const nn::psc::PmModuleId g_PmModuleDependencies[] = {
        // ファイル取扱い
        nn::psc::PmModuleId_Fs,
        // 半起床対応
        nn::psc::PmModuleId_Bgtc
    };
}

void ProcessPowerStateEvent(nn::psc::PmModule& pmModule) NN_NOEXCEPT
{
    // ※ メンテナンスモードでは電源イベントハンドルをしないので
    //    ここでは非メンテナンスモード前提

    NN_FUNCTION_LOCAL_STATIC(nn::psc::PmState, s_PmStateCurrent, = nn::psc::PmState_FullAwake);

    nn::psc::PmState state;
    nn::psc::PmFlagSet flags;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pmModule.GetRequest(&state, &flags));
    NN_UNUSED(flags);
    watcher::PowerTransition transition = watcher::PowerTransition_None;
    NN_DETAIL_PCTL_TRACE("[pctl] Power state event: state = %d, previous state = %d\n",
        static_cast<int>(state), static_cast<int>(s_PmStateCurrent));

    switch (state)
    {
        case nn::psc::PmState_FullAwake:
            if (s_PmStateCurrent != nn::psc::PmState_FullAwake)
            {
                transition = watcher::PowerTransition_ToFullAwake;
            }
            break;
        case nn::psc::PmState_MinimumAwake:
            if (s_PmStateCurrent == nn::psc::PmState_FullAwake)
            {
                transition = watcher::PowerTransition_FullAwakeToMinimumAwake;
                // 通常モードから遷移する際にスケジューリングすることをマークする
                // (通常モード以外から遷移する際は不要)
                watcher::IntermittentOperation::SetSchedulingNecessary(true);
            }
            break;
        case nn::psc::PmState_SleepReady:
            // スリープに入る直前では間欠起動中処理のキャンセルを行っておく
            if (IsWatcherAvailable())
            {
                g_pWatcher->GetWatcherEventManager().CancelIntermittentTask();
            }
            // スケジューリングが必要な場合はその処理を行う
            if (watcher::IntermittentOperation::IsSchedulingNecessary())
            {
                // MEMO: pscの依存にbgtcを入れているのでSleepReadyで実行可能
                watcher::IntermittentOperation::ScheduleTask();
                watcher::IntermittentOperation::SetSchedulingNecessary(false);
            }
            break;
        case nn::psc::PmState_EssentialServicesSleepReady:
        case nn::psc::PmState_EssentialServicesAwake:
            // これらの状態はハンドルしない
            break;
        case nn::psc::PmState_ShutdownReady:
            transition = watcher::PowerTransition_ToShutdown;
            break;
        default:
            break;
    }
    s_PmStateCurrent = state;

    // 上記case分でハンドルしていない遷移状態は興味ないので呼び出さない
    if (transition != watcher::PowerTransition_None)
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Power state changed: transition = %d\n",
            static_cast<int>(transition));

        if (IsWatcherAvailable())
        {
            g_pWatcher->GetWatcherEventManager().HandlePowerStateEvent(transition);
        }
    }

    NN_DETAIL_PCTL_TRACE("[pctl] Power state event: acknowledge\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(pmModule.Acknowledge(state, nn::ResultSuccess()));
}

void PerformIntermittentOperation() NN_NOEXCEPT
{
    // (※ nn::bgtc::GetScheduleEvent のトリガーで実行・シグナルのクリアは外で行われる)
    NN_DETAIL_PCTL_TRACE("[pctl] PerformIntermittentOperation()\n");

    // 何も実行されずに終了することもあるため先にスケジュールされるようにマークしておく
    watcher::IntermittentOperation::SetSchedulingNecessary(true);

    if (IsWatcherAvailable())
    {
        auto result = g_pWatcher->GetWatcherEventManager().RequestStartIntermittentTask();
        if (result.IsSuccess())
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Intermittent operation started.\n");
            // (成功時はネットワークのBGタスクとして別スレッドで処理が継続される)
        }
        else
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Failed to start intermittent operation: result = 0x%08lX\n", result.GetInnerValueForDebug());
        }
    }
}

void InitializeServer() NN_NOEXCEPT
{
    {
        auto threadType = nn::os::GetCurrentThread();
        nn::os::ChangeThreadPriority(threadType, NN_SYSTEM_THREAD_PRIORITY(pctl, IpcServer));
        nn::os::SetThreadNamePointer(threadType, NN_SYSTEM_THREAD_NAME(pctl, IpcServer));
    }

    g_pServerManager = new(&g_ServerManagerStorage) PctlServiceServerManager;

    for (const auto& definition : PortDefinitions)
    {
        service::InitializeFactory(definition.index, definition.capability);
        auto result = g_pServerManager->RegisterObjectForPort(
            service::GetFactory(definition.index),
            definition.sessionCount,
            definition.portName);
        if (result.IsFailure())
        {
            NN_DETAIL_PCTL_INFO("[pctl] Failed to initialize port '%s': 0x%08lX\n",
                definition.portName, result.GetInnerValueForDebug());
        }
    }
}

void FinalizeServer() NN_NOEXCEPT
{
    service::FinalizeAllFactories();

    g_pServerManager->~PctlServiceServerManager();
    g_pServerManager = nullptr;
}

void RunServer() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::psc::PmModule, s_PmModule);
    NN_FUNCTION_LOCAL_STATIC(nn::os::MultiWaitHolderType, s_MultiWaitHolderPm);
    NN_FUNCTION_LOCAL_STATIC(nn::os::MultiWaitHolderType, s_MultiWaitHolderBgtcSchedule);
#if !defined(NN_SDK_BUILD_RELEASE)
    NN_FUNCTION_LOCAL_STATIC(nn::os::MultiWaitHolderType, s_MultiWaitHolderBgtcTrigger);
#endif

    bool isMaintenanceMode = (nn::pm::GetBootMode() == nn::pm::BootMode::BootMode_Maintenance);

    InitializeServer();
    InitializeMain();
    // メンテナンスモードのときはWatcherの初期化を行わない
    if (!isMaintenanceMode)
    {
        InitializeWatcher();
    }

    // メンテナンスモード時は電源イベントハンドリングが不要なので省略する
    if (!isMaintenanceMode)
    {
        auto result = s_PmModule.Initialize(
            nn::psc::PmModuleId_Pctl,
            g_PmModuleDependencies,
            static_cast<uint32_t>(std::extent<decltype(g_PmModuleDependencies)>::value),
            nn::os::EventClearMode_ManualClear
            );
        NN_ABORT_UNLESS(result.IsSuccess(),
            "Failed to initialize PmModule: result = 0x%08lX", result.GetInnerValueForDebug());
        nn::os::InitializeMultiWaitHolder(&s_MultiWaitHolderPm, s_PmModule.GetEventPointer()->GetBase());
        g_pServerManager->AddUserWaitHolder(&s_MultiWaitHolderPm);

        result = nn::bgtc::Initialize();
        NN_ABORT_UNLESS(result.IsSuccess(),
            "Failed to initialize bgtc: result = 0x%08lX", result.GetInnerValueForDebug());
        // スケジュールされたイベントのハンドル
        nn::os::InitializeMultiWaitHolder(&s_MultiWaitHolderBgtcSchedule, nn::bgtc::GetScheduleEvent().GetBase());
        g_pServerManager->AddUserWaitHolder(&s_MultiWaitHolderBgtcSchedule);
#if !defined(NN_SDK_BUILD_RELEASE)
        // 半起床に入ったイベントのハンドル(デバッグ用)
        nn::os::InitializeMultiWaitHolder(&s_MultiWaitHolderBgtcTrigger, nn::bgtc::GetTriggerEvent().GetBase());
        g_pServerManager->AddUserWaitHolder(&s_MultiWaitHolderBgtcTrigger);
#endif
    }

    g_pServerManager->Start();

    while (NN_STATIC_CONDITION(true))
    {
        auto p = g_pServerManager->Wait();
        if (p == nullptr)
        {
            break;
        }
        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
            case PctlServiceServerManager::AcceptTag:
            case PctlServiceServerManager::InvokeTag:
                g_pServerManager->ProcessAuto(p);
                break;
            default:
                if (p == &s_MultiWaitHolderPm)
                {
                    s_PmModule.GetEventPointer()->Clear();
                    g_pServerManager->AddUserWaitHolder(&s_MultiWaitHolderPm);
                    ProcessPowerStateEvent(s_PmModule);
                }
                else if (p == &s_MultiWaitHolderBgtcSchedule)
                {
                    NN_DETAIL_PCTL_TRACE("[pctl] Bgtc schedule event triggered\n");
                    nn::bgtc::GetScheduleEvent().Clear();
                    g_pServerManager->AddUserWaitHolder(&s_MultiWaitHolderBgtcSchedule);
                    if (nn::bgtc::IsInHalfAwake())
                    {
                        PerformIntermittentOperation();
                    }
                }
#if !defined(NN_SDK_BUILD_RELEASE)
                else if (p == &s_MultiWaitHolderBgtcTrigger)
                {
                    NN_DETAIL_PCTL_TRACE("[pctl] Bgtc event triggered\n");
                    nn::bgtc::GetTriggerEvent().Clear();
                    g_pServerManager->AddUserWaitHolder(&s_MultiWaitHolderBgtcTrigger);
                    if (nn::bgtc::IsInHalfAwake())
                    {
                        NN_DETAIL_PCTL_TRACE("[pctl]   (in half awake)\n");
                    }
                }
#endif
                break;
        }
    }

    if (!isMaintenanceMode)
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        nn::os::UnlinkMultiWaitHolder(&s_MultiWaitHolderBgtcTrigger);
        nn::os::FinalizeMultiWaitHolder(&s_MultiWaitHolderBgtcTrigger);
#endif
        nn::os::UnlinkMultiWaitHolder(&s_MultiWaitHolderBgtcSchedule);
        nn::os::FinalizeMultiWaitHolder(&s_MultiWaitHolderBgtcSchedule);
        nn::bgtc::Finalize();

        nn::os::UnlinkMultiWaitHolder(&s_MultiWaitHolderPm);
        nn::os::FinalizeMultiWaitHolder(&s_MultiWaitHolderPm);
        NN_ABORT_UNLESS_RESULT_SUCCESS(s_PmModule.Finalize());
    }

    // (FinalizeWatcher は未初期化でも呼び出し可能)
    FinalizeWatcher();
    FinalizeMain();
    FinalizeServer();
}

}}}}

extern "C" void nninitStartup()
{
    nn::pctl::detail::service::InitializeAllocatorForServer();
}

extern "C" void nnMain()
{
    nn::pctl::detail::service::RunServer();
}

// 以下は例外関連コードがリンクされてプロセスが肥大化するのを防ぐためのコード。

void* operator new(size_t size)
{
    return std::malloc(size);
}

void* operator new(size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return std::malloc(size);
}

void operator delete(void* ptr) NN_NOEXCEPT
{
    std::free(ptr);
}

void operator delete(void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    std::free(ptr);
}

void* operator new[](size_t size)
{
    return std::malloc(size);
}

void* operator new[](size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return std::malloc(size);
}

void operator delete[](void* ptr) NN_NOEXCEPT
{
    std::free(ptr);
}

void operator delete[](void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    std::free(ptr);
}
