﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/init.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/pm/pm_BootModeApi.h>
#include <nn/pm/pm_ShellApi.h>
#include <nn/ns/srv/ns_DevelopInterfaceServerFactory.h>
#include <nn/ns/srv/ns_Shell.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Thread.h>
#include <nn/lr/lr_Service.h>
#include <nn/ncm/ncm_Service.h>

#include <nn/ldr/ldr_ShellApi.h>

#include "ns_Infrastructure.h"
#include "ns_DelayedInitializer.h"
#include "ns_NotificationAgentProxy.h"
#include "ns_ApplicationManagerMain.h"
#include "ns_AddOnContentManagerMain.h"

using namespace nn;

// 例外関連コードのリンク抑制のためのオーバーライド
// 参考：http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-34162
void* operator new(std::size_t size) { return malloc(size); }
void* operator new(std::size_t size, const std::nothrow_t&) NN_NOEXCEPT { return malloc(size); }
void  operator delete(void* ptr) NN_NOEXCEPT { free(ptr); }

void* operator new[](std::size_t size) { return malloc(size); }
void* operator new[](std::size_t size, const std::nothrow_t&) NN_NOEXCEPT{ return malloc(size); }
void  operator delete[](void* ptr) NN_NOEXCEPT{ free(ptr); }

namespace
{
    char  g_FilesystemAllocatorHeap[64 * 1024];
    nn::mem::StandardAllocator g_FilesystemAllocator(g_FilesystemAllocatorHeap, sizeof(g_FilesystemAllocatorHeap));
    void* AllocateForFilesystem(size_t size)
    {
        return g_FilesystemAllocator.Allocate(size);
    }
    void DeallocateForFileSystem(void* p, size_t size)
    {
        NN_UNUSED(size);
        return g_FilesystemAllocator.Free(p);
    }

    template <typename Interface>
    void RegisterObjectForPort(nn::sf::SharedPointer<Interface> p, nn::ns::Port port, const char* pPortName)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::ns::GetServerManager()->RegisterObjectForPort(
                p, nn::ns::SessionCount[port], pPortName ));
    }
}   // anonymous namespace

// workaround for SIGLO-52915
#include <nn/sf/sf_Types.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
namespace nn { namespace fs { namespace detail {
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> GetFileSystemProxyServiceObject() NN_NOEXCEPT;
}}}

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[256 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::ldr::InitializeForShell() );

    // DeveloperInterface が利用するので、先に初期化
    nn::ncm::Initialize();
    nn::lr::Initialize();

    auto threadType = nn::os::GetCurrentThread();
    nn::os::ChangeThreadPriority(threadType, NN_SYSTEM_THREAD_PRIORITY(ns, MainEventHandler));
    nn::os::SetThreadNamePointer(threadType, NN_SYSTEM_THREAD_NAME(ns, MainEventHandler));

// TORIAEZU: 汎用ボード (Generic) では DeveloperInterface 以外を初期化しない
// TODO: DeveloperInterface を ns から分離 (SIGLO-83277)
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    nn::ns::InitializeNotificationSenderAndReceiverLibraryForServer();
#endif

    fs::InitializeWithMultiSessionForSystem();
    fs::SetAllocator(AllocateForFilesystem, DeallocateForFileSystem);
    fs::SetEnabledAutoAbort(false);
    fs::detail::GetFileSystemProxyServiceObject();

    auto developInterfaceServer = nn::ns::srv::GetSharedDevelopInterfaceServer();
    nn::ns::srv::DevelopInterfaceServer::SetEventDispatcher(nn::ns::GetServerManager());

    RegisterObjectForPort(developInterfaceServer, nn::ns::Port_Develop, ns::detail::PortNameForDevelop);

// TORIAEZU: 汎用ボード (Generic) では DeveloperInterface 以外を初期化しない
// TODO: DeveloperInterface を ns から分離 (SIGLO-83277)
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    nn::ns::RegisterApplicationManager();

    if(nn::pm::GetBootMode() == nn::pm::BootMode_Normal || nn::pm::GetBootMode() == nn::pm::BootMode_Maintenance)
    {
        nn::ns::RegisterNotificationAgentProxy();
    }

    const auto bootMode = nn::pm::GetBootMode();
    if(bootMode == nn::pm::BootMode_Normal)
    {
        nn::ns::RegisterAddOnContentManager();
    }

    nn::ns::GetServerManager()->Start();

    pm::InitializeForShell();
    ns::srv::StartLaunchControl();

    // 各サービス向けの遅延初期化の開始
    ns::StartDelayedInitialization(bootMode);

    nn::ns::StartApplicationManagerMain();
    nn::ns::StartNotificationAgentProxy();

    if(bootMode == nn::pm::BootMode_Normal)
    {
        nn::ns::StartAddOnContentManagerMain();
    }
#endif

    nn::ns::LoopServerWithEventHandler();
}
