﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Thread.h>

// 初期化対象
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/bgtc/bgtc_Api.h>
#include <nn/es/es_InitializationApi.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemApiForDfc.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/pm/pm_BootModeApi.h>
#include <nn/prepo.h>
#include <nn/time.h>

namespace nn { namespace ns {
namespace
{
struct TransferArguments
{
    nn::pm::BootMode bootMode;
};

os::Event g_EventCompletion(nn::os::EventClearMode_ManualClear);

const size_t StackSize = 4096 * 2;
NN_OS_ALIGNAS_THREAD_STACK Bit8 g_Stack[StackSize];
os::ThreadType g_Thread;

void DelayedInitializationThread(void* arg) NN_NOEXCEPT
{
    const TransferArguments& arguments = *reinterpret_cast<const TransferArguments*>(arg);

    nifm::InitializeSystem();

    NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());
    // nn::time::Finalize()

    if(nn::pm::GetBootMode() == nn::pm::BootMode_Normal || nn::pm::GetBootMode() == nn::pm::BootMode_Maintenance)
    {
        account::InitializeForAdministrator();
    }

    nim::InitializeForNetworkInstallManager();
    // nim::FinalizeForNetworkInstallManager();

    if (nn::pm::BootMode_Normal == arguments.bootMode)
    {
        pdm::InitializeForNotification();
        // pdm::FinalizeForNotification();
    }

    ns::Initialize();
    // nn::ns::Finalize()

    if(nn::pm::GetBootMode() == nn::pm::BootMode_Normal || nn::pm::GetBootMode() == nn::pm::BootMode_Maintenance)
    {
        es::Initialize();
        // nn::es::Finalize()
    }

    bgtc::Initialize();
    // nn::bgtc::Finalize()

    g_EventCompletion.Signal();
}
} // ~namespace nn::ns::<anonymous>

void StartDelayedInitialization(const nn::pm::BootMode& bootMode) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(TransferArguments, s_Arguments);

    s_Arguments.bootMode = bootMode;
    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(
        &g_Thread,
        DelayedInitializationThread, &s_Arguments,
        g_Stack, StackSize,
        NN_SYSTEM_THREAD_PRIORITY(ns, DelayedInitialization)));
    os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(ns, DelayedInitialization));
    os::StartThread(&g_Thread);
}

void WaitDelayedInitialization() NN_NOEXCEPT
{
    g_EventCompletion.Wait();
}

}}  // ~namespace nn::ns
