﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_ApiBaasAccessToken.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include <nn/arp/arp_Api.h>
#endif

#include "npns_Log.h"
#include "npns_Instance.h"
#include "npns_ClientThread.h"
#include "npns_StateMachineThread.h"
#include "npns_SafeStringBuffer.h"
#include "npns_ServiceImpl.h"
#include "npns_Controller.h"
#include "npns_HipcServer.h"
#include "npns_SuspendManager.h"

namespace nn {
namespace npns {

#define NN_NPNS_INVOKE_ON_WORKER_THREAD do{ \
    if (NN_STATIC_CONDITION(false) && !IsWorkerContext())\
    { \
        return sf::DeferProcess(); \
    } \
}while(NN_STATIC_CONDITION(false))

SuspendManager ServiceImpl::s_SuspendManager;
const ApplicationId ApplicationIdForBaaS = ApplicationId::GetInvalidId();

namespace
{
    bool IsTestModeEnabled() NN_NOEXCEPT
    {
# ifndef NN_BUILD_CONFIG_OS_WIN
        bool isEnabled;
        size_t bytes = nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "npns", "test_mode");
        return (bytes == sizeof(isEnabled)) && isEnabled;
#else
        return true; // WIN では常に有効
#endif
    }

    nn::Result GetApplicationId(nn::ApplicationId* pOut, nn::Bit64 processId) NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_HORIZON
        // arp経由でapplicationId取得
        nn::os::ProcessId osProcessId = {processId};
        nn::arp::ApplicationLaunchProperty property;

        Result result = nn::arp::GetApplicationLaunchProperty(&property, osProcessId);
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

        *pOut = { property.id.value };
        return nn::ResultSuccess();
#else
        NN_UNUSED(pOut);
        NN_UNUSED(processId);
        return nn::npns::ResultNotImplemented();
#endif
    }
}

ServiceImpl::ServiceImpl()
    : m_event(os::EventClearMode_AutoClear, true)
    , m_queue(&m_event)
    , m_routerEntry(m_queue)
    , m_SuspendClient(s_SuspendManager)
    , m_SystemEventNode(os::EventClearMode_AutoClear, true)
{
    NN_NPNS_TRACE("IPC session accepted.(%p)\n", this);
    g_Daemon.GetRouter().RegisterRecipient(m_routerEntry);
    g_Daemon.GetStateMachineThread().AttachStateChangeEvent(&m_SystemEventNode);
}

ServiceImpl::~ServiceImpl()
{
    g_Daemon.GetStateMachineThread().DetachStateChangeEvent(&m_SystemEventNode);
    g_Daemon.GetRouter().UnregisterRecipient(m_routerEntry);
    NN_NPNS_TRACE("IPC session closed.(%p)\n", this);
}

Result ServiceImpl::ListenAll() NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    return ResultNotImplemented();
}

Result ServiceImpl::ListenTo(nn::ApplicationId toId) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("(0x%016llx)", toId.value);

    Result result = g_Daemon.GetRouter().AddRceiveTarget(m_routerEntry, toId);
    return result;
}

nn::Result ServiceImpl::ListenToMyApplicationId(nn::Bit64 processId) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    // applicationId 取得
    nn::ApplicationId applicationId;
    Result result = GetApplicationId(&applicationId, processId);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    return ListenTo(applicationId);
}

nn::Result ServiceImpl::ListenUndelivered() NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    Result result = g_Daemon.GetRouter().AddRceiveTarget(m_routerEntry, ApplicationId::GetInvalidId());
    return result;
}

Result ServiceImpl::Receive(const nn::sf::OutBuffer& outBuffer, std::uint16_t version) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    Result result = m_queue.Receive(outBuffer.GetPointerUnsafe(), outBuffer.GetSize(), version);
    return result;
}

Result ServiceImpl::ReceiveRaw(const nn::sf::OutBuffer& outBuffer, std::uint16_t version) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    NN_UNUSED(outBuffer);
    NN_UNUSED(version);
    return ResultNotImplemented();
}

Result ServiceImpl::GetReceiveEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    outHandle.Set(nn::sf::NativeHandle(m_event.GetReadableHandle(), false));
    return ResultSuccess();
}

nn::Result ServiceImpl::GetStateChangeEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    outHandle.Set(nn::sf::NativeHandle(m_SystemEventNode.GetReadableHandle(), false));
    return ResultSuccess();
}

Result ServiceImpl::SubscribeTopic(const nn::sf::InArray<char>& name) NN_NOEXCEPT
{
    NN_NPNS_INVOKE_ON_WORKER_THREAD;

    SafeStringBuffer<TopicNameLength> buffer(name);
    NN_NPNS_TRACE_FUNCTION("(%s)", buffer.GetString());

    Result result = g_Daemon.GetClientThread().SubscribeTopic(buffer.GetString());
    return result;
}

Result ServiceImpl::UnsubscribeTopic(const nn::sf::InArray<char>& name) NN_NOEXCEPT
{
    NN_NPNS_INVOKE_ON_WORKER_THREAD;

    SafeStringBuffer<TopicNameLength> buffer(name);
    NN_NPNS_TRACE_FUNCTION("(%s)", buffer.GetString());

    Result result = g_Daemon.GetClientThread().UnsubscribeTopic(buffer.GetString());
    return result;
}

Result ServiceImpl::QueryIsTopicExist(nn::sf::Out<bool> bResult, const nn::sf::InArray<char>& name) NN_NOEXCEPT
{
    NN_NPNS_INVOKE_ON_WORKER_THREAD;

    SafeStringBuffer<TopicNameLength> buffer(name);
    NN_NPNS_TRACE_FUNCTION("(%s)", buffer.GetString());

    bool bResultValue;
    Result result = g_Daemon.GetClientThread().QueryIsTopicExist(&bResultValue, buffer.GetString());
    bResult.Set(bResultValue);

    return result;
}

Result ServiceImpl::CreateToken(sf::Out<NotificationToken> outToken, const account::Uid& uid, nn::Bit64 processId) NN_NOEXCEPT
{
    NN_NPNS_INVOKE_ON_WORKER_THREAD;

    NN_NPNS_TRACE_FUNCTION("(uid: %016llx)", uid._data);

    // applicationId 取得
    nn::ApplicationId applicationId;
    Result result = GetApplicationId(&applicationId, processId);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    return CreateTokenWithApplicationId(outToken, uid, applicationId);

}

Result ServiceImpl::CreateTokenWithApplicationId(sf::Out<NotificationToken> outToken, const account::Uid& uid, ApplicationId applicationId) NN_NOEXCEPT
{
    NN_NPNS_INVOKE_ON_WORKER_THREAD;

    NN_NPNS_TRACE_FUNCTION("(uid: %016llx, appId: %016llx)", uid._data, applicationId.value);

    if (applicationId == ApplicationIdForBaaS)
    {
        return ResultInvalidArgument();
    }

    Result result;
    NotificationToken& token = *outToken;
    std::memset(&token, 0, sizeof(NotificationToken));
    result = g_Daemon.GetController().CreateNotificationToken(&token, ReceiverId::Create(uid), applicationId);
    return result;
}

nn::Result ServiceImpl::DestroyToken(const account::Uid& uid, nn::Bit64 processId) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    // applicationId 取得
    nn::ApplicationId applicationId;
    Result result = GetApplicationId(&applicationId, processId);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    return DestroyTokenWithApplicationId(uid, applicationId);
}

Result ServiceImpl::DestroyTokenWithApplicationId(const account::Uid& uid, ApplicationId applicationId) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    Result result;
    result = g_Daemon.GetController().DestroyNotificationToken(ReceiverId::Create(uid), applicationId);
    return result;

}

Result ServiceImpl::QueryIsTokenValid(sf::Out<bool> bResult, const NotificationToken& /*token*/) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    return ResultNotImplemented();
}

nn::Result ServiceImpl::UploadTokenToBaaS(const nn::account::Uid & uid) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    Result result;
    account::AsyncContext async;
    {
        NotificationToken token;
        result = g_Daemon.GetController().CreateNotificationToken(&token, ReceiverId::Create(uid), ApplicationIdForBaaS);
        if (result.IsFailure())
        {
            NN_NPNS_WARN("CreateNotificationToken failed.(0x%08x)\n", result.GetInnerValueForDebug());
            return result;
        }

        result = account::PutBaasChannelResourceAsync(&async, uid, token);
        if (result.IsFailure())
        {
            NN_NPNS_WARN("account::PutBaasChannelResourceAsync failed.(0x%08x)\n", result.GetInnerValueForDebug());
            return result;
        }
    }

    NN_UTIL_SCOPE_EXIT
    {
        bool done;
        NN_ABORT_UNLESS_RESULT_SUCCESS(async.HasDone(&done));
        if (!done)
        {
            async.Cancel();
            do
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                NN_ABORT_UNLESS_RESULT_SUCCESS(async.HasDone(&done));
            } while (!done);
        }
    };

    os::SystemEvent event;
    result = async.GetSystemEvent(&event);
    if (result.IsFailure())
    {
        NN_NPNS_WARN("account::AsyncContext::GetSystemEvent failed.(0x%08x)\n", result.GetInnerValueForDebug());
        async.Cancel();
        return result;
    }

    if (!event.TimedWait(nn::TimeSpan::FromSeconds(20)))
    {
        NN_NPNS_WARN("account::PutBaasChannelResourceAsync timed out\n");
        return ResultConnectionTimeOut();
    }

    return async.GetResult();
}

nn::Result ServiceImpl::DestroyTokenForBaaS(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    Result result;
    result = g_Daemon.GetController().DestroyNotificationToken(ReceiverId::Create(uid), ApplicationIdForBaaS);
    return result;
}

Result ServiceImpl::Suspend() NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    m_SuspendClient.Suspend();

    return ResultSuccess();
}

Result ServiceImpl::Resume() NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    m_SuspendClient.Resume();

    return ResultSuccess();
}

nn::Result ServiceImpl::GetState(nn::sf::Out<std::int32_t> outState) NN_NOEXCEPT
{
//    NN_NPNS_TRACE_FUNCTION("");

    outState.Set(static_cast<int32_t>(g_Daemon.GetStateMachineThread().GetCurrentState()));

    return ResultSuccess();
}

nn::Result nn::npns::ServiceImpl::GetStatistics(const nn::sf::OutBuffer & outStatistics) NN_NOEXCEPT
{
    Statistics& statistics = g_Daemon.GetStatistics();

    size_t sizeToCopy = std::min<size_t>(outStatistics.GetSize(), sizeof(statistics));
    std::memcpy(outStatistics.GetPointerUnsafe(), &statistics, sizeToCopy);

    return ResultSuccess();
}

nn::Result nn::npns::ServiceImpl::GetPlayReportRequestEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    outHandle.Set(sf::NativeHandle(g_Daemon.GetController().GetPlayReportRequestEventReadableHandle(), false));
    return ResultSuccess();
}

Result ServiceImpl::CreateJid() NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");
    SuspendManager::ScopedSuspender scoped(m_SuspendClient);

    Result result;
    result = g_Daemon.GetController().RequestJid();

    return result;
}

Result ServiceImpl::DestroyJid() NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");
    SuspendManager::ScopedSuspender scoped(m_SuspendClient);

    Result result;
    result = g_Daemon.GetController().DestroyJid();

    return result;
}

Result ServiceImpl::AttachJid(const nn::sf::InArray<char>& jid, const nn::sf::InArray<char>& password) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");
    SuspendManager::ScopedSuspender scoped(m_SuspendClient);

    Credential credential ={ {0}, {0} };
    std::memcpy(credential.username, jid.GetData(),      std::min<size_t>(jid.GetLength(),      UsernameLength - 1));
    std::memcpy(credential.password, password.GetData(), std::min<size_t>(password.GetLength(), PasswordLength - 1));

    Result result;
    result = g_Daemon.GetController().SetJid(credential);

    return result;
}

Result ServiceImpl::DetachJid(const nn::sf::OutArray<char>& jid, const nn::sf::OutArray<char>& password) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    Result result;
    SuspendManager::ScopedSuspender scoped(m_SuspendClient);
    Controller& controller = g_Daemon.GetController();

    Credential credential;
    result = controller.GetJid(&credential);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    const size_t sizeJid      = std::strlen(credential.username);
    const size_t sizePassword = std::strlen(credential.password);

    if (jid.GetLength() < sizeJid || password.GetLength() < sizePassword)
    {
        return ResultInsufficientBuffer();
    }

    std::memcpy(jid.GetData(),      credential.username, sizeJid);
    std::memcpy(password.GetData(), credential.password, sizePassword);

    result = g_Daemon.GetController().ClearJid();

    return result;
}

Result ServiceImpl::GetJid(const nn::sf::OutArray<char>& jid) NN_NOEXCEPT
{
    NN_NPNS_TRACE_FUNCTION("");

    Result result;
    SuspendManager::ScopedSuspender scoped(m_SuspendClient);
    Controller& controller = g_Daemon.GetController();

    Credential credential;
    result = controller.GetJid(&credential);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    const size_t size = std::strlen(credential.username);
    if (jid.GetLength() < size)
    {
        return ResultInsufficientBuffer();
    }

    std::memcpy(jid.GetData(), credential.username, size);

    return ResultSuccess();
}

bool ServiceImpl::IsWorkerContext()
{
    return g_Daemon.GetIpcServerManager().IsWorkerThreadContext();
}

nn::Result ServiceImpl::RequestChangeStateForceTimed(int32_t targetState, nn::TimeSpan timeout)
{
    NN_RESULT_THROW_UNLESS(IsTestModeEnabled(), ResultNotImplemented());

    os::SystemEvent interrupt(nn::os::EventClearMode_AutoClear, true);
    return g_Daemon.GetStateMachineThread().RequestChangeStateForceTimed(
        static_cast<nn::npns::State>(targetState), ResultCanceledByHardwareEvents(), timeout, &interrupt);
}

nn::Result ServiceImpl::RequestChangeStateForceAsync(int32_t targetState)
{
    NN_RESULT_THROW_UNLESS(IsTestModeEnabled(), ResultNotImplemented());

    return g_Daemon.GetStateMachineThread().RequestChangeStateForceAsync(
        static_cast<nn::npns::State>(targetState), ResultCanceledByHardwareEvents());
}

}}
