﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <mutex>
#include <cstdlib>
#include <algorithm>

#include "npns_Xmpp.h"
#include "npns_Router.h"
#include "npns_Common.h"
#include "npns_Config.h"

namespace nn{ namespace npns{

const size_t Router::Entry::ListSize;

Router::Router()
    : m_queueNotificationDefault(nullptr), m_mutex(true)
{
}

void Router::RegisterRecipient(Entry& entry)
{
    std::lock_guard<nn::os::Mutex> locker(m_mutex);

    m_listEntry.push_back(entry);
}

void Router::UnregisterRecipient(Entry& entry)
{
    std::lock_guard<nn::os::Mutex> locker(m_mutex);

    Entry* pEntry = &entry;
    auto it = std::find_if(
        m_listEntry.begin(),
        m_listEntry.end(),
        [=](Entry& e) { return &e == pEntry; });
    m_listEntry.erase(it);
}

Result Router::AddRceiveTarget(Entry& entry, ApplicationId toId)
{
    std::lock_guard<nn::os::Mutex> locker(m_mutex);

    Result result = entry.Add(toId);
    if (result.IsFailure())
    {
        return result;
    }

    if (toId.value == 0)
    {
        m_queueNotificationDefault.MoveToAll(entry.m_queue);
    }
    else
    {
        m_queueNotificationDefault.MoveToPartial(toId, entry.m_queue);
    }

    return ResultSuccess();
}

void Router::DeliverNotification(const NotificationData& nd)
{
    std::lock_guard<nn::os::Mutex> locker(m_mutex);

    ApplicationId toId = nd.GetApplicationId();
    // Note: DeliverNotificationToId() は配送先を見つかったかどうかを返す
    if (DeliverNotificationToId(toId, nd)
        || DeliverNotificationToId(ApplicationId::GetInvalidId(), nd))
    {
        return;
    }

    m_queueNotificationDefault.Send(nd);
}

bool Router::DeliverNotificationToId(ApplicationId toId, const NotificationData & nd)
{
    for (auto& entry : m_listEntry)
    {
        if (entry.IsTarget(toId))
        {
            NN_NPNS_INFO("Found listener for 0x%016llx.\n", toId.value);
            Result result = entry.m_queue.Send(nd);
            if (result.IsFailure())
            {
                NN_NPNS_WARN("Notification deliverly is failed.(0x%08x)\n", result.GetInnerValueForDebug());
            }
            return true;
        }
    }
    NN_NPNS_INFO("No listener found for 0x%016llx. Deliver to pending queue.\n", toId.value);
    return false;
}


}}
