﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/bgsu/bgsu_HeapTemplate.h>

#include "npns_HipcServer.h"

#include "npns_Instance.h"
#include "npns_Config.h"
#include "npns_Common.h"

//-----------------------------------------------------------------------------
// メモリ関連の初期化とオーバーライド
struct DefaultHeapTag;
#if NN_NPNS_ENABLE_STANDARD_ALLOCATOR
nn::bgsu::StaticHeapTemplate<DefaultHeapTag, (576) * 1024, nn::mem::StandardAllocator> s_DefaultHeap("default");
#else
nn::bgsu::StaticHeapTemplate<DefaultHeapTag, (576) * 1024, nn::bgsu::ExpHeapAllocator> s_DefaultHeap("default");
#endif
struct FsHeapTag;
nn::bgsu::StaticHeapTemplate<FsHeapTag, 4 * 1024, nn::bgsu::ExpHeapAllocator> s_FsHeap("fs");

#ifndef NN_BUILD_CONFIG_OS_WIN
extern "C" void* malloc(size_t size)               { return s_DefaultHeap.Allocate(size); }
extern "C" void* realloc(void* p, size_t size)     { return s_DefaultHeap.Reallocate(p, size); }
extern "C" void  free(void* p)                     { s_DefaultHeap.Free(p); }
extern "C" void* calloc(size_t n, size_t size)     { return s_DefaultHeap.AllocateZeroInitializedArray(n, size); }
extern "C" void* aligned_alloc(size_t align, size_t size)
{
    return s_DefaultHeap.AllocateAligned(size, align);
}
extern "C" size_t malloc_usable_size(void* p)
{
    NN_UNUSED(p);
    NN_ABORT("malloc_usable_size is not implemented");
}
#endif

void* operator new(std::size_t size)                                      { return malloc(size); }
void* operator new(std::size_t size, const std::nothrow_t&) NN_NOEXCEPT   { return malloc(size); }
void  operator delete(void* ptr) NN_NOEXCEPT                              { free(ptr); }

void* operator new[](std::size_t size)                                    { return malloc(size); }
void* operator new[](std::size_t size, const std::nothrow_t&) NN_NOEXCEPT { return malloc(size); }
void  operator delete[](void* ptr) NN_NOEXCEPT                            { free(ptr); }

//-----------------------------------------------------------------------------
// nn::lm::Initialize の宣言
//

namespace nn { namespace lm {

void Initialize() NN_NOEXCEPT;

}} // namespace nn::lm

//-----------------------------------------------------------------------------
// npns のメインループ
//

namespace nn{ namespace npns{

void MainLoop()
{
#if 0
    NN_SDK_LOG   ("This is SDK_LOG.\n");
    NN_NPNS_LOG  ("This is LOG.\n");
    NN_NPNS_TRACE("This is TRACE.\n");
    NN_NPNS_INFO ("This is INFO.\n");
    NN_NPNS_WARN ("This is WARN.\n");
    NN_NPNS_ERROR("This is ERROR.\n");
    NN_NPNS_FATAL("This is FATAL.\n");
    NN_NPNS_TRACE("This is Very long TRACE!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"
        "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"
        "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!.\n");
#endif
    g_Daemon.Initialize();
#if defined(NN_SDK_BUILD_DEBUG) && NN_NPNS_TEST_ALLOCATION_FAILURE == 1
    s_DefaultHeap.EnableRandomFailure();
#endif
    g_Daemon.GetIpcServerManager().LoopAutoWithDefer();

    g_Daemon.Finalize();
}

bool IsLogManagerEnabled()
{
    bool isEnabled;

    if (nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, 1, "npns", "logmanager_redirection") == 1)
    {
        return isEnabled;
    }
    else
    {
        return false;
    }
}

}}

//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0 );
#ifdef NN_SDK_BUILD_DEBUG
    s_DefaultHeap.EnablePeriodicStatusDump(nn::TimeSpan::FromSeconds(1));
#endif
}

#ifdef NN_BUILD_CONFIG_OS_HORIZON
extern "C" void nndiagStartup()
{
#ifdef NN_NPNS_ENABLE_LOGMANAGER
    // NN_NPNS_ENABLE_LOGMANAGER が定義されていたら静的に分岐
#   if NN_NPNS_ENABLE_LOGMANAGER == 1
    nn::lm::Initialize();
#   endif
#else
    // NN_NPNS_ENABLE_LOGMANAGER が定義されていなかったら動的に分岐
    if (nn::npns::IsLogManagerEnabled())
    {
        nn::lm::Initialize();
    }
#endif
}
#endif

//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_NPNS_THREAD_NAME(Main));
#ifndef NN_SDK_BUILD_RELEASE
    if (nn::os::GetThreadPriority(nn::os::GetCurrentThread()) != NN_NPNS_THREAD_PRIORITY(Main))
    {
        NN_NPNS_WARN("Priority of the Main thread is mismatch between actual and definition.\n");
    }
#endif
    nn::fs::SetAllocator(s_FsHeap.Allocate, s_FsHeap.FreeWithSize);
    nn::npns::MainLoop();
}

//-----------------------------------------------------------------------------

