﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_FormatString.h>
#include <nn/http.h>
#include <nn/http/http_JsonResponse.h>
#include "npns_Common.h"
#include "npns_JidConsumer.h"

namespace nn {
namespace npns {

JidConsumer::JidConsumer(http::ConnectionBroker& cb, DeviceTokenHolder& dth)
    : ResourceConsumerBase(cb, dth)
{
}

JidConsumer::~JidConsumer()
{
}

Result JidConsumer::Create(Credential * pOut)
{
    http::Request req(m_ConnectionBroker);

    req.SetUrlPointer("https://consumer." NN_NPNS_SERVER_ENVIRONMENT ".npns.srv.nintendo.net/api/v1/jids");
    req.SetMethodType(http::MethodType_Post);
    SetupRequestHeaders(&req);

    int64_t resultCode;
    const nn::http::JsonResponse::ParseRule parseRules[] = {
        { "result_code", nullptr, nn::http::JsonResponse::ValueType_Int64Force, &resultCode,    sizeof(resultCode) },
        { "jid",         nullptr, nn::http::JsonResponse::ValueType_String,     pOut->username, UsernameLength },
        { "password",    nullptr, nn::http::JsonResponse::ValueType_String,     pOut->password, PasswordLength },
        {}
    };

    Result result = SendRequest(&req, parseRules, resultCode);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    NN_NPNS_INFO("result_code = %04d\n", resultCode);
    NN_NPNS_INFO("jid = %s\n", pOut->username);
    NN_NPNS_INFO("password = %s\n", pOut->password);
    return ResultSuccess();
}

Result JidConsumer::Renew(Credential * pOut, const Credential & currentCredential)
{
    http::Request req(m_ConnectionBroker);

    req.SetUrlPointer("https://consumer." NN_NPNS_SERVER_ENVIRONMENT ".npns.srv.nintendo.net/api/v1/jids");
    req.SetMethodType(http::MethodType_Patch);
    req.SetPostDataByFormatString("jid=%s&password=%s", currentCredential.username, currentCredential.password);
    SetupRequestHeaders(&req);

    int64_t resultCode;
    const nn::http::JsonResponse::ParseRule parseRules[] = {
        { "result_code", nullptr, nn::http::JsonResponse::ValueType_Int64Force, &resultCode,    sizeof(resultCode) },
        { "jid",         nullptr, nn::http::JsonResponse::ValueType_String,     pOut->username, UsernameLength },
        { "password",    nullptr, nn::http::JsonResponse::ValueType_String,     pOut->password, PasswordLength },
        {}
    };

    Result result = SendRequest(&req, parseRules, resultCode);

    NN_NPNS_INFO("result_code = %04d\n", resultCode);

    if (resultCode == ResultCode_JidPasswordIsNotNull)
    {
        NN_NPNS_WARN("Current JID is not suspended. We continue to use the current account. (%s)\n", currentCredential.username);
        *pOut = currentCredential;
        return ResultSuccess();
    }
    if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

Result JidConsumer::Remove(const Credential& currentCredential)
{
    http::Request req(m_ConnectionBroker);

    req.SetUrlPointer("https://consumer." NN_NPNS_SERVER_ENVIRONMENT ".npns.srv.nintendo.net/api/v1/jids");
    req.SetMethodType(http::MethodType_Delete);
    req.SetPostDataByFormatString("jid=%s", currentCredential.username);
    SetupRequestHeaders(&req);

    int64_t resultCode;
    const nn::http::JsonResponse::ParseRule parseRules[] = {
        { "result_code", nullptr, nn::http::JsonResponse::ValueType_Int64Force, &resultCode,    sizeof(resultCode) },
        {}
    };

    Result result = SendRequest(&req, parseRules, resultCode);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    NN_NPNS_INFO("result_code = %04d\n", resultCode);

    return ResultSuccess();
}

}
}
