﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/util/util_StringUtil.h>
#include <nn/bgtc.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/settings/system/settings_PushNotification.h>
#include <mutex>
#include <cstdlib>
#include <algorithm>

#include "npns_Common.h"
#include "npns_Controller.h"
#include "npns_ClientThread.h"

namespace nn {
namespace npns {

Controller::Controller(ClientThread & client)
    : m_eventIntrrupt(os::EventClearMode_AutoClear)
    , m_eventPlayReportRequest(os::EventClearMode_ManualClear, true)
    , m_tickLastRequestPlayReport(0)
    , m_mutexJid(true)
    , m_Client(client)
    , m_DeviceTokenConsumer()
    , m_JidConsumer(m_ConnectionBroker, m_DeviceTokenHolder)
    , m_NotificationTokenConsumer(m_ConnectionBroker, m_DeviceTokenHolder)
    , m_ArrivalConsumer(m_ConnectionBroker, m_DeviceTokenHolder)
    , m_LastBrokerCheckingIntervalFromServer(0)
    , m_LastKeepSessionTryCountFromServer(0)
{
}

Controller::~Controller()
{
}

Result Controller::Initialize()
{
    Result result;

    nifm::NetworkConnection* pContinuousNetworkConnection = new(&m_pContinuousNetworkConnection) nifm::NetworkConnection();
    nifm::SetRequestRequirementPreset(pContinuousNetworkConnection->GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessContinuous);
#if NN_NPNS_ENABLE_WOWLAN
    nifm::SetRequestKeptInSleep(pContinuousNetworkConnection->GetRequestHandle(), true);
#endif

    result = m_ConnectionBroker.Initialize();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    m_ConnectionBroker.SetEnableClientCert(true);
    m_ConnectionBroker.SetSocketBufferSize(8 * 1024, 8 * 1024);

#if NN_NPNS_DEBUG_HTTP
    m_ConnectionBroker.SetSkipSslVerificationForDebug(true);
    m_ConnectionBroker.SetVerbose(true);
#endif

#if NN_NPNS_TEST_USE_PROXY
    m_ConnectionBroker.SetProxy(NN_NPNS_TEST_PROXY_HOST, NN_NPNS_TEST_PROXY_PORT);
#if defined(NN_NPNS_TEST_PROXY_USER)
    m_ConnectionBroker.SetProxyAuthentication(NN_NPNS_TEST_PROXY_USER, NN_NPNS_TEST_PROXY_PASS);
#endif
#endif

    result = m_SaveDataManager.Initialize();
    if (result.IsFailure())
    {
        NN_NPNS_ERROR("Failed to initialize SaveDataManager.\n");
        m_ConnectionBroker.Finalize();
        return result;
    }

    result = m_SaveDataManager.Read(&m_PersistentData);
    if (result.IsSuccess() && std::strncmp(m_PersistentData.environment, GetCurrentEnvironment(), 4) != 0)
    {
        NN_NPNS_INFO("Environment will change '%s' to '%s'.\n",
            m_PersistentData.environment, GetCurrentEnvironment());
        m_PersistentData.state = PersistentData::CredentialState_Initial;
    }
    else
    {
        NN_NPNS_INFO("Environment is '%s'.\n", GetCurrentEnvironment());
    }

    // (SIGLO-80171)
    // JID のワールド名が空になるバグの復旧をここで行う.
    // ワールド名が空になると"-."という文字列がJIDに含まれる.
    // CredentialState_Invalid にすることで、 JID Renew が行われ復旧される.
    if (m_PersistentData.state == PersistentData::CredentialState_Valid
        && std::strstr(m_PersistentData.credential.username, "-.") != nullptr)
    {
        NN_NPNS_INFO("Invalid credential username. (%s)\n", m_PersistentData.credential.username);
        m_PersistentData.state = PersistentData::CredentialState_Invalid;
    }

    m_tickLastRequestPlayReport = os::GetSystemTick();

    return ResultSuccess();
}

void Controller::Finalize()
{
    m_SaveDataManager.Finalize();

    m_ConnectionBroker.Finalize();

    GetNetworkConnection().~NetworkConnection();
}

Result Controller::XmppConnect()
{
    Result result;

    NN_NPNS_INFO("Connecting to XMPP server... (JID=%s)\n", m_PersistentData.credential.username);

    result = m_Client.Connect(m_PersistentData.credential.username, m_PersistentData.credential.password);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    return ResultSuccess();
}

void Controller::XmppEnableWhiteSpacePing(os::Event* pEventInterrupt, bool isPowerModeNormal)
{
    m_Client.EnableWhiteSpacePing(pEventInterrupt, isPowerModeNormal);
}

void Controller::XmppSetupConnectionForSleep()
{
#if NN_NPNS_ENABLE_WOWLAN
    Result result = nifm::RegisterRequestSocketDescriptor(GetNetworkConnection().GetRequestHandle(), m_Client.GetSocket());
    if (result.IsFailure())
    {
        NN_NPNS_WARN("nifm::RegisterRequestSocketDescriptor failed.(0x%08x)\n", result.GetInnerValueForDebug());
    }
#endif
}

void Controller::XmppDisconnect()
{
#if NN_NPNS_ENABLE_WOWLAN
    if (IsOnline())
    {
        Result result = nifm::UnregisterRequestSocketDescriptor(GetNetworkConnection().GetRequestHandle());
        if (result.IsFailure() && m_Client.GetSocket() >= 0)
        {
            NN_NPNS_WARN("nifm::UnregisterRequestSocketDescriptor failed.(0x%08x)\n", result.GetInnerValueForDebug());
            NN_UNUSED(result);
        }
    }
#endif
    (void)m_Client.Disconnect();
}

bool Controller::XmppIsConnected()
{
    return m_Client.IsConnected();
}

bool Controller::XmppIsConnectionInProgress()
{
    return m_Client.IsConnectionInProgress();
}

bool Controller::XmppIsUsingProxy()
{
    return m_Client.IsUsingProxy();
}

Result Controller::XmppGetLastError() const
{
    return m_Client.GetLastError();
}

void Controller::XmppSetStatus(Client::Status status)
{
    m_Client.SetStatus(status);
}

Result Controller::RequestJid()
{
    std::lock_guard<os::Mutex> lock(m_mutexJid);
    Result result;

    NN_NPNS_INFO("Requesting an account of XMPP server...\n");

    result = EnsureDeviceToken();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    if (m_PersistentData.state == PersistentData::CredentialState_Initial)
    {
        NN_NPNS_INFO("Creating an new account...\n");
        result = m_JidConsumer.Create(&m_PersistentData.credential);
        if (result.IsFailure())
        {
            NN_NPNS_WARN("Failed to create an account.\n");
            return result;
        }
    }
    else
    {
        Credential outCredential;
        NN_NPNS_INFO("Renewing the current account(%s)...\n", m_PersistentData.credential.username);
        result = m_JidConsumer.Renew(&outCredential, m_PersistentData.credential);
        if (result.IsFailure())
        {
            NN_NPNS_WARN("Failed to renew the account.\n");
            return result;
        }
        m_PersistentData.credential = outCredential;
    }

    m_PersistentData.state = PersistentData::CredentialState_Valid;
    std::strncpy(m_PersistentData.environment, GetCurrentEnvironment(), 4);

    FlushSaveData();

    return ResultSuccess();
}

Result Controller::DestroyJid()
{
    std::lock_guard<os::Mutex> lock(m_mutexJid);
    Result result;

    NN_NPNS_INFO("Deleting the account of XMPP server...\n");

    if (m_PersistentData.state == PersistentData::CredentialState_Initial)
    {
        return ResultAccountNotAvailable();
    }

    result = EnsureDeviceToken();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    result = m_JidConsumer.Remove(m_PersistentData.credential);
    if (result.IsFailure())
    {
        NN_NPNS_WARN("Failed to delete the account.\n");
        return result;
    }

    m_PersistentData.state = PersistentData::CredentialState_Initial;
    std::memset(&m_PersistentData.credential, 0, sizeof(m_PersistentData.credential));

    FlushSaveData();

    return ResultSuccess();
}

Result Controller::SetJid(const Credential & credential)
{
    std::lock_guard<os::Mutex> lock(m_mutexJid);

    if (m_PersistentData.state != PersistentData::CredentialState_Initial)
    {
        return ResultAccountAvailable();
    }

    m_PersistentData.credential = credential;
    m_PersistentData.state = PersistentData::CredentialState_Valid;

    return ResultSuccess();
}

Result Controller::ClearJid()
{
    std::lock_guard<os::Mutex> lock(m_mutexJid);
    Result result;

    m_PersistentData.state = PersistentData::CredentialState_Initial;
    std::memset(&m_PersistentData.credential, 0, sizeof(m_PersistentData.credential));

    result = FlushSaveData();

    return result;
}

void Controller::InvalidateJid()
{
    std::lock_guard<os::Mutex> lock(m_mutexJid);

    m_PersistentData.state = PersistentData::CredentialState_Invalid;
}

Result Controller::CreateNotificationToken(NotificationToken* pOutToken, const ReceiverId& uid, ApplicationId applicationId)
{
    Result result;

    NN_NPNS_INFO("Creating a notification token...\n");

    if (!HasValidJid())
    {
        NN_NPNS_WARN("JID required.\n");
        return ResultAccountNotAvailable();
    }

    result = EnsureDeviceToken();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    result = m_NotificationTokenConsumer.Create(pOutToken, uid, m_PersistentData.credential.username, applicationId);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    NN_NPNS_INFO("Created notification token: %s\n", pOutToken->data);

    return ResultSuccess();
}

Result Controller::DestroyNotificationToken(const ReceiverId& uid, ApplicationId applicationId)
{
    Result result;

    NN_NPNS_INFO("Destroying the notification token...\n");

    result = EnsureDeviceToken();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    result = m_NotificationTokenConsumer.Destroy(uid, m_PersistentData.credential.username, applicationId);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    return ResultSuccess();
}

Result Controller::CheckMessageArrived(bool* pIsArrived, os::Event* pEventInterrupt)
{
    int32_t wait = 0, tryCount = 0;
    Result result = m_ArrivalConsumer.Check(pIsArrived, &wait, &tryCount, m_PersistentData.credential.username, pEventInterrupt);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    m_LastBrokerCheckingIntervalFromServer = wait;
    m_LastKeepSessionTryCountFromServer    = tryCount;

    return ResultSuccess();
}

void Controller::RequestPlayReportSubmissionIfRequired()
{
#if NN_NPNS_PLAYREPORT_INTERVAL_HOUR > 0
    const TimeSpan interval = TimeSpan::FromHours(NN_NPNS_PLAYREPORT_INTERVAL_HOUR);
    const os::Tick diff = os::GetSystemTick() - m_tickLastRequestPlayReport;
    if (diff.GetInt64Value() > 0 && diff.ToTimeSpan() < interval)
    {
        return;
    }

    m_eventPlayReportRequest.Signal();

    m_tickLastRequestPlayReport = os::GetSystemTick();
#endif
}

os::NativeHandle Controller::GetPlayReportRequestEventReadableHandle()
{
    return m_eventPlayReportRequest.GetReadableHandle();
}

bool Controller::HasValidJid()
{
    return m_PersistentData.state == PersistentData::CredentialState_Valid;
}

Result Controller::WaitForOnline(os::Event* pEventInterrupt, bool bEnableTimeout)
{
    os::SystemEvent& eventNetwork = GetNetworkConnection().GetSystemEvent();
    while (!IsOnline())
    {
        int ret;
        if (bEnableTimeout)
        {
            ret = os::TimedWaitAny(TimeSpan::FromSeconds(NN_NPNS_TIMEOUT_ONLINE),
                pEventInterrupt->GetBase(),
                eventNetwork.GetBase()
            );
        }
        else
        {
            ret = os::WaitAny(
                pEventInterrupt->GetBase(),
                eventNetwork.GetBase()
            );
        }
        switch (ret)
        {
        case -1:
            NN_NPNS_TRACE("WaitForOnline: timeout\n");
            return ResultNetworkIsNotAvailable();

        case 0:
            NN_NPNS_TRACE("WaitForOnline: interrupted\n");
            pEventInterrupt->Clear();
            return ResultInterruptByRequest();

        case 1:
            NN_NPNS_TRACE("WaitForOnline: changed network state: ->%s\n",
                IsOnline() ? "online" : "offline");
            eventNetwork.Clear();
            if (!IsOnline() && bgtc::IsInHalfAwake())
            {
                NN_NPNS_TRACE("WaitForOnline: oneshot failure\n");
                return ResultNetworkIsNotAvailable();
            }
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    return ResultSuccess();
}

Result Controller::WaitAnyEvents(os::Event* pEventInterrupt, const TimeSpan & time)
{
    int ret = os::TimedWaitAny(time,
        pEventInterrupt->GetBase(),                             // 0
        m_Client.GetConnectionStateChangedEvent().GetBase(),    // 1
        GetNetworkConnection().GetSystemEvent().GetBase()       // 2
    );
    switch (ret)
    {
    case -1:
        NN_NPNS_TRACE("WaitAnyEvents: timeout\n");
        return ResultCommandTimeOut();

    case 0:
        NN_NPNS_TRACE("WaitAnyEvents: interrupted\n");
        pEventInterrupt->Clear();
        return ResultInterruptByRequest();

    case 1:
        NN_NPNS_TRACE("WaitAnyEvents: changed client state: connected = %s\n",
            m_Client.IsConnected() ? "yes" : "no");
        m_Client.GetConnectionStateChangedEvent().Clear();
        break;

    case 2:
        NN_NPNS_TRACE("WaitAnyEvents: changed network state: ->%s\n",
            IsOnline() ? "online" : "offline");
        GetNetworkConnection().GetSystemEvent().Clear();
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return ResultSuccess();
}

Result Controller::WaitAnyEvents(os::Event* pEventInterrupt)
{
    int index = os::WaitAny(
        pEventInterrupt->GetBase(),                             // 0
        m_Client.GetConnectionStateChangedEvent().GetBase(),    // 1
        GetNetworkConnection().GetSystemEvent().GetBase()       // 2
    );

    switch (index)
    {
    case 0:
        NN_NPNS_TRACE("WaitAnyEvents: interrupted\n");
        pEventInterrupt->Clear();
        return ResultInterruptByRequest();

    case 1:
        NN_NPNS_TRACE("WaitAnyEvents: changed client state: connected = %s\n",
            m_Client.IsConnected() ? "yes" : "no");
        m_Client.GetConnectionStateChangedEvent().Clear();
        break;

    case 2:
        NN_NPNS_TRACE("WaitAnyEvents: changed network state: ->%s\n",
            IsOnline() ? "online" : "offline");
        GetNetworkConnection().GetSystemEvent().Clear();
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return ResultSuccess();
}

void Controller::SubmitOnlineRequest()
{
    NN_NPNS_TRACE("NetworkConnection::SubmitRequest\n");
    GetNetworkConnection().GetSystemEvent().Clear();
    GetNetworkConnection().SubmitRequest();
}

void Controller::CancelOnlineRequest()
{
    NN_NPNS_TRACE("NetworkConnection::CancelRequest\n");
    GetNetworkConnection().CancelRequest();
}

bool Controller::IsOnline()
{
    return GetNetworkConnection().IsAvailable();
}

Result Controller::FlushSaveData()
{
    return m_SaveDataManager.Write(m_PersistentData);
}

const char * Controller::GetCurrentEnvironment()
{
    static nsd::EnvironmentIdentifier envName = {};

    if (envName.value[0] == '\0')
    {
        if(util::Strncmp(NN_NPNS_SERVER_ENVIRONMENT, "%", 1) == 0)
        {
            nn::nsd::GetEnvironmentIdentifier(&envName);
        }
        else
        {
            util::Strlcpy<char>(envName.value, NN_NPNS_SERVER_ENVIRONMENT, sizeof(envName.value));
        }
    }
    return envName.value;
}

Result Controller::GetJid(Credential* pCredential)
{
    if (!HasValidJid())
    {
        return ResultAccountNotAvailable();
    }

    *pCredential = m_PersistentData.credential;
    return ResultSuccess();
}

Result Controller::EnsureDeviceToken()
{
    Result result;

    // ignoreCache == false かつ nn::dauth 側で有効なキャッシュがあれば、すぐに返る
    bool ignoreCache = m_DeviceTokenHolder.IsRefreshRequired();
    result = m_DeviceTokenConsumer.GetToken(&m_DeviceTokenHolder.GetStorage(), NdasClientId, ignoreCache);
    if (result.IsFailure())
    {
        NN_NPNS_WARN("Failed to obtain a device token. (0x%08x)\n", result.GetInnerValueForDebug());
        return result;
    }
    m_DeviceTokenHolder.NotifySet();

    return ResultSuccess();
}

TimeSpan Controller::GetBackoffWaitFromServer() const
{
    return m_Client.GetBackoffWaitFromServer();
}

bool Controller::IsEnabledRealTimePushDuringSleep()
{
#if !NN_NPNS_ENABLE_WOWLAN
    return false;
#else
    int32_t mode = settings::system::GetPushNotificationActivityModeOnSleep();
    switch (mode)
    {
    default:
    case ActivityMode_SystemDefault:
    case ActivityMode_RealTime:
        break;

    case ActivityMode_Periodical:
        return false;
    }

    // スリープ時に接続を維持できる条件が揃っているか？
    if (!bgtc::WillDisconnectNetworkWhenEnteringSleep())
    {
        // リアルタイムPush通信を有効にするかは電源状態に応じて NPNS が決める
        bgtc::OperationMode operationMode = bgtc::GetOperationMode();
        return operationMode == bgtc::OperationMode_Active
            || operationMode == bgtc::OperationMode_Balance;
    }
    else
    {
        // 接続が維持できないなら半起床を使った疑似スリープができるか
        // bgtc のポリシー判断に従う
        return bgtc::WillStayHalfAwakeInsteadSleep();
    }
#endif
}

}
}
