﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/npns/detail/npns_Utility.h>
#include <nn/srepo/srepo_StateNotifier.h>

#include <mutex>

#include "npns_Common.h"
#include "npns_ClientThread.h"

namespace
{
    void NotifyOnlineToSrepo() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::srepo::NotifyNotificationConnectivityChanged(nn::srepo::NotificationConnectivity::Available);
#endif
    }

    void NotifyOfflineToSrepo() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::srepo::NotifyNotificationConnectivityChanged(nn::srepo::NotificationConnectivity::Unavailable);
#endif
    }
}

namespace nn{ namespace npns{

ClientThread::ClientThread()
    : ThreadBase(NN_NPNS_THREAD_CONFIG(Client))
    , m_eventUpdated(nn::os::EventClearMode_ManualClear)
    , m_eventFlush(nn::os::EventClearMode_AutoClear)
    , m_eventConnectionStateChanged(nn::os::EventClearMode_AutoClear)
    , m_mutexConnection(true)
{
}

Result ClientThread::Initialize()
{
    Client::Initialize();
    return ThreadBase::Initialize();
}

void ClientThread::Finalize()
{
    ThreadBase::Finalize();
    Client::Finalize();
}

Result ClientThread::Connect(const char* jid, const char* password)
{
    std::lock_guard<os::Mutex> lock(m_mutexConnection);

    m_eventConnectionStateChanged.Clear();

    Result result = Client::Connect(jid, password);
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    m_eventUpdated.Signal();

    return ResultSuccess();
}

void ClientThread::Disconnect()
{
    {
        std::lock_guard<os::Mutex> lock(m_mutexConnection);

        // 切断
        Client::Disconnect();

        // 通信処理終了時に Loop スレッドを一時停止
        m_eventUpdated.Clear();
    }

    // 切断ハンドラの処理が終わるのを待つ
    if (!WaitDisconnect())
    {
        NN_NPNS_WARN("Disconnection timed out.\n");
    }

    {
        std::lock_guard<os::Mutex> lock(m_mutexConnection);

        // libstrophe 再初期化
        Client::Reinitialize();
    }
}

os::Event & ClientThread::GetConnectionStateChangedEvent()
{
    return m_eventConnectionStateChanged;
}

void ClientThread::OnShutdown()
{
}

bool ClientThread::WaitDisconnect()
{
    const int32_t sleepIntervalMs = 20;
    for (int32_t i = 0; i < DisconnectionTimeOutInMs / sleepIntervalMs; ++i)
    {
        if (!IsRunning())
        {
            return true;
        }
        os::SleepThread(TimeSpan::FromMilliSeconds(sleepIntervalMs));
    }
    return false;
}

bool ClientThread::WaitPendingDataEmpty()
{
    const int32_t interval = 20;
    for (int32_t i = 0; i < (NN_NPNS_NETWORK_POLL_INTERVAL / interval) + 1; ++i)
    {
        if (!HasPendingData())
        {
            return true;
        }
        os::SleepThread(TimeSpan::FromMilliSeconds(interval));
    }
    return false;
}

void ClientThread::OnConnected()
{
    NotifyOnlineToSrepo();

    Client::OnConnected();
}

void ClientThread::OnDisconnected()
{
    NotifyOfflineToSrepo();

    m_eventConnectionStateChanged.Signal();
}

void ClientThread::OnReceiveIdle()
{
    Client::OnReceiveIdle();

    m_eventConnectionStateChanged.Signal();
}

void ClientThread::OnFlush()
{
    m_eventFlush.Signal();
}

void ClientThread::ThreadBody()
{
    /* start the event loop */
    while(!IsExiting())
    {
        m_eventUpdated.Wait();

        do {
            Client::RunOnce(NN_NPNS_NETWORK_POLL_INTERVAL);

            m_eventFlush.TimedWait(TimeSpan::FromMilliSeconds(NN_NPNS_NETWORK_WAIT_INTERVAL));
        } while (Client::IsRunning() && !IsExiting());
    }
}

void ClientThread::RequestExit()
{
    m_eventUpdated.Signal();
    ThreadBase::RequestExit();
}

}}


